<?php
// MÓDULO DE CONFIGURACIÓN - SISCONECTA ERP
// configuracion.php

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Verificar sesión activa
if (!isset($_SESSION['user_id']) || !isset($_SESSION['empresa_id'])) {
    header('Location: login.php');
    exit;
}

// Configuración de base de datos
$host = 'localhost';
$dbname = 'conectae_sisconecta';
$username = 'conectae_sisconectauser';
$password = 'pt125824caraud';

$mensaje = '';
$tipo_mensaje = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $empresa_id = $_SESSION['empresa_id'];
    
    // Verificar estado de empresa
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ? AND estado = 'activo'");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch();
    
    if (!$empresa) {
        session_destroy();
        header('Location: login.php?error=empresa_inactiva');
        exit;
    }
    
    // Verificar vencimiento (excepto AUDITOREX CHILE)
    if (($empresa['codigo_empresa'] ?? '') !== 'AUDITOREX_CHILE') {
        $fecha_actual = new DateTime();
        $fecha_vencimiento = new DateTime($empresa['fecha_vencimiento'] ?? '2024-12-31');
        
        if ($fecha_actual > $fecha_vencimiento) {
            session_destroy();
            header('Location: login.php?error=suscripcion_vencida');
            exit;
        }
    }
    
    $seccion = $_GET['seccion'] ?? 'empresa';
    
    // Actualizar datos de empresa
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['actualizar_empresa'])) {
        $razon_social = trim($_POST['razon_social']);
        $rut = trim($_POST['rut']);
        $direccion = trim($_POST['direccion']);
        $comuna = trim($_POST['comuna']);
        $ciudad = trim($_POST['ciudad']);
        $region = trim($_POST['region']);
        $telefono = trim($_POST['telefono']);
        $email = trim($_POST['email']);
        $giro = trim($_POST['giro']);
        $actividad_economica = trim($_POST['actividad_economica']);
        
        if (!empty($razon_social) && !empty($rut)) {
            try {
                $stmt = $pdo->prepare("
                    UPDATE empresas 
                    SET razon_social = ?, rut = ?, direccion = ?, comuna = ?, ciudad = ?, region = ?,
                        telefono = ?, email = ?, giro = ?, actividad_economica = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $razon_social, $rut, $direccion, $comuna, $ciudad, $region,
                    $telefono, $email, $giro, $actividad_economica, $empresa_id
                ]);
                
                // Actualizar datos en sesión
                $_SESSION['empresa_razon_social'] = $razon_social;
                
                $mensaje = 'Datos de empresa actualizados correctamente';
                $tipo_mensaje = 'success';
                
                // Recargar datos
                $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ?");
                $stmt->execute([$empresa_id]);
                $empresa = $stmt->fetch();
                
            } catch (PDOException $e) {
                $mensaje = 'Error al actualizar empresa: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Razón social y RUT son obligatorios';
            $tipo_mensaje = 'warning';
        }
    }
    
    // Crear nuevo usuario
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['crear_usuario'])) {
        $username = trim($_POST['username']);
        $password = trim($_POST['password']);
        $nombre = trim($_POST['nombre']);
        $email = trim($_POST['email']);
        $rol = $_POST['rol'];
        
        if (!empty($username) && !empty($password) && !empty($nombre) && !empty($email)) {
            try {
                // Verificar que el username no exista
                $stmt = $pdo->prepare("SELECT id FROM usuarios WHERE username = ? AND empresa_id = ?");
                $stmt->execute([$username, $empresa_id]);
                if ($stmt->fetch()) {
                    throw new Exception('El nombre de usuario ya existe');
                }
                
                $password_hash = password_hash($password, PASSWORD_DEFAULT);
                
                $stmt = $pdo->prepare("
                    INSERT INTO usuarios (empresa_id, username, password, nombre, email, rol, estado) 
                    VALUES (?, ?, ?, ?, ?, ?, 'activo')
                ");
                $stmt->execute([$empresa_id, $username, $password_hash, $nombre, $email, $rol]);
                
                $mensaje = 'Usuario creado correctamente';
                $tipo_mensaje = 'success';
                
            } catch (Exception $e) {
                $mensaje = 'Error al crear usuario: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Complete todos los campos obligatorios';
            $tipo_mensaje = 'warning';
        }
    }
    
    // Cambiar estado de usuario
    if (isset($_GET['cambiar_estado_usuario'])) {
        $usuario_id = $_GET['cambiar_estado_usuario'];
        $nuevo_estado = $_GET['estado'];
        
        try {
            $stmt = $pdo->prepare("UPDATE usuarios SET estado = ? WHERE id = ? AND empresa_id = ? AND id != ?");
            $stmt->execute([$nuevo_estado, $usuario_id, $empresa_id, $_SESSION['user_id']]);
            
            $mensaje = 'Estado de usuario actualizado';
            $tipo_mensaje = 'success';
        } catch (PDOException $e) {
            $mensaje = 'Error al cambiar estado: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
    
    // Agregar/Actualizar moneda
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['guardar_moneda'])) {
        $codigo = strtoupper(trim($_POST['codigo']));
        $nombre = trim($_POST['nombre']);
        $simbolo = trim($_POST['simbolo']);
        $tasa_cambio = floatval($_POST['tasa_cambio']);
        
        if (!empty($codigo) && !empty($nombre) && !empty($simbolo) && $tasa_cambio > 0) {
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO monedas (empresa_id, codigo, nombre, simbolo, tasa_cambio) 
                    VALUES (?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE 
                    nombre = VALUES(nombre), simbolo = VALUES(simbolo), tasa_cambio = VALUES(tasa_cambio)
                ");
                $stmt->execute([$empresa_id, $codigo, $nombre, $simbolo, $tasa_cambio]);
                
                $mensaje = 'Moneda guardada correctamente';
                $tipo_mensaje = 'success';
                
            } catch (PDOException $e) {
                $mensaje = 'Error al guardar moneda: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Complete todos los campos de la moneda';
            $tipo_mensaje = 'warning';
        }
    }
    
    // Cambiar moneda principal
    if (isset($_GET['moneda_principal'])) {
        $nueva_moneda = $_GET['moneda_principal'];
        
        try {
            $stmt = $pdo->prepare("UPDATE empresas SET moneda_principal = ? WHERE id = ?");
            $stmt->execute([$nueva_moneda, $empresa_id]);
            
            $mensaje = 'Moneda principal actualizada';
            $tipo_mensaje = 'success';
            
            // Recargar datos
            $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ?");
            $stmt->execute([$empresa_id]);
            $empresa = $stmt->fetch();
            
        } catch (PDOException $e) {
            $mensaje = 'Error al cambiar moneda principal: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
    
    // Obtener usuarios de la empresa
    $stmt = $pdo->prepare("SELECT * FROM usuarios WHERE empresa_id = ? ORDER BY nombre ASC");
    $stmt->execute([$empresa_id]);
    $usuarios = $stmt->fetchAll();
    
    // Obtener monedas
    $stmt = $pdo->prepare("SELECT * FROM monedas WHERE empresa_id = ? OR empresa_id IS NULL ORDER BY codigo ASC");
    $stmt->execute([$empresa_id]);
    $monedas = $stmt->fetchAll();
    
    // Obtener configuración DTE
    $stmt = $pdo->prepare("SELECT * FROM configuracion_dte WHERE empresa_id = ?");
    $stmt->execute([$empresa_id]);
    $config_dte = $stmt->fetch();
    
    // Estadísticas del sistema
    $stmt = $pdo->prepare("
        SELECT 
            (SELECT COUNT(*) FROM usuarios WHERE empresa_id = ? AND estado = 'activo') as usuarios_activos,
            (SELECT COUNT(*) FROM productos WHERE empresa_id = ? AND activo = 1) as productos_activos,
            (SELECT COUNT(*) FROM documentos_tributarios WHERE empresa_id = ?) as total_documentos,
            (SELECT COUNT(*) FROM asientos_contables WHERE empresa_id = ?) as total_asientos
    ");
    $stmt->execute([$empresa_id, $empresa_id, $empresa_id, $empresa_id]);
    $estadisticas = $stmt->fetch();
    
} catch (PDOException $e) {
    die('Error de conexión: ' . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Configuración - SISCONECTA ERP</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .main-container {
            padding: 20px;
            max-width: 1600px;
            margin: 0 auto;
        }
        
        .header-section {
            background: linear-gradient(45deg, #2c3e50, #34495e);
            color: white;
            padding: 25px 30px;
            border-radius: 15px;
            margin-bottom: 25px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }
        
        .header-title {
            font-size: 1.9rem;
            font-weight: 700;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
        }
        
        .btn-header {
            background: rgba(255,255,255,0.15);
            border: 2px solid rgba(255,255,255,0.3);
            color: white;
            border-radius: 10px;
            padding: 12px 20px;
            text-decoration: none;
            transition: all 0.3s ease;
            font-weight: 600;
        }
        
        .btn-header:hover {
            background: rgba(255,255,255,0.25);
            color: white;
            transform: translateY(-2px);
        }
        
        .content-card {
            background: white;
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            margin-bottom: 25px;
        }
        
        .stats-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border-left: 5px solid;
            transition: all 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.15);
        }
        
        .stat-card.primary { border-left-color: #3498db; }
        .stat-card.success { border-left-color: #27ae60; }
        .stat-card.warning { border-left-color: #f39c12; }
        .stat-card.info { border-left-color: #9b59b6; }
        
        .stat-value {
            font-size: 2.2rem;
            font-weight: 800;
            color: #2c3e50;
            margin-bottom: 8px;
        }
        
        .stat-label {
            color: #666;
            font-weight: 600;
            font-size: 0.95rem;
        }
        
        .stat-icon {
            float: right;
            font-size: 2.5rem;
            opacity: 0.2;
        }
        
        .nav-pills .nav-link {
            border-radius: 10px;
            font-weight: 600;
            margin-right: 10px;
            padding: 12px 20px;
            transition: all 0.3s ease;
        }
        
        .nav-pills .nav-link.active {
            background: linear-gradient(135deg, #3498db, #2980b9);
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(52,152,219,0.3);
        }
        
        .nav-pills .nav-link:hover:not(.active) {
            background: #f8f9fa;
            transform: translateY(-1px);
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 12px 15px;
            transition: all 0.3s ease;
            font-weight: 500;
        }
        
        .form-control:focus {
            border-color: #3498db;
            box-shadow: 0 0 0 0.25rem rgba(52,152,219,0.25);
            transform: translateY(-1px);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #3498db, #2980b9);
            border: none;
            border-radius: 10px;
            padding: 12px 25px;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(52,152,219,0.3);
        }
        
        .btn-success {
            background: linear-gradient(135deg, #27ae60, #229954);
            border: none;
            border-radius: 10px;
            padding: 12px 25px;
            font-weight: 600;
        }
        
        .btn-warning {
            background: linear-gradient(135deg, #f39c12, #e67e22);
            border: none;
            border-radius: 10px;
            padding: 8px 16px;
            font-weight: 600;
        }
        
        .btn-danger {
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            border: none;
            border-radius: 10px;
            padding: 8px 16px;
            font-weight: 600;
        }
        
        .table {
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .table th {
            background: linear-gradient(135deg, #34495e, #2c3e50);
            color: white;
            border: none;
            font-weight: 700;
            padding: 15px 12px;
        }
        
        .table td {
            padding: 15px 12px;
            vertical-align: middle;
            border-top: 1px solid #f1f3f4;
        }
        
        .table tbody tr:hover {
            background: #f8f9fa;
            transform: scale(1.01);
            transition: all 0.2s ease;
        }
        
        .badge {
            padding: 8px 12px;
            border-radius: 20px;
            font-weight: 600;
            font-size: 0.8rem;
        }
        
        .empresa-info {
            background: linear-gradient(135deg, #3498db, #2980b9);
            color: white;
            padding: 25px;
            border-radius: 15px;
            margin-bottom: 30px;
            position: relative;
            overflow: hidden;
        }
        
        .empresa-info::before {
            content: '';
            position: absolute;
            top: -50%;
            right: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(255,255,255,0.1) 1px, transparent 1px);
            background-size: 30px 30px;
            animation: float 20s linear infinite;
        }
        
        @keyframes float {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .empresa-info h4 {
            margin: 0 0 15px 0;
            font-weight: 700;
            position: relative;
            z-index: 2;
        }
        
        .empresa-info p {
            margin: 0;
            opacity: 0.9;
            position: relative;
            z-index: 2;
        }
        
        .config-section {
            background: #f8f9fa;
            padding: 25px;
            border-radius: 15px;
            margin-bottom: 25px;
            border: 2px solid #e9ecef;
        }
        
        .config-title {
            font-size: 1.3rem;
            font-weight: 700;
            margin-bottom: 20px;
            color: #2c3e50;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .moneda-item {
            background: white;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 10px;
            border: 1px solid #e9ecef;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s ease;
        }
        
        .moneda-item:hover {
            background: #f8f9fa;
            transform: translateX(5px);
        }
        
        .moneda-principal {
            border: 2px solid #27ae60;
            background: #e8f5e8;
        }
        
        .usuario-card {
            background: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
        }
        
        .usuario-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.15);
        }
        
        .usuario-avatar {
            width: 60px;
            height: 60px;
            background: linear-gradient(135deg, #3498db, #2980b9);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 700;
            font-size: 1.5rem;
            margin-right: 15px;
        }
        
        .alert {
            border-radius: 15px;
            border: none;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            font-weight: 600;
        }
        
        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .info-item {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 10px;
            border-left: 4px solid #3498db;
        }
        
        .info-label {
            font-weight: 600;
            color: #666;
            font-size: 0.9rem;
            margin-bottom: 5px;
        }
        
        .info-value {
            font-weight: 700;
            color: #2c3e50;
            font-size: 1.1rem;
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header -->
        <div class="header-section">
            <h1 class="header-title">
                <i class="fas fa-cogs"></i>
                Configuración del Sistema
            </h1>
            <div class="header-actions">
                <a href="dashboard.php" class="btn-header">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <a href="configuracion_dte.php" class="btn-header">
                    <i class="fas fa-certificate"></i> Config DTE
                </a>
            </div>
        </div>

        <!-- Mensajes -->
        <?php if (!empty($mensaje)): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show">
                <i class="fas fa-<?php echo $tipo_mensaje === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                <?php echo htmlspecialchars($mensaje); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Información de la Empresa -->
        <div class="empresa-info">
            <h4><?php echo htmlspecialchars($empresa['razon_social']); ?></h4>
            <p><strong>RUT:</strong> <?php echo htmlspecialchars($empresa['rut']); ?> | 
               <strong>Código:</strong> <?php echo htmlspecialchars($empresa['codigo_empresa']); ?> | 
               <strong>Estado:</strong> <?php echo ucfirst($empresa['estado']); ?>
               <?php if ($empresa['codigo_empresa'] === 'AUDITOREX_CHILE'): ?>
                   | <span class="badge bg-light text-dark">Acceso Permanente</span>
               <?php else: ?>
                   | <strong>Vence:</strong> <?php echo date('d/m/Y', strtotime($empresa['fecha_vencimiento'])); ?>
               <?php endif; ?>
            </p>
        </div>

        <!-- Estadísticas -->
        <div class="stats-cards">
            <div class="stat-card primary">
                <div class="stat-value"><?php echo number_format($estadisticas['usuarios_activos'] ?? 0); ?></div>
                <div class="stat-label">Usuarios Activos</div>
                <i class="fas fa-users stat-icon"></i>
            </div>
            
            <div class="stat-card success">
                <div class="stat-value"><?php echo number_format($estadisticas['productos_activos'] ?? 0); ?></div>
                <div class="stat-label">Productos Activos</div>
                <i class="fas fa-boxes stat-icon"></i>
            </div>
            
            <div class="stat-card warning">
                <div class="stat-value"><?php echo number_format($estadisticas['total_documentos'] ?? 0); ?></div>
                <div class="stat-label">Total Documentos</div>
                <i class="fas fa-file-invoice stat-icon"></i>
            </div>
            
            <div class="stat-card info">
                <div class="stat-value"><?php echo number_format($estadisticas['total_asientos'] ?? 0); ?></div>
                <div class="stat-label">Asientos Contables</div>
                <i class="fas fa-calculator stat-icon"></i>
            </div>
        </div>

        <!-- Navegación de configuración -->
        <div class="content-card">
            <ul class="nav nav-pills mb-4" id="configTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <a class="nav-link <?php echo $seccion === 'empresa' ? 'active' : ''; ?>" href="?seccion=empresa">
                        <i class="fas fa-building"></i> Datos de Empresa
                    </a>
                </li>
                <li class="nav-item" role="presentation">
                    <a class="nav-link <?php echo $seccion === 'usuarios' ? 'active' : ''; ?>" href="?seccion=usuarios">
                        <i class="fas fa-users"></i> Gestión de Usuarios
                    </a>
                </li>
                <li class="nav-item" role="presentation">
                    <a class="nav-link <?php echo $seccion === 'monedas' ? 'active' : ''; ?>" href="?seccion=monedas">
                        <i class="fas fa-coins"></i> Monedas
                    </a>
                </li>
                <li class="nav-item" role="presentation">
                    <a class="nav-link <?php echo $seccion === 'sistema' ? 'active' : ''; ?>" href="?seccion=sistema">
                        <i class="fas fa-server"></i> Sistema
                    </a>
                </li>
            </ul>

            <?php if ($seccion === 'empresa'): ?>
                <!-- Configuración de Empresa -->
                <div class="config-section">
                    <div class="config-title">
                        <i class="fas fa-building"></i> Datos de la Empresa
                    </div>
                    
                    <form method="POST">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Razón Social *</label>
                                <input type="text" name="razon_social" class="form-control" 
                                       value="<?php echo htmlspecialchars($empresa['razon_social']); ?>" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">RUT *</label>
                                <input type="text" name="rut" class="form-control" 
                                       value="<?php echo htmlspecialchars($empresa['rut']); ?>" required>
                            </div>
                            
                            <div class="col-md-12">
                                <label class="form-label">Dirección</label>
                                <input type="text" name="direccion" class="form-control" 
                                       value="<?php echo htmlspecialchars($empresa['direccion'] ?? ''); ?>">
                            </div>
                            
                            <div class="col-md-4">
                                <label class="form-label">Comuna</label>
                                <input type="text" name="comuna" class="form-control" 
                                       value="<?php echo htmlspecialchars($empresa['comuna'] ?? ''); ?>">
                            </div>
                            
                            <div class="col-md-4">
                                <label class="form-label">Ciudad</label>
                                <input type="text" name="ciudad" class="form-control" 
                                       value="<?php echo htmlspecialchars($empresa['ciudad'] ?? ''); ?>">
                            </div>
                            
                            <div class="col-md-4">
                                <label class="form-label">Región</label>
                                <input type="text" name="region" class="form-control" 
                                       value="<?php echo htmlspecialchars($empresa['region'] ?? ''); ?>">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Teléfono</label>
                                <input type="tel" name="telefono" class="form-control" 
                                       value="<?php echo htmlspecialchars($empresa['telefono'] ?? ''); ?>">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Email</label>
                                <input type="email" name="email" class="form-control" 
                                       value="<?php echo htmlspecialchars($empresa['email'] ?? ''); ?>">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Giro</label>
                                <input type="text" name="giro" class="form-control" 
                                       value="<?php echo htmlspecialchars($empresa['giro'] ?? ''); ?>">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Actividad Económica</label>
                                <input type="text" name="actividad_economica" class="form-control" 
                                       value="<?php echo htmlspecialchars($empresa['actividad_economica'] ?? '749000'); ?>">
                            </div>
                            
                            <div class="col-12">
                                <button type="submit" name="actualizar_empresa" class="btn btn-primary">
                                    <i class="fas fa-save"></i> Actualizar Datos de Empresa
                                </button>
                            </div>
                        </div>
                    </form>
                </div>

            <?php elseif ($seccion === 'usuarios'): ?>
                <!-- Gestión de Usuarios -->
                <div class="config-section">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div class="config-title">
                            <i class="fas fa-users"></i> Gestión de Usuarios
                        </div>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalNuevoUsuario">
                            <i class="fas fa-user-plus"></i> Nuevo Usuario
                        </button>
                    </div>
                    
                    <div class="row">
                        <?php foreach ($usuarios as $usuario): ?>
                            <div class="col-md-6 col-lg-4">
                                <div class="usuario-card">
                                    <div class="d-flex align-items-center mb-3">
                                        <div class="usuario-avatar">
                                            <?php echo strtoupper(substr($usuario['nombre'], 0, 1)); ?>
                                        </div>
                                        <div class="flex-grow-1">
                                            <h6 class="mb-1"><?php echo htmlspecialchars($usuario['nombre']); ?></h6>
                                            <small class="text-muted">@<?php echo htmlspecialchars($usuario['username']); ?></small>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-2">
                                        <strong>Email:</strong> <?php echo htmlspecialchars($usuario['email']); ?>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <span class="badge bg-<?php 
                                            switch($usuario['rol']) {
                                                case 'admin': echo 'danger'; break;
                                                case 'contador': echo 'primary'; break;
                                                case 'vendedor': echo 'success'; break;
                                                case 'auditor': echo 'warning'; break;
                                                default: echo 'secondary'; break;
                                            }
                                        ?>">
                                            <?php echo ucfirst($usuario['rol']); ?>
                                        </span>
                                        
                                        <span class="badge bg-<?php echo $usuario['estado'] === 'activo' ? 'success' : 'secondary'; ?>">
                                            <?php echo ucfirst($usuario['estado']); ?>
                                        </span>
                                    </div>
                                    
                                    <?php if ($usuario['id'] != $_SESSION['user_id']): ?>
                                        <div class="d-flex gap-2">
                                            <?php if ($usuario['estado'] === 'activo'): ?>
                                                <a href="?seccion=usuarios&cambiar_estado_usuario=<?php echo $usuario['id']; ?>&estado=inactivo" 
                                                   class="btn btn-warning btn-sm" onclick="return confirm('¿Desactivar usuario?')">
                                                    <i class="fas fa-user-slash"></i> Desactivar
                                                </a>
                                            <?php else: ?>
                                                <a href="?seccion=usuarios&cambiar_estado_usuario=<?php echo $usuario['id']; ?>&estado=activo" 
                                                   class="btn btn-success btn-sm">
                                                    <i class="fas fa-user-check"></i> Activar
                                                </a>
                                            <?php endif; ?>
                                        </div>
                                    <?php else: ?>
                                        <div class="text-muted">
                                            <i class="fas fa-crown"></i> Usuario actual
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>

            <?php elseif ($seccion === 'monedas'): ?>
                <!-- Configuración de Monedas -->
                <div class="config-section">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div class="config-title">
                            <i class="fas fa-coins"></i> Configuración de Monedas
                        </div>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalNuevaMoneda">
                            <i class="fas fa-plus"></i> Nueva Moneda
                        </button>
                    </div>
                    
                    <div class="row mb-4">
                        <div class="col-12">
                            <div class="alert alert-info">
                                <strong>Moneda Principal:</strong> <?php echo $empresa['moneda_principal'] ?? 'CLP'; ?>
                                <p class="mb-0">Esta es la moneda por defecto para todas las transacciones</p>
                            </div>
                        </div>
                    </div>
                    
                    <?php foreach ($monedas as $moneda): ?>
                        <div class="moneda-item <?php echo $moneda['codigo'] === ($empresa['moneda_principal'] ?? 'CLP') ? 'moneda-principal' : ''; ?>">
                            <div>
                                <strong><?php echo htmlspecialchars($moneda['codigo']); ?> - <?php echo htmlspecialchars($moneda['nombre']); ?></strong>
                                <br>
                                <small class="text-muted">
                                    Símbolo: <?php echo htmlspecialchars($moneda['simbolo']); ?> | 
                                    Tasa: <?php echo number_format($moneda['tasa_cambio'], 6); ?>
                                    <?php if ($moneda['codigo'] === ($empresa['moneda_principal'] ?? 'CLP')): ?>
                                        | <span class="badge bg-success">Principal</span>
                                    <?php endif; ?>
                                </small>
                            </div>
                            
                            <div>
                                <?php if ($moneda['codigo'] !== ($empresa['moneda_principal'] ?? 'CLP')): ?>
                                    <a href="?seccion=monedas&moneda_principal=<?php echo $moneda['codigo']; ?>" 
                                       class="btn btn-sm btn-success" onclick="return confirm('¿Cambiar moneda principal?')">
                                        <i class="fas fa-star"></i> Principal
                                    </a>
                                <?php endif; ?>
                                
                                <span class="badge bg-<?php echo $moneda['activo'] ? 'success' : 'secondary'; ?>">
                                    <?php echo $moneda['activo'] ? 'Activa' : 'Inactiva'; ?>
                                </span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

            <?php elseif ($seccion === 'sistema'): ?>
                <!-- Información del Sistema -->
                <div class="config-section">
                    <div class="config-title">
                        <i class="fas fa-server"></i> Información del Sistema
                    </div>
                    
                    <div class="info-grid">
                        <div class="info-item">
                            <div class="info-label">Versión del Sistema</div>
                            <div class="info-value">SISCONECTA ERP v2.0</div>
                        </div>
                        
                        <div class="info-item">
                            <div class="info-label">Empresa</div>
                            <div class="info-value"><?php echo htmlspecialchars($empresa['razon_social']); ?></div>
                        </div>
                        
                        <div class="info-item">
                            <div class="info-label">Código de Empresa</div>
                            <div class="info-value"><?php echo htmlspecialchars($empresa['codigo_empresa']); ?></div>
                        </div>
                        
                        <div class="info-item">
                            <div class="info-label">Estado de Suscripción</div>
                            <div class="info-value">
                                <?php if ($empresa['codigo_empresa'] === 'AUDITOREX_CHILE'): ?>
                                    <span class="text-success">Acceso Permanente</span>
                                <?php else: ?>
                                    <?php
                                    $fecha_vencimiento = new DateTime($empresa['fecha_vencimiento']);
                                    $hoy = new DateTime();
                                    $dias_restantes = $hoy->diff($fecha_vencimiento)->days;
                                    $vencido = $fecha_vencimiento < $hoy;
                                    ?>
                                    <span class="<?php echo $vencido ? 'text-danger' : ($dias_restantes < 30 ? 'text-warning' : 'text-success'); ?>">
                                        <?php if ($vencido): ?>
                                            Vencida
                                        <?php else: ?>
                                            <?php echo $dias_restantes; ?> días restantes
                                        <?php endif; ?>
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="info-item">
                            <div class="info-label">Moneda Principal</div>
                            <div class="info-value"><?php echo $empresa['moneda_principal'] ?? 'CLP'; ?></div>
                        </div>
                        
                        <div class="info-item">
                            <div class="info-label">Configuración DTE</div>
                            <div class="info-value">
                                <?php if ($config_dte && $config_dte['certificado_digital']): ?>
                                    <span class="text-success">Configurado</span>
                                <?php else: ?>
                                    <span class="text-warning">Pendiente</span>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="info-item">
                            <div class="info-label">Ambiente DTE</div>
                            <div class="info-value">
                                <?php echo ucfirst($config_dte['ambiente'] ?? 'No configurado'); ?>
                            </div>
                        </div>
                        
                        <div class="info-item">
                            <div class="info-label">Fecha de Registro</div>
                            <div class="info-value"><?php echo date('d/m/Y', strtotime($empresa['fecha_registro'])); ?></div>
                        </div>
                    </div>
                    
                    <div class="mt-4">
                        <h5><i class="fas fa-info-circle"></i> Información Técnica</h5>
                        <div class="alert alert-light">
                            <p><strong>Base de datos:</strong> <?php echo $dbname; ?></p>
                            <p><strong>Servidor:</strong> <?php echo $host; ?></p>
                            <p><strong>Versión PHP:</strong> <?php echo phpversion(); ?></p>
                            <p class="mb-0"><strong>Zona horaria:</strong> <?php echo date_default_timezone_get(); ?></p>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Modal Nuevo Usuario -->
    <div class="modal fade" id="modalNuevoUsuario" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Crear Nuevo Usuario</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Nombre Completo *</label>
                            <input type="text" name="nombre" class="form-control" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Nombre de Usuario *</label>
                            <input type="text" name="username" class="form-control" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Email *</label>
                            <input type="email" name="email" class="form-control" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Contraseña *</label>
                            <input type="password" name="password" class="form-control" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Rol *</label>
                            <select name="rol" class="form-control" required>
                                <option value="">Seleccionar rol...</option>
                                <option value="admin">Administrador</option>
                                <option value="contador">Contador</option>
                                <option value="vendedor">Vendedor</option>
                                <option value="auditor">Auditor</option>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" name="crear_usuario" class="btn btn-primary">Crear Usuario</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Nueva Moneda -->
    <div class="modal fade" id="modalNuevaMoneda" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Agregar Nueva Moneda</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Código (3 letras) *</label>
                            <input type="text" name="codigo" class="form-control" maxlength="3" placeholder="USD" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Nombre *</label>
                            <input type="text" name="nombre" class="form-control" placeholder="Dólar Estadounidense" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Símbolo *</label>
                            <input type="text" name="simbolo" class="form-control" maxlength="5" placeholder="US$" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Tasa de Cambio *</label>
                            <input type="number" name="tasa_cambio" class="form-control" step="0.000001" min="0.000001" placeholder="950.000000" required>
                            <small class="text-muted">Valor en relación a la moneda principal</small>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" name="guardar_moneda" class="btn btn-primary">Guardar Moneda</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Animaciones de entrada
        document.addEventListener('DOMContentLoaded', function() {
            const cards = document.querySelectorAll('.stat-card, .usuario-card, .moneda-item');
            cards.forEach((card, index) => {
                card.style.opacity = '0';
                card.style.transform = 'translateY(20px)';
                setTimeout(() => {
                    card.style.transition = 'all 0.6s ease';
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                }, index * 100);
            });
        });
        
        // Formateo automático de campos
        document.addEventListener('DOMContentLoaded', function() {
            // Formatear RUT mientras se escribe
            const rutInput = document.querySelector('input[name="rut"]');
            if (rutInput) {
                rutInput.addEventListener('input', function(e) {
                    let value = e.target.value.replace(/[^0-9kK]/g, '');
                    if (value.length > 1) {
                        const numero = value.slice(0, -1);
                        const dv = value.slice(-1).toUpperCase();
                        if (numero.length > 0) {
                            const formatted = numero.replace(/\B(?=(\d{3})+(?!\d))/g, '.') + '-' + dv;
                            e.target.value = formatted;
                        }
                    }
                });
            }
            
            // Convertir código de moneda a mayúsculas
            const codigoInput = document.querySelector('input[name="codigo"]');
            if (codigoInput) {
                codigoInput.addEventListener('input', function(e) {
                    e.target.value = e.target.value.toUpperCase();
                });
            }
        });
        
        // Confirmaciones para acciones importantes
        document.querySelectorAll('a[onclick*="confirm"]').forEach(link => {
            link.addEventListener('click', function(e) {
                const confirmText = this.getAttribute('onclick').match(/confirm\('([^']+)'\)/)[1];
                if (!confirm(confirmText)) {
                    e.preventDefault();
                    return false;
                }
            });
        });
        
        console.log('Módulo de Configuración - SISCONECTA ERP v2.0');
        console.log('Sistema de gestión empresarial para Chile');
    </script>
</body>
</html>