<?php
// CONFIGURACIÓN DTE - CERTIFICADOS DIGITALES Y CAF
// configuracion_dte.php

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

if (!isset($_SESSION['user_id']) || !isset($_SESSION['empresa_id'])) {
    header('Location: login.php');
    exit;
}

// Configuración de base de datos
$config = [
    'host' => 'localhost',
    'dbname' => 'conectae_sisconecta',
    'username' => 'conectae_sisconectauser',
    'password' => 'pt125824caraud',
    'charset' => 'utf8mb4'
];

$mensaje = '';
$tipo_mensaje = '';

try {
    $pdo = new PDO(
        "mysql:host={$config['host']};dbname={$config['dbname']};charset={$config['charset']}", 
        $config['username'], 
        $config['password'],
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ]
    );
    
    $empresa_id = $_SESSION['empresa_id'];
    
    // Obtener datos de la empresa
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ? AND estado = 'activo'");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch();
    
    if (!$empresa) {
        header('Location: login.php?error=empresa_inactiva');
        exit;
    }
    
    // Obtener configuración DTE existente
    $stmt = $pdo->prepare("SELECT * FROM configuracion_dte WHERE empresa_id = ?");
    $stmt->execute([$empresa_id]);
    $config_dte = $stmt->fetch();
    
    // Procesar guardado de configuración
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['guardar_config'])) {
        $ambiente = $_POST['ambiente'] ?? 'certificacion';
        $resolucion_sii = trim($_POST['resolucion_sii'] ?? '');
        $fecha_resolucion = !empty($_POST['fecha_resolucion']) ? $_POST['fecha_resolucion'] : null;
        $actividad_economica = trim($_POST['actividad_economica'] ?? '749000');
        $codigo_sucursal = trim($_POST['codigo_sucursal'] ?? '1');
        $password_certificado = $_POST['password_certificado'] ?? '';
        
        try {
            $pdo->beginTransaction();
            
            // Procesar certificado digital si se subió
            $certificado_digital = null;
            if (isset($_FILES['certificado_digital']) && $_FILES['certificado_digital']['error'] === UPLOAD_ERR_OK) {
                $archivo_temp = $_FILES['certificado_digital']['tmp_name'];
                $extension = strtolower(pathinfo($_FILES['certificado_digital']['name'], PATHINFO_EXTENSION));
                
                if ($extension !== 'p12' && $extension !== 'pfx') {
                    throw new Exception('El certificado debe ser un archivo .p12 o .pfx');
                }
                
                $contenido_certificado = file_get_contents($archivo_temp);
                if (!$contenido_certificado) {
                    throw new Exception('Error al leer el archivo del certificado');
                }
                
                // Validar certificado
                $certs = [];
                if (!openssl_pkcs12_read($contenido_certificado, $certs, $password_certificado)) {
                    throw new Exception('No se pudo abrir el certificado. Verifique la contraseña.');
                }
                
                // Validar que el certificado sea válido para DTE
                $cert_info = openssl_x509_parse($certs['cert']);
                if (!$cert_info) {
                    throw new Exception('Certificado inválido');
                }
                
                // Verificar RUT en el certificado
                $subject = $cert_info['subject']['CN'] ?? '';
                if (!preg_match('/(\d{1,2}\.\d{3}\.\d{3}-[\dkK])/i', $subject, $matches)) {
                    throw new Exception('El certificado no contiene un RUT válido');
                }
                
                $rut_certificado = str_replace(['.', '-'], '', strtoupper($matches[1]));
                $rut_empresa = str_replace(['.', '-'], '', strtoupper($empresa['rut']));
                
                if ($rut_certificado !== $rut_empresa) {
                    throw new Exception('El RUT del certificado no coincide con el RUT de la empresa');
                }
                
                $certificado_digital = base64_encode($contenido_certificado);
            } else if ($config_dte) {
                $certificado_digital = $config_dte['certificado_digital'];
            }
            
            // Guardar o actualizar configuración
            if ($config_dte) {
                $stmt = $pdo->prepare("
                    UPDATE configuracion_dte 
                    SET certificado_digital = COALESCE(?, certificado_digital),
                        password_certificado = COALESCE(NULLIF(?, ''), password_certificado),
                        ambiente = ?, resolucion_sii = ?, fecha_resolucion = ?, 
                        actividad_economica = ?, codigo_sucursal = ?, fecha_actualizacion = NOW()
                    WHERE empresa_id = ?
                ");
                $stmt->execute([
                    $certificado_digital, $password_certificado, $ambiente, $resolucion_sii, 
                    $fecha_resolucion, $actividad_economica, $codigo_sucursal, $empresa_id
                ]);
            } else {
                $stmt = $pdo->prepare("
                    INSERT INTO configuracion_dte 
                    (empresa_id, certificado_digital, password_certificado, ambiente, resolucion_sii, 
                     fecha_resolucion, actividad_economica, codigo_sucursal) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $empresa_id, $certificado_digital, $password_certificado, $ambiente, $resolucion_sii,
                    $fecha_resolucion, $actividad_economica, $codigo_sucursal
                ]);
            }
            
            $pdo->commit();
            
            $mensaje = 'Configuración DTE guardada exitosamente';
            $tipo_mensaje = 'success';
            
            // Recargar configuración
            $stmt = $pdo->prepare("SELECT * FROM configuracion_dte WHERE empresa_id = ?");
            $stmt->execute([$empresa_id]);
            $config_dte = $stmt->fetch();
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $mensaje = 'Error al guardar configuración: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
    
    // Procesar subida de CAF
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['subir_caf'])) {
        $tipo_documento = $_POST['tipo_documento_caf'] ?? '';
        
        if (!isset($_FILES['archivo_caf']) || $_FILES['archivo_caf']['error'] !== UPLOAD_ERR_OK) {
            $mensaje = 'Debe seleccionar un archivo CAF válido';
            $tipo_mensaje = 'warning';
        } else {
            try {
                $archivo_temp = $_FILES['archivo_caf']['tmp_name'];
                $contenido_caf = file_get_contents($archivo_temp);
                
                if (!$contenido_caf) {
                    throw new Exception('Error al leer el archivo CAF');
                }
                
                // Validar XML CAF
                $dom = new DOMDocument();
                if (!$dom->loadXML($contenido_caf)) {
                    throw new Exception('El archivo CAF no es un XML válido');
                }
                
                // Extraer información del CAF
                $xpath = new DOMXPath($dom);
                
                // Buscar nodos principales
                $td_node = $xpath->query('//TD')->item(0);
                $folio_desde_node = $xpath->query('//RNG/D')->item(0);
                $folio_hasta_node = $xpath->query('//RNG/H')->item(0);
                $fecha_autorizacion_node = $xpath->query('//FA')->item(0);
                $fecha_vencimiento_node = $xpath->query('//FECHA')->item(0);
                
                if (!$td_node || !$folio_desde_node || !$folio_hasta_node) {
                    throw new Exception('El archivo CAF no tiene el formato correcto');
                }
                
                $tipo_documento_caf = (int)$td_node->nodeValue;
                $folio_desde = (int)$folio_desde_node->nodeValue;
                $folio_hasta = (int)$folio_hasta_node->nodeValue;
                $fecha_autorizacion = $fecha_autorizacion_node ? $fecha_autorizacion_node->nodeValue : date('Y-m-d');
                $fecha_vencimiento = $fecha_vencimiento_node ? $fecha_vencimiento_node->nodeValue : date('Y-m-d', strtotime('+6 months'));
                
                // Mapear tipo de documento
                $tipos_documento = [
                    33 => 'factura',
                    39 => 'boleta', 
                    61 => 'nota_credito',
                    56 => 'nota_debito',
                    52 => 'guia_despacho'
                ];
                
                $tipo_doc_sistema = $tipos_documento[$tipo_documento_caf] ?? null;
                if (!$tipo_doc_sistema) {
                    throw new Exception('Tipo de documento no soportado: ' . $tipo_documento_caf);
                }
                
                if ($tipo_documento && $tipo_doc_sistema !== $tipo_documento) {
                    throw new Exception('El tipo de documento del CAF no coincide con el seleccionado');
                }
                
                // Verificar que no exista un CAF activo para este rango
                $stmt = $pdo->prepare("
                    SELECT id FROM folios_caf 
                    WHERE empresa_id = ? AND tipo_documento = ? 
                    AND ((folio_desde <= ? AND folio_hasta >= ?) OR (folio_desde <= ? AND folio_hasta >= ?))
                    AND estado = 'activo'
                ");
                $stmt->execute([$empresa_id, $tipo_doc_sistema, $folio_desde, $folio_desde, $folio_hasta, $folio_hasta]);
                
                if ($stmt->fetch()) {
                    throw new Exception('Ya existe un CAF activo que se superpone con este rango de folios');
                }
                
                // Insertar CAF
                $stmt = $pdo->prepare("
                    INSERT INTO folios_caf 
                    (empresa_id, tipo_documento, folio_desde, folio_hasta, folio_actual, 
                     caf_xml, fecha_autorizacion, fecha_vencimiento, estado) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'activo')
                ");
                $stmt->execute([
                    $empresa_id, $tipo_doc_sistema, $folio_desde, $folio_hasta, $folio_desde,
                    $contenido_caf, $fecha_autorizacion, $fecha_vencimiento
                ]);
                
                $mensaje = "CAF cargado exitosamente: {$tipo_doc_sistema} (folios {$folio_desde} - {$folio_hasta})";
                $tipo_mensaje = 'success';
                
            } catch (Exception $e) {
                $mensaje = 'Error al procesar CAF: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        }
    }
    
    // Obtener CAFs existentes
    $stmt = $pdo->prepare("
        SELECT *, 
               (folio_hasta - folio_actual + 1) as folios_disponibles,
               CASE 
                   WHEN fecha_vencimiento < CURDATE() THEN 'vencido'
                   WHEN folio_actual > folio_hasta THEN 'agotado'
                   ELSE estado 
               END as estado_real
        FROM folios_caf 
        WHERE empresa_id = ? 
        ORDER BY tipo_documento, fecha_autorizacion DESC
    ");
    $stmt->execute([$empresa_id]);
    $cafs = $stmt->fetchAll();
    
} catch (PDOException $e) {
    die('Error de conexión: ' . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Configuración DTE - SISCONECTA ERP</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .main-container {
            padding: 20px;
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .header-section {
            background: linear-gradient(45deg, #2c3e50, #34495e);
            color: white;
            padding: 25px 30px;
            border-radius: 15px;
            margin-bottom: 25px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }
        
        .content-card {
            background: white;
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            margin-bottom: 25px;
        }
        
        .nav-pills .nav-link {
            border-radius: 10px;
            font-weight: 600;
            margin-right: 10px;
        }
        
        .nav-pills .nav-link.active {
            background: linear-gradient(135deg, #3498db, #2980b9);
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 15px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #3498db;
            box-shadow: 0 0 0 0.25rem rgba(52,152,219,0.25);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #3498db, #2980b9);
            border: none;
            border-radius: 10px;
            padding: 12px 25px;
            font-weight: 600;
        }
        
        .table {
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .table th {
            background: linear-gradient(135deg, #34495e, #2c3e50);
            color: white;
            border: none;
            font-weight: 700;
        }
        
        .badge {
            padding: 8px 12px;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .alert {
            border-radius: 15px;
            border: none;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .file-upload {
            border: 2px dashed #3498db;
            border-radius: 10px;
            padding: 40px;
            text-align: center;
            transition: all 0.3s ease;
        }
        
        .file-upload:hover {
            background: rgba(52,152,219,0.05);
        }
        
        .progress {
            height: 8px;
            border-radius: 10px;
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header -->
        <div class="header-section">
            <div class="d-flex justify-content-between align-items-center">
                <h1 class="mb-0">
                    <i class="fas fa-cog"></i>
                    Configuración DTE
                    <span class="badge bg-danger ms-2">IMPORTANTE</span>
                </h1>
                <a href="facturacion.php" class="btn btn-light">
                    <i class="fas fa-arrow-left"></i> Volver a Facturación
                </a>
            </div>
        </div>

        <!-- Mensajes -->
        <?php if (!empty($mensaje)): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show">
                <i class="fas fa-<?php echo $tipo_mensaje === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                <?php echo htmlspecialchars($mensaje); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Navegación por pestañas -->
        <div class="content-card">
            <ul class="nav nav-pills mb-4" id="configTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="certificado-tab" data-bs-toggle="pill" data-bs-target="#certificado" type="button" role="tab">
                        <i class="fas fa-certificate"></i> Certificado Digital
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="caf-tab" data-bs-toggle="pill" data-bs-target="#caf" type="button" role="tab">
                        <i class="fas fa-file-signature"></i> Folios CAF
                    </button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="parametros-tab" data-bs-toggle="pill" data-bs-target="#parametros" type="button" role="tab">
                        <i class="fas fa-sliders-h"></i> Parámetros
                    </button>
                </li>
            </ul>

            <div class="tab-content" id="configTabsContent">
                <!-- Certificado Digital -->
                <div class="tab-pane fade show active" id="certificado" role="tabpanel">
                    <h4><i class="fas fa-certificate"></i> Certificado Digital</h4>
                    <p class="text-muted mb-4">Configure el certificado digital (.p12/.pfx) para firmar documentos DTE</p>
                    
                    <form method="POST" enctype="multipart/form-data">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Ambiente SII</label>
                                <select name="ambiente" class="form-control" required>
                                    <option value="certificacion" <?php echo ($config_dte['ambiente'] ?? 'certificacion') === 'certificacion' ? 'selected' : ''; ?>>
                                        Certificación (Pruebas)
                                    </option>
                                    <option value="produccion" <?php echo ($config_dte['ambiente'] ?? '') === 'produccion' ? 'selected' : ''; ?>>
                                        Producción (Real)
                                    </option>
                                </select>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Contraseña del Certificado</label>
                                <input type="password" name="password_certificado" class="form-control" 
                                       placeholder="Ingrese la contraseña del certificado"
                                       <?php echo $config_dte ? '' : 'required'; ?>>
                                <?php if ($config_dte && $config_dte['certificado_digital']): ?>
                                    <small class="text-muted">Certificado actual instalado ✓</small>
                                <?php endif; ?>
                            </div>
                            
                            <div class="col-12">
                                <label class="form-label">Archivo del Certificado (.p12 / .pfx)</label>
                                <div class="file-upload">
                                    <i class="fas fa-cloud-upload-alt fa-3x text-primary mb-3"></i>
                                    <h5>Subir Certificado Digital</h5>
                                    <p class="text-muted">Seleccione el archivo .p12 o .pfx del certificado digital</p>
                                    <input type="file" name="certificado_digital" class="form-control" 
                                           accept=".p12,.pfx" <?php echo $config_dte ? '' : 'required'; ?>>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Resolución SII</label>
                                <input type="text" name="resolucion_sii" class="form-control" 
                                       value="<?php echo htmlspecialchars($config_dte['resolucion_sii'] ?? ''); ?>"
                                       placeholder="Número de resolución del SII">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Fecha Resolución</label>
                                <input type="date" name="fecha_resolucion" class="form-control" 
                                       value="<?php echo $config_dte['fecha_resolucion'] ?? ''; ?>">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Actividad Económica</label>
                                <input type="text" name="actividad_economica" class="form-control" 
                                       value="<?php echo htmlspecialchars($config_dte['actividad_economica'] ?? '749000'); ?>"
                                       placeholder="Código actividad económica">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Código Sucursal</label>
                                <input type="text" name="codigo_sucursal" class="form-control" 
                                       value="<?php echo htmlspecialchars($config_dte['codigo_sucursal'] ?? '1'); ?>"
                                       placeholder="Código de sucursal">
                            </div>
                            
                            <div class="col-12">
                                <button type="submit" name="guardar_config" class="btn btn-primary">
                                    <i class="fas fa-save"></i> Guardar Configuración
                                </button>
                            </div>
                        </div>
                    </form>
                </div>

                <!-- Folios CAF -->
                <div class="tab-pane fade" id="caf" role="tabpanel">
                    <h4><i class="fas fa-file-signature"></i> Folios CAF (Código de Autorización de Folios)</h4>
                    <p class="text-muted mb-4">Suba los archivos CAF obtenidos desde el sitio web del SII</p>
                    
                    <!-- Formulario subir CAF -->
                    <div class="row mb-4">
                        <div class="col-12">
                            <form method="POST" enctype="multipart/form-data" class="border rounded p-4">
                                <h5><i class="fas fa-upload"></i> Subir Nuevo CAF</h5>
                                <div class="row g-3">
                                    <div class="col-md-4">
                                        <label class="form-label">Tipo de Documento</label>
                                        <select name="tipo_documento_caf" class="form-control">
                                            <option value="">Detectar automáticamente</option>
                                            <option value="factura">Factura Electrónica</option>
                                            <option value="boleta">Boleta Electrónica</option>
                                            <option value="nota_credito">Nota de Crédito</option>
                                            <option value="nota_debito">Nota de Débito</option>
                                            <option value="guia_despacho">Guía de Despacho</option>
                                        </select>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <label class="form-label">Archivo CAF (.xml)</label>
                                        <input type="file" name="archivo_caf" class="form-control" 
                                               accept=".xml" required>
                                    </div>
                                    
                                    <div class="col-md-2">
                                        <label class="form-label">&nbsp;</label>
                                        <button type="submit" name="subir_caf" class="btn btn-primary w-100">
                                            <i class="fas fa-upload"></i> Subir
                                        </button>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Lista de CAFs -->
                    <h5><i class="fas fa-list"></i> CAFs Instalados</h5>
                    <?php if (empty($cafs)): ?>
                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle"></i>
                            No hay archivos CAF instalados. Debe descargar y subir los CAFs desde el sitio web del SII.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Tipo Documento</th>
                                        <th>Rango Folios</th>
                                        <th>Folio Actual</th>
                                        <th>Disponibles</th>
                                        <th>Autorización</th>
                                        <th>Vencimiento</th>
                                        <th>Estado</th>
                                        <th>Progreso</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($cafs as $caf): ?>
                                        <tr>
                                            <td>
                                                <span class="badge bg-info">
                                                    <?php echo ucfirst(str_replace('_', ' ', $caf['tipo_documento'])); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <code><?php echo number_format($caf['folio_desde']); ?> - <?php echo number_format($caf['folio_hasta']); ?></code>
                                            </td>
                                            <td>
                                                <strong><?php echo number_format($caf['folio_actual']); ?></strong>
                                            </td>
                                            <td>
                                                <span class="badge bg-<?php echo $caf['folios_disponibles'] > 10 ? 'success' : ($caf['folios_disponibles'] > 0 ? 'warning' : 'danger'); ?>">
                                                    <?php echo number_format($caf['folios_disponibles']); ?>
                                                </span>
                                            </td>
                                            <td><?php echo date('d/m/Y', strtotime($caf['fecha_autorizacion'])); ?></td>
                                            <td>
                                                <?php 
                                                $vencimiento = new DateTime($caf['fecha_vencimiento']);
                                                $hoy = new DateTime();
                                                $dias_venc = $hoy->diff($vencimiento)->days;
                                                $vencido = $vencimiento < $hoy;
                                                ?>
                                                <span class="<?php echo $vencido ? 'text-danger' : ($dias_venc < 30 ? 'text-warning' : 'text-success'); ?>">
                                                    <?php echo date('d/m/Y', strtotime($caf['fecha_vencimiento'])); ?>
                                                    <?php if ($vencido): ?>
                                                        <br><small>(Vencido)</small>
                                                    <?php elseif ($dias_venc < 30): ?>
                                                        <br><small>(<?php echo $dias_venc; ?> días)</small>
                                                    <?php endif; ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge bg-<?php 
                                                    switch($caf['estado_real']) {
                                                        case 'activo': echo 'success'; break;
                                                        case 'agotado': echo 'warning'; break;
                                                        case 'vencido': echo 'danger'; break;
                                                        default: echo 'secondary'; break;
                                                    }
                                                ?>">
                                                    <?php echo ucfirst($caf['estado_real']); ?>
                                                </span>
                                            </td>
                                            <td style="width: 150px;">
                                                <?php 
                                                $total_folios = $caf['folio_hasta'] - $caf['folio_desde'] + 1;
                                                $folios_usados = $caf['folio_actual'] - $caf['folio_desde'];
                                                $porcentaje = min(100, ($folios_usados / $total_folios) * 100);
                                                ?>
                                                <div class="progress">
                                                    <div class="progress-bar bg-<?php echo $porcentaje > 80 ? 'danger' : ($porcentaje > 60 ? 'warning' : 'success'); ?>" 
                                                         style="width: <?php echo $porcentaje; ?>%">
                                                    </div>
                                                </div>
                                                <small class="text-muted"><?php echo number_format($porcentaje, 1); ?>% usado</small>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Parámetros -->
                <div class="tab-pane fade" id="parametros" role="tabpanel">
                    <h4><i class="fas fa-sliders-h"></i> Parámetros del Sistema</h4>
                    
                    <div class="row g-4">
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header">
                                    <h5><i class="fas fa-server"></i> Configuración SII</h5>
                                </div>
                                <div class="card-body">
                                    <dl class="row">
                                        <dt class="col-sm-6">Ambiente:</dt>
                                        <dd class="col-sm-6">
                                            <span class="badge bg-<?php echo ($config_dte['ambiente'] ?? 'certificacion') === 'produccion' ? 'success' : 'warning'; ?>">
                                                <?php echo ucfirst($config_dte['ambiente'] ?? 'certificacion'); ?>
                                            </span>
                                        </dd>
                                        
                                        <dt class="col-sm-6">Resolución:</dt>
                                        <dd class="col-sm-6"><?php echo $config_dte['resolucion_sii'] ?? 'No configurada'; ?></dd>
                                        
                                        <dt class="col-sm-6">Actividad Econ.:</dt>
                                        <dd class="col-sm-6"><?php echo $config_dte['actividad_economica'] ?? '749000'; ?></dd>
                                    </dl>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header">
                                    <h5><i class="fas fa-info-circle"></i> Estado del Sistema</h5>
                                </div>
                                <div class="card-body">
                                    <dl class="row">
                                        <dt class="col-sm-6">Certificado:</dt>
                                        <dd class="col-sm-6">
                                            <?php if ($config_dte && $config_dte['certificado_digital']): ?>
                                                <span class="badge bg-success">Instalado</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">No instalado</span>
                                            <?php endif; ?>
                                        </dd>
                                        
                                        <dt class="col-sm-6">CAFs Activos:</dt>
                                        <dd class="col-sm-6">
                                            <span class="badge bg-info">
                                                <?php echo count(array_filter($cafs, fn($c) => $c['estado_real'] === 'activo')); ?>
                                            </span>
                                        </dd>
                                        
                                        <dt class="col-sm-6">Estado DTE:</dt>
                                        <dd class="col-sm-6">
                                            <?php 
                                            $tiene_cert = $config_dte && $config_dte['certificado_digital'];
                                            $tiene_caf = count(array_filter($cafs, fn($c) => $c['estado_real'] === 'activo')) > 0;
                                            ?>
                                            <?php if ($tiene_cert && $tiene_caf): ?>
                                                <span class="badge bg-success">Operativo</span>
                                            <?php else: ?>
                                                <span class="badge bg-warning">Configuración incompleta</span>
                                            <?php endif; ?>
                                        </dd>
                                    </dl>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="alert alert-info mt-4">
                        <h5><i class="fas fa-info-circle"></i> Instrucciones</h5>
                        <ol>
                            <li><strong>Certificado Digital:</strong> Obtenga su certificado digital desde un proveedor autorizado (ej: E-Sign, AC-Camerfirma)</li>
                            <li><strong>Resolución SII:</strong> Solicite autorización para emisión de DTE en el sitio web del SII</li>
                            <li><strong>Folios CAF:</strong> Descargue los archivos CAF desde el portal SII para cada tipo de documento</li>
                            <li><strong>Ambiente:</strong> Use "Certificación" para pruebas y "Producción" para documentos reales</li>
                        </ol>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Validar formato de archivo al seleccionar
        document.querySelectorAll('input[type="file"]').forEach(input => {
            input.addEventListener('change', function(e) {
                const file = e.target.files[0];
                if (!file) return;
                
                const allowedExtensions = this.accept.split(',').map(ext => ext.trim());
                const fileName = file.name.toLowerCase();
                const isValidExtension = allowedExtensions.some(ext => 
                    fileName.endsWith(ext.replace('*', '').replace('.', ''))
                );
                
                if (!isValidExtension) {
                    alert(`Archivo no válido. Extensiones permitidas: ${allowedExtensions.join(', ')}`);
                    this.value = '';
                    return;
                }
                
                // Mostrar información del archivo
                const fileInfo = document.createElement('small');
                fileInfo.className = 'text-muted d-block mt-1';
                fileInfo.textContent = `Archivo seleccionado: ${file.name} (${(file.size / 1024).toFixed(1)} KB)`;
                
                // Remover info anterior
                const existingInfo = this.parentNode.querySelector('small.text-muted');
                if (existingInfo && existingInfo.textContent.includes('Archivo seleccionado:')) {
                    existingInfo.remove();
                }
                
                this.parentNode.appendChild(fileInfo);
            });
        });
        
        // Confirmar antes de subir archivos importantes
        document.querySelectorAll('form').forEach(form => {
            form.addEventListener('submit', function(e) {
                const submitBtn = form.querySelector('button[type="submit"]');
                const isCAF = submitBtn && submitBtn.name === 'subir_caf';
                const isCert = form.querySelector('input[name="certificado_digital"]')?.files?.length > 0;
                
                if (isCAF || isCert) {
                    const message = isCAF 
                        ? '¿Confirma la subida del archivo CAF? Esto habilitará nuevos folios para facturación.'
                        : '¿Confirma la instalación del certificado digital? Esto reemplazará el certificado actual.';
                    
                    if (!confirm(message)) {
                        e.preventDefault();
                        return false;
                    }
                }
                
                // Mostrar indicador de carga
                if (submitBtn) {
                    submitBtn.disabled = true;
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Procesando...';
                }
            });
        });
        
        console.log('Configuración DTE - SISCONECTA ERP');
        console.log('Sistema de configuración de certificados digitales y CAF para DTE SII Chile');
    </script>
</body>
</html>