<?php
// MÓDULO DE CONTABILIDAD - SISCONECTA ERP
// contabilidad.php

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Verificar sesión activa
if (!isset($_SESSION['user_id']) || !isset($_SESSION['empresa_id'])) {
    header('Location: login.php');
    exit;
}

// Configuración de base de datos
$host = 'localhost';
$dbname = 'conectae_sisconecta';
$username = 'conectae_sisconectauser';
$password = 'pt125824caraud';

$mensaje = '';
$tipo_mensaje = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $empresa_id = $_SESSION['empresa_id'];
    
    // Verificar estado de empresa
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ? AND estado = 'activo'");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch();
    
    if (!$empresa) {
        session_destroy();
        header('Location: login.php?error=empresa_inactiva');
        exit;
    }
    
    // Verificar vencimiento (excepto AUDITOREX CHILE)
    if (($empresa['codigo_empresa'] ?? '') !== 'AUDITOREX_CHILE') {
        $fecha_actual = new DateTime();
        $fecha_vencimiento = new DateTime($empresa['fecha_vencimiento'] ?? '2024-12-31');
        
        if ($fecha_actual > $fecha_vencimiento) {
            session_destroy();
            header('Location: login.php?error=suscripcion_vencida');
            exit;
        }
    }
    
    // Procesar acciones
    $accion = $_GET['accion'] ?? 'dashboard';
    
    // Agregar nueva cuenta
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['agregar_cuenta'])) {
        $codigo_cuenta = trim($_POST['codigo_cuenta']);
        $nombre_cuenta = trim($_POST['nombre_cuenta']);
        $tipo_cuenta = $_POST['tipo_cuenta'];
        $cuenta_padre = !empty($_POST['cuenta_padre']) ? $_POST['cuenta_padre'] : null;
        $nivel = $_POST['nivel'] ?? 1;
        
        if (!empty($codigo_cuenta) && !empty($nombre_cuenta)) {
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO plan_cuentas (empresa_id, codigo_cuenta, nombre_cuenta, tipo_cuenta, cuenta_padre, nivel) 
                    VALUES (?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$empresa_id, $codigo_cuenta, $nombre_cuenta, $tipo_cuenta, $cuenta_padre, $nivel]);
                
                $mensaje = 'Cuenta agregada exitosamente';
                $tipo_mensaje = 'success';
            } catch (PDOException $e) {
                $mensaje = 'Error al agregar cuenta: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Complete todos los campos obligatorios';
            $tipo_mensaje = 'warning';
        }
    }
    
    // Crear asiento contable
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['crear_asiento'])) {
        $fecha = $_POST['fecha'];
        $descripcion = trim($_POST['descripcion']);
        $movimientos = json_decode($_POST['movimientos'], true);
        
        if (!empty($descripcion) && !empty($movimientos) && count($movimientos) >= 2) {
            try {
                $pdo->beginTransaction();
                
                // Obtener siguiente número de asiento
                $stmt = $pdo->prepare("SELECT COALESCE(MAX(numero_asiento), 0) + 1 as siguiente FROM asientos_contables WHERE empresa_id = ?");
                $stmt->execute([$empresa_id]);
                $numero_asiento = $stmt->fetch()['siguiente'];
                
                // Validar que debe = haber
                $total_debe = array_sum(array_column($movimientos, 'debe'));
                $total_haber = array_sum(array_column($movimientos, 'haber'));
                
                if (abs($total_debe - $total_haber) > 0.01) {
                    throw new Exception('El total del DEBE debe ser igual al total del HABER');
                }
                
                // Crear asiento
                $stmt = $pdo->prepare("
                    INSERT INTO asientos_contables (empresa_id, numero_asiento, fecha, descripcion, tipo_asiento, estado, usuario_registro) 
                    VALUES (?, ?, ?, ?, 'manual', 'contabilizado', ?)
                ");
                $stmt->execute([$empresa_id, $numero_asiento, $fecha, $descripcion, $_SESSION['user_id']]);
                $asiento_id = $pdo->lastInsertId();
                
                // Crear movimientos
                foreach ($movimientos as $mov) {
                    if (!empty($mov['cuenta']) && (!empty($mov['debe']) || !empty($mov['haber']))) {
                        $stmt = $pdo->prepare("
                            INSERT INTO movimientos_contables (asiento_id, cuenta_contable, nombre_cuenta, debe, haber, observaciones) 
                            VALUES (?, ?, ?, ?, ?, ?)
                        ");
                        $stmt->execute([
                            $asiento_id, 
                            $mov['cuenta'], 
                            $mov['nombre_cuenta'] ?? '', 
                            $mov['debe'] ?? 0, 
                            $mov['haber'] ?? 0, 
                            $mov['observaciones'] ?? ''
                        ]);
                    }
                }
                
                $pdo->commit();
                
                $mensaje = "Asiento contable #{$numero_asiento} creado exitosamente";
                $tipo_mensaje = 'success';
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $mensaje = 'Error al crear asiento: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Complete la descripción y agregue al menos 2 movimientos válidos';
            $tipo_mensaje = 'warning';
        }
    }
    
    // Obtener plan de cuentas
    $stmt = $pdo->prepare("
        SELECT * FROM plan_cuentas 
        WHERE (empresa_id = ? OR empresa_id IS NULL) AND activo = 1 
        ORDER BY codigo_cuenta ASC
    ");
    $stmt->execute([$empresa_id]);
    $plan_cuentas = $stmt->fetchAll();
    
    // Obtener últimos asientos
    $stmt = $pdo->prepare("
        SELECT a.*, u.nombre as usuario_nombre,
               COUNT(m.id) as total_movimientos,
               SUM(m.debe) as total_debe,
               SUM(m.haber) as total_haber
        FROM asientos_contables a
        LEFT JOIN usuarios u ON a.usuario_registro = u.id
        LEFT JOIN movimientos_contables m ON a.id = m.asiento_id
        WHERE a.empresa_id = ?
        GROUP BY a.id
        ORDER BY a.numero_asiento DESC
        LIMIT 20
    ");
    $stmt->execute([$empresa_id]);
    $asientos_recientes = $stmt->fetchAll();
    
    // Estadísticas contables
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(DISTINCT a.id) as total_asientos,
            COUNT(m.id) as total_movimientos,
            SUM(CASE WHEN MONTH(a.fecha) = MONTH(CURDATE()) THEN 1 ELSE 0 END) as asientos_mes,
            COUNT(DISTINCT a.id) as asientos_contabilizados
        FROM asientos_contables a
        LEFT JOIN movimientos_contables m ON a.id = m.asiento_id
        WHERE a.empresa_id = ? AND a.estado = 'contabilizado'
    ");
    $stmt->execute([$empresa_id]);
    $estadisticas = $stmt->fetch();
    
    // Balance de comprobación
    if ($accion === 'balance') {
        $fecha_desde = $_GET['fecha_desde'] ?? date('Y-m-01');
        $fecha_hasta = $_GET['fecha_hasta'] ?? date('Y-m-t');
        
        $stmt = $pdo->prepare("
            SELECT 
                m.cuenta_contable,
                m.nombre_cuenta,
                pc.tipo_cuenta,
                SUM(m.debe) as total_debe,
                SUM(m.haber) as total_haber,
                (SUM(m.debe) - SUM(m.haber)) as saldo
            FROM movimientos_contables m
            JOIN asientos_contables a ON m.asiento_id = a.id
            LEFT JOIN plan_cuentas pc ON m.cuenta_contable = pc.codigo_cuenta AND (pc.empresa_id = ? OR pc.empresa_id IS NULL)
            WHERE a.empresa_id = ? 
            AND a.fecha BETWEEN ? AND ?
            AND a.estado = 'contabilizado'
            GROUP BY m.cuenta_contable, m.nombre_cuenta, pc.tipo_cuenta
            HAVING (SUM(m.debe) != 0 OR SUM(m.haber) != 0)
            ORDER BY m.cuenta_contable
        ");
        $stmt->execute([$empresa_id, $empresa_id, $fecha_desde, $fecha_hasta]);
        $balance_comprobacion = $stmt->fetchAll();
    }
    
} catch (PDOException $e) {
    die('Error de conexión: ' . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Contabilidad - SISCONECTA ERP</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .main-container {
            padding: 20px;
            max-width: 1600px;
            margin: 0 auto;
        }
        
        .header-section {
            background: linear-gradient(45deg, #2c3e50, #34495e);
            color: white;
            padding: 25px 30px;
            border-radius: 15px;
            margin-bottom: 25px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }
        
        .header-title {
            font-size: 1.9rem;
            font-weight: 700;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
        }
        
        .btn-header {
            background: rgba(255,255,255,0.15);
            border: 2px solid rgba(255,255,255,0.3);
            color: white;
            border-radius: 10px;
            padding: 12px 20px;
            text-decoration: none;
            transition: all 0.3s ease;
            font-weight: 600;
        }
        
        .btn-header:hover {
            background: rgba(255,255,255,0.25);
            color: white;
            transform: translateY(-2px);
        }
        
        .stats-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border-left: 5px solid;
            transition: all 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.15);
        }
        
        .stat-card.primary { border-left-color: #3498db; }
        .stat-card.success { border-left-color: #27ae60; }
        .stat-card.warning { border-left-color: #f39c12; }
        .stat-card.info { border-left-color: #9b59b6; }
        
        .stat-value {
            font-size: 2.2rem;
            font-weight: 800;
            color: #2c3e50;
            margin-bottom: 8px;
        }
        
        .stat-label {
            color: #666;
            font-weight: 600;
            font-size: 0.95rem;
        }
        
        .stat-icon {
            float: right;
            font-size: 2.5rem;
            opacity: 0.2;
        }
        
        .content-card {
            background: white;
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            margin-bottom: 25px;
        }
        
        .nav-pills .nav-link {
            border-radius: 10px;
            font-weight: 600;
            margin-right: 10px;
            padding: 12px 20px;
        }
        
        .nav-pills .nav-link.active {
            background: linear-gradient(135deg, #3498db, #2980b9);
        }
        
        .table {
            border-radius: 15px;
            overflow: hidden;
        }
        
        .table th {
            background: linear-gradient(135deg, #34495e, #2c3e50);
            color: white;
            border: none;
            font-weight: 700;
            padding: 15px 12px;
        }
        
        .table td {
            padding: 15px 12px;
            vertical-align: middle;
            border-top: 1px solid #f1f3f4;
        }
        
        .table tbody tr:hover {
            background: #f8f9fa;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 12px 15px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #3498db;
            box-shadow: 0 0 0 0.25rem rgba(52,152,219,0.25);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #3498db, #2980b9);
            border: none;
            border-radius: 10px;
            padding: 12px 25px;
            font-weight: 600;
        }
        
        .btn-success {
            background: linear-gradient(135deg, #27ae60, #229954);
            border: none;
            border-radius: 10px;
            padding: 12px 25px;
            font-weight: 600;
        }
        
        .btn-warning {
            background: linear-gradient(135deg, #f39c12, #e67e22);
            border: none;
            border-radius: 10px;
            padding: 12px 25px;
            font-weight: 600;
        }
        
        .badge {
            padding: 8px 12px;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .asiento-movimientos {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 10px;
        }
        
        .movimiento-row {
            background: white;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 10px;
            border: 1px solid #e9ecef;
        }
        
        .balance-totales {
            background: linear-gradient(135deg, #2c3e50, #34495e);
            color: white;
            padding: 15px;
            border-radius: 10px;
            margin-top: 20px;
        }
        
        .cuenta-nivel-1 { font-weight: 700; background: #e3f2fd; }
        .cuenta-nivel-2 { font-weight: 600; background: #f3e5f5; padding-left: 20px; }
        .cuenta-nivel-3 { font-weight: 500; background: #e8f5e8; padding-left: 40px; }
        .cuenta-nivel-4 { font-weight: 400; padding-left: 60px; }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header -->
        <div class="header-section">
            <h1 class="header-title">
                <i class="fas fa-calculator"></i>
                Módulo de Contabilidad
            </h1>
            <div class="header-actions">
                <a href="dashboard.php" class="btn-header">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <a href="reportes.php" class="btn-header">
                    <i class="fas fa-chart-bar"></i> Reportes
                </a>
            </div>
        </div>

        <!-- Mensajes -->
        <?php if (!empty($mensaje)): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show">
                <i class="fas fa-<?php echo $tipo_mensaje === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                <?php echo htmlspecialchars($mensaje); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if ($accion === 'dashboard'): ?>
            <!-- Estadísticas -->
            <div class="stats-cards">
                <div class="stat-card primary">
                    <div class="stat-value"><?php echo number_format($estadisticas['total_asientos'] ?? 0); ?></div>
                    <div class="stat-label">Total Asientos</div>
                    <i class="fas fa-file-alt stat-icon"></i>
                </div>
                
                <div class="stat-card success">
                    <div class="stat-value"><?php echo number_format($estadisticas['total_movimientos'] ?? 0); ?></div>
                    <div class="stat-label">Total Movimientos</div>
                    <i class="fas fa-exchange-alt stat-icon"></i>
                </div>
                
                <div class="stat-card warning">
                    <div class="stat-value"><?php echo number_format($estadisticas['asientos_mes'] ?? 0); ?></div>
                    <div class="stat-label">Asientos del Mes</div>
                    <i class="fas fa-calendar stat-icon"></i>
                </div>
                
                <div class="stat-card info">
                    <div class="stat-value"><?php echo count($plan_cuentas); ?></div>
                    <div class="stat-label">Cuentas Contables</div>
                    <i class="fas fa-list stat-icon"></i>
                </div>
            </div>

            <!-- Navegación de módulos -->
            <div class="content-card">
                <ul class="nav nav-pills mb-4">
                    <li class="nav-item">
                        <a class="nav-link active" href="?accion=dashboard">
                            <i class="fas fa-tachometer-alt"></i> Dashboard Contable
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="?accion=plan_cuentas">
                            <i class="fas fa-list"></i> Plan de Cuentas
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="?accion=asientos">
                            <i class="fas fa-edit"></i> Asientos Contables
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="?accion=balance">
                            <i class="fas fa-balance-scale"></i> Balance de Comprobación
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="?accion=libro_diario">
                            <i class="fas fa-book"></i> Libro Diario
                        </a>
                    </li>
                </ul>

                <!-- Asientos Recientes -->
                <h4><i class="fas fa-clock"></i> Últimos Asientos Contables</h4>
                
                <?php if (empty($asientos_recientes)): ?>
                    <div class="text-center text-muted py-5">
                        <i class="fas fa-calculator fa-3x mb-3"></i>
                        <h5>No hay asientos contables registrados</h5>
                        <p>Comience creando su primer asiento contable</p>
                        <a href="?accion=asientos" class="btn btn-primary">
                            <i class="fas fa-plus"></i> Crear Primer Asiento
                        </a>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>N° Asiento</th>
                                    <th>Fecha</th>
                                    <th>Descripción</th>
                                    <th>Debe</th>
                                    <th>Haber</th>
                                    <th>Movimientos</th>
                                    <th>Usuario</th>
                                    <th>Estado</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($asientos_recientes as $asiento): ?>
                                    <tr>
                                        <td><strong><?php echo $asiento['numero_asiento']; ?></strong></td>
                                        <td><?php echo date('d/m/Y', strtotime($asiento['fecha'])); ?></td>
                                        <td><?php echo htmlspecialchars($asiento['descripcion']); ?></td>
                                        <td class="text-success">$<?php echo number_format($asiento['total_debe'], 2); ?></td>
                                        <td class="text-danger">$<?php echo number_format($asiento['total_haber'], 2); ?></td>
                                        <td><span class="badge bg-info"><?php echo $asiento['total_movimientos']; ?></span></td>
                                        <td><?php echo htmlspecialchars($asiento['usuario_nombre'] ?? 'Sistema'); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo $asiento['estado'] === 'contabilizado' ? 'success' : 'warning'; ?>">
                                                <?php echo ucfirst($asiento['estado']); ?>
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>

        <?php elseif ($accion === 'plan_cuentas'): ?>
            <!-- Plan de Cuentas -->
            <div class="content-card">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3><i class="fas fa-list"></i> Plan de Cuentas</h3>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalNuevaCuenta">
                        <i class="fas fa-plus"></i> Nueva Cuenta
                    </button>
                </div>

                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Código</th>
                                <th>Nombre de la Cuenta</th>
                                <th>Tipo</th>
                                <th>Nivel</th>
                                <th>Cuenta Padre</th>
                                <th>Estado</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($plan_cuentas as $cuenta): ?>
                                <tr class="cuenta-nivel-<?php echo $cuenta['nivel']; ?>">
                                    <td><strong><?php echo htmlspecialchars($cuenta['codigo_cuenta']); ?></strong></td>
                                    <td><?php echo htmlspecialchars($cuenta['nombre_cuenta']); ?></td>
                                    <td>
                                        <span class="badge bg-<?php 
                                            switch($cuenta['tipo_cuenta']) {
                                                case 'activo': echo 'primary'; break;
                                                case 'pasivo': echo 'warning'; break;
                                                case 'patrimonio': echo 'info'; break;
                                                case 'ingreso': echo 'success'; break;
                                                case 'gasto': echo 'danger'; break;
                                                default: echo 'secondary'; break;
                                            }
                                        ?>">
                                            <?php echo ucfirst($cuenta['tipo_cuenta']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo $cuenta['nivel']; ?></td>
                                    <td><?php echo htmlspecialchars($cuenta['cuenta_padre'] ?? '-'); ?></td>
                                    <td>
                                        <span class="badge bg-<?php echo $cuenta['activo'] ? 'success' : 'secondary'; ?>">
                                            <?php echo $cuenta['activo'] ? 'Activa' : 'Inactiva'; ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

        <?php elseif ($accion === 'asientos'): ?>
            <!-- Crear Asiento Contable -->
            <div class="content-card">
                <h3><i class="fas fa-edit"></i> Crear Asiento Contable</h3>
                
                <form method="POST" id="formAsiento">
                    <input type="hidden" name="movimientos" id="movimientosData">
                    
                    <div class="row g-3 mb-4">
                        <div class="col-md-6">
                            <label class="form-label">Fecha del Asiento</label>
                            <input type="date" name="fecha" class="form-control" value="<?php echo date('Y-m-d'); ?>" required>
                        </div>
                        
                        <div class="col-md-6">
                            <label class="form-label">Descripción</label>
                            <input type="text" name="descripcion" class="form-control" placeholder="Descripción del asiento contable" required>
                        </div>
                    </div>
                    
                    <h5><i class="fas fa-exchange-alt"></i> Movimientos Contables</h5>
                    <div id="movimientosContainer">
                        <div class="movimiento-row">
                            <div class="row g-3">
                                <div class="col-md-3">
                                    <label class="form-label">Cuenta Contable</label>
                                    <select class="form-control cuenta-select">
                                        <option value="">Seleccionar cuenta...</option>
                                        <?php foreach ($plan_cuentas as $cuenta): ?>
                                            <option value="<?php echo $cuenta['codigo_cuenta']; ?>" data-nombre="<?php echo htmlspecialchars($cuenta['nombre_cuenta']); ?>">
                                                <?php echo $cuenta['codigo_cuenta']; ?> - <?php echo htmlspecialchars($cuenta['nombre_cuenta']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Debe</label>
                                    <input type="number" class="form-control debe-input" step="0.01" min="0" placeholder="0.00">
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Haber</label>
                                    <input type="number" class="form-control haber-input" step="0.01" min="0" placeholder="0.00">
                                </div>
                                <div class="col-md-4">
                                    <label class="form-label">Observaciones</label>
                                    <input type="text" class="form-control observaciones-input" placeholder="Observaciones del movimiento">
                                </div>
                                <div class="col-md-1">
                                    <label class="form-label">&nbsp;</label>
                                    <button type="button" class="btn btn-danger w-100 eliminar-movimiento">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="d-flex justify-content-between align-items-center mt-3">
                        <button type="button" class="btn btn-success" onclick="agregarMovimiento()">
                            <i class="fas fa-plus"></i> Agregar Movimiento
                        </button>
                        
                        <div class="d-flex gap-3">
                            <div class="text-success">
                                <strong>Total Debe: $<span id="totalDebe">0.00</span></strong>
                            </div>
                            <div class="text-danger">
                                <strong>Total Haber: $<span id="totalHaber">0.00</span></strong>
                            </div>
                            <div class="text-info">
                                <strong>Diferencia: $<span id="diferencia">0.00</span></strong>
                            </div>
                        </div>
                    </div>
                    
                    <div class="text-center mt-4">
                        <button type="submit" name="crear_asiento" class="btn btn-primary btn-lg" id="btnCrearAsiento" disabled>
                            <i class="fas fa-save"></i> Crear Asiento Contable
                        </button>
                    </div>
                </form>
            </div>

        <?php elseif ($accion === 'balance'): ?>
            <!-- Balance de Comprobación -->
            <div class="content-card">
                <h3><i class="fas fa-balance-scale"></i> Balance de Comprobación</h3>
                
                <form method="GET" class="row g-3 mb-4">
                    <input type="hidden" name="accion" value="balance">
                    <div class="col-md-4">
                        <label class="form-label">Fecha Desde</label>
                        <input type="date" name="fecha_desde" class="form-control" value="<?php echo $_GET['fecha_desde'] ?? date('Y-m-01'); ?>">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Fecha Hasta</label>
                        <input type="date" name="fecha_hasta" class="form-control" value="<?php echo $_GET['fecha_hasta'] ?? date('Y-m-t'); ?>">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">&nbsp;</label>
                        <button type="submit" class="btn btn-primary w-100">
                            <i class="fas fa-search"></i> Generar Balance
                        </button>
                    </div>
                </form>

                <?php if (isset($balance_comprobacion)): ?>
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Código</th>
                                    <th>Nombre de la Cuenta</th>
                                    <th>Tipo</th>
                                    <th class="text-end">Debe</th>
                                    <th class="text-end">Haber</th>
                                    <th class="text-end">Saldo</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                $total_debe = 0;
                                $total_haber = 0;
                                $total_saldo = 0;
                                
                                foreach ($balance_comprobacion as $cuenta): 
                                    $total_debe += $cuenta['total_debe'];
                                    $total_haber += $cuenta['total_haber'];
                                    $total_saldo += $cuenta['saldo'];
                                ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($cuenta['cuenta_contable']); ?></strong></td>
                                        <td><?php echo htmlspecialchars($cuenta['nombre_cuenta']); ?></td>
                                        <td>
                                            <span class="badge bg-<?php 
                                                switch($cuenta['tipo_cuenta']) {
                                                    case 'activo': echo 'primary'; break;
                                                    case 'pasivo': echo 'warning'; break;
                                                    case 'patrimonio': echo 'info'; break;
                                                    case 'ingreso': echo 'success'; break;
                                                    case 'gasto': echo 'danger'; break;
                                                    default: echo 'secondary'; break;
                                                }
                                            ?>">
                                                <?php echo ucfirst($cuenta['tipo_cuenta'] ?? 'N/A'); ?>
                                            </span>
                                        </td>
                                        <td class="text-end"><?php echo number_format($cuenta['total_debe'], 2); ?></td>
                                        <td class="text-end"><?php echo number_format($cuenta['total_haber'], 2); ?></td>
                                        <td class="text-end">
                                            <span class="<?php echo $cuenta['saldo'] >= 0 ? 'text-success' : 'text-danger'; ?>">
                                                <?php echo number_format($cuenta['saldo'], 2); ?>
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="balance-totales">
                        <div class="row text-center">
                            <div class="col-md-4">
                                <h5>Total Debe</h5>
                                <h3>$<?php echo number_format($total_debe, 2); ?></h3>
                            </div>
                            <div class="col-md-4">
                                <h5>Total Haber</h5>
                                <h3>$<?php echo number_format($total_haber, 2); ?></h3>
                            </div>
                            <div class="col-md-4">
                                <h5>Diferencia</h5>
                                <h3 class="<?php echo abs($total_debe - $total_haber) < 0.01 ? 'text-success' : 'text-danger'; ?>">
                                    $<?php echo number_format($total_debe - $total_haber, 2); ?>
                                </h3>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

        <?php else: ?>
            <div class="content-card">
                <div class="text-center">
                    <i class="fas fa-cog fa-3x text-muted mb-3"></i>
                    <h4>Módulo en desarrollo</h4>
                    <p>Esta funcionalidad estará disponible próximamente.</p>
                    <a href="?accion=dashboard" class="btn btn-primary">Volver al Dashboard</a>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- Modal Nueva Cuenta -->
    <div class="modal fade" id="modalNuevaCuenta" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Nueva Cuenta Contable</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Código de la Cuenta</label>
                            <input type="text" name="codigo_cuenta" class="form-control" placeholder="Ej: 1101" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Nombre de la Cuenta</label>
                            <input type="text" name="nombre_cuenta" class="form-control" placeholder="Ej: CAJA" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Tipo de Cuenta</label>
                            <select name="tipo_cuenta" class="form-control" required>
                                <option value="">Seleccionar tipo...</option>
                                <option value="activo">Activo</option>
                                <option value="pasivo">Pasivo</option>
                                <option value="patrimonio">Patrimonio</option>
                                <option value="ingreso">Ingreso</option>
                                <option value="gasto">Gasto</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Cuenta Padre (opcional)</label>
                            <select name="cuenta_padre" class="form-control">
                                <option value="">Sin cuenta padre</option>
                                <?php foreach ($plan_cuentas as $cuenta): ?>
                                    <option value="<?php echo $cuenta['codigo_cuenta']; ?>">
                                        <?php echo $cuenta['codigo_cuenta']; ?> - <?php echo htmlspecialchars($cuenta['nombre_cuenta']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Nivel</label>
                            <select name="nivel" class="form-control" required>
                                <option value="1">1 - Clase</option>
                                <option value="2">2 - Grupo</option>
                                <option value="3">3 - Rubro</option>
                                <option value="4">4 - Cuenta</option>
                                <option value="5">5 - Subcuenta</option>
                            </select>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" name="agregar_cuenta" class="btn btn-primary">Agregar Cuenta</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Gestión de movimientos contables
        function agregarMovimiento() {
            const container = document.getElementById('movimientosContainer');
            const nuevoMovimiento = container.children[0].cloneNode(true);
            
            // Limpiar valores
            nuevoMovimiento.querySelectorAll('input, select').forEach(input => {
                if (input.type !== 'button') {
                    input.value = '';
                }
            });
            
            container.appendChild(nuevoMovimiento);
            configurarEventosMovimiento();
        }
        
        function configurarEventosMovimiento() {
            // Eliminar movimiento
            document.querySelectorAll('.eliminar-movimiento').forEach(btn => {
                btn.onclick = function() {
                    if (document.querySelectorAll('.movimiento-row').length > 1) {
                        this.closest('.movimiento-row').remove();
                        calcularTotales();
                    }
                };
            });
            
            // Calcular totales cuando cambian los valores
            document.querySelectorAll('.debe-input, .haber-input').forEach(input => {
                input.addEventListener('input', calcularTotales);
            });
            
            // Cuando se selecciona una cuenta, limpiar debe/haber del lado opuesto
            document.querySelectorAll('.debe-input').forEach(input => {
                input.addEventListener('input', function() {
                    if (parseFloat(this.value) > 0) {
                        const haberInput = this.closest('.movimiento-row').querySelector('.haber-input');
                        haberInput.value = '';
                    }
                });
            });
            
            document.querySelectorAll('.haber-input').forEach(input => {
                input.addEventListener('input', function() {
                    if (parseFloat(this.value) > 0) {
                        const debeInput = this.closest('.movimiento-row').querySelector('.debe-input');
                        debeInput.value = '';
                    }
                });
            });
        }
        
        function calcularTotales() {
            let totalDebe = 0;
            let totalHaber = 0;
            
            document.querySelectorAll('.debe-input').forEach(input => {
                totalDebe += parseFloat(input.value) || 0;
            });
            
            document.querySelectorAll('.haber-input').forEach(input => {
                totalHaber += parseFloat(input.value) || 0;
            });
            
            const diferencia = totalDebe - totalHaber;
            
            document.getElementById('totalDebe').textContent = totalDebe.toFixed(2);
            document.getElementById('totalHaber').textContent = totalHaber.toFixed(2);
            document.getElementById('diferencia').textContent = Math.abs(diferencia).toFixed(2);
            
            // Habilitar/deshabilitar botón según balance
            const btnCrear = document.getElementById('btnCrearAsiento');
            if (Math.abs(diferencia) < 0.01 && totalDebe > 0 && totalHaber > 0) {
                btnCrear.disabled = false;
                btnCrear.className = 'btn btn-primary btn-lg';
            } else {
                btnCrear.disabled = true;
                btnCrear.className = 'btn btn-secondary btn-lg';
            }
        }
        
        // Al enviar el formulario, recopilar movimientos
        document.getElementById('formAsiento')?.addEventListener('submit', function(e) {
            const movimientos = [];
            
            document.querySelectorAll('.movimiento-row').forEach(row => {
                const cuenta = row.querySelector('.cuenta-select').value;
                const nombreCuenta = row.querySelector('.cuenta-select option:checked')?.dataset?.nombre || '';
                const debe = parseFloat(row.querySelector('.debe-input').value) || 0;
                const haber = parseFloat(row.querySelector('.haber-input').value) || 0;
                const observaciones = row.querySelector('.observaciones-input').value;
                
                if (cuenta && (debe > 0 || haber > 0)) {
                    movimientos.push({
                        cuenta: cuenta,
                        nombre_cuenta: nombreCuenta,
                        debe: debe,
                        haber: haber,
                        observaciones: observaciones
                    });
                }
            });
            
            if (movimientos.length < 2) {
                e.preventDefault();
                alert('Debe agregar al menos 2 movimientos válidos');
                return false;
            }
            
            document.getElementById('movimientosData').value = JSON.stringify(movimientos);
        });
        
        // Inicializar eventos
        document.addEventListener('DOMContentLoaded', function() {
            configurarEventosMovimiento();
        });
    </script>
</body>
</html>