<?php
// MÓDULO CUENTAS POR PAGAR - SISCONECTA ERP
// cuentas_por_pagar.php

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Verificar sesión activa
if (!isset($_SESSION['user_id']) || !isset($_SESSION['empresa_id'])) {
    header('Location: login.php');
    exit;
}

// Configuración de base de datos
$host = 'localhost';
$dbname = 'conectae_sisconecta';
$username = 'conectae_sisconectauser';
$password = 'pt125824caraud';

$mensaje = '';
$tipo_mensaje = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $empresa_id = $_SESSION['empresa_id'];
    
    // Verificar estado de empresa
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ? AND estado = 'activo'");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch();
    
    if (!$empresa) {
        session_destroy();
        header('Location: login.php?error=empresa_inactiva');
        exit;
    }
    
    // Crear tablas si no existen
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS proveedores (
            id INT AUTO_INCREMENT PRIMARY KEY,
            empresa_id INT NOT NULL,
            rut VARCHAR(20) NOT NULL,
            razon_social VARCHAR(255) NOT NULL,
            nombre_contacto VARCHAR(255),
            direccion TEXT,
            telefono VARCHAR(50),
            email VARCHAR(100),
            giro VARCHAR(255),
            condiciones_pago VARCHAR(100) DEFAULT 'contado',
            dias_credito INT DEFAULT 0,
            limite_credito DECIMAL(15,2) DEFAULT 0.00,
            activo BOOLEAN DEFAULT TRUE,
            fecha_registro TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (empresa_id) REFERENCES empresas(id) ON DELETE CASCADE,
            UNIQUE KEY unique_rut_empresa (rut, empresa_id),
            INDEX idx_razon_social (razon_social),
            INDEX idx_activo (activo)
        ) ENGINE=InnoDB
    ");
    
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS cuentas_por_pagar (
            id INT AUTO_INCREMENT PRIMARY KEY,
            empresa_id INT NOT NULL,
            proveedor_id INT NOT NULL,
            numero_factura VARCHAR(50) NOT NULL,
            fecha_factura DATE NOT NULL,
            fecha_vencimiento DATE NOT NULL,
            descripcion TEXT,
            monto_original DECIMAL(15,2) NOT NULL,
            monto_pendiente DECIMAL(15,2) NOT NULL,
            moneda VARCHAR(3) DEFAULT 'CLP',
            estado ENUM('pendiente', 'pagado_parcial', 'pagado', 'vencido', 'anulado') DEFAULT 'pendiente',
            categoria VARCHAR(100),
            centro_costo VARCHAR(100),
            observaciones TEXT,
            archivo_factura TEXT,
            usuario_registro INT,
            fecha_registro TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (empresa_id) REFERENCES empresas(id) ON DELETE CASCADE,
            FOREIGN KEY (proveedor_id) REFERENCES proveedores(id) ON DELETE CASCADE,
            FOREIGN KEY (usuario_registro) REFERENCES usuarios(id),
            INDEX idx_empresa_estado (empresa_id, estado),
            INDEX idx_fecha_vencimiento (fecha_vencimiento),
            INDEX idx_proveedor (proveedor_id)
        ) ENGINE=InnoDB
    ");
    
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS pagos_proveedores (
            id INT AUTO_INCREMENT PRIMARY KEY,
            empresa_id INT NOT NULL,
            cuenta_por_pagar_id INT NOT NULL,
            numero_pago VARCHAR(50) NOT NULL,
            fecha_pago DATE NOT NULL,
            monto_pago DECIMAL(15,2) NOT NULL,
            metodo_pago ENUM('efectivo', 'transferencia', 'cheque', 'debito', 'credito') NOT NULL,
            numero_comprobante VARCHAR(100),
            banco VARCHAR(100),
            observaciones TEXT,
            usuario_registro INT,
            fecha_registro TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (empresa_id) REFERENCES empresas(id) ON DELETE CASCADE,
            FOREIGN KEY (cuenta_por_pagar_id) REFERENCES cuentas_por_pagar(id) ON DELETE CASCADE,
            FOREIGN KEY (usuario_registro) REFERENCES usuarios(id),
            INDEX idx_fecha_pago (fecha_pago),
            INDEX idx_cuenta_pagar (cuenta_por_pagar_id)
        ) ENGINE=InnoDB
    ");
    
    $accion = $_GET['accion'] ?? 'dashboard';
    
    // Crear nuevo proveedor
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['crear_proveedor'])) {
        $rut = trim($_POST['rut']);
        $razon_social = trim($_POST['razon_social']);
        $nombre_contacto = trim($_POST['nombre_contacto']);
        $direccion = trim($_POST['direccion']);
        $telefono = trim($_POST['telefono']);
        $email = trim($_POST['email']);
        $giro = trim($_POST['giro']);
        $condiciones_pago = $_POST['condiciones_pago'];
        $dias_credito = intval($_POST['dias_credito']);
        $limite_credito = floatval($_POST['limite_credito']);
        
        if (!empty($rut) && !empty($razon_social)) {
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO proveedores 
                    (empresa_id, rut, razon_social, nombre_contacto, direccion, telefono, email, giro, 
                     condiciones_pago, dias_credito, limite_credito) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $empresa_id, $rut, $razon_social, $nombre_contacto, $direccion, 
                    $telefono, $email, $giro, $condiciones_pago, $dias_credito, $limite_credito
                ]);
                
                $mensaje = 'Proveedor creado exitosamente';
                $tipo_mensaje = 'success';
                
            } catch (PDOException $e) {
                $mensaje = 'Error al crear proveedor: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'RUT y razón social son obligatorios';
            $tipo_mensaje = 'warning';
        }
    }
    
    // Crear cuenta por pagar
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['crear_cuenta_pagar'])) {
        $proveedor_id = intval($_POST['proveedor_id']);
        $numero_factura = trim($_POST['numero_factura']);
        $fecha_factura = $_POST['fecha_factura'];
        $fecha_vencimiento = $_POST['fecha_vencimiento'];
        $descripcion = trim($_POST['descripcion']);
        $monto_original = floatval($_POST['monto_original']);
        $moneda = $_POST['moneda'];
        $categoria = trim($_POST['categoria']);
        $centro_costo = trim($_POST['centro_costo']);
        $observaciones = trim($_POST['observaciones']);
        
        if ($proveedor_id > 0 && !empty($numero_factura) && !empty($fecha_factura) && $monto_original > 0) {
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO cuentas_por_pagar 
                    (empresa_id, proveedor_id, numero_factura, fecha_factura, fecha_vencimiento, 
                     descripcion, monto_original, monto_pendiente, moneda, categoria, centro_costo, 
                     observaciones, usuario_registro) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $empresa_id, $proveedor_id, $numero_factura, $fecha_factura, $fecha_vencimiento,
                    $descripcion, $monto_original, $monto_original, $moneda, $categoria, $centro_costo,
                    $observaciones, $_SESSION['user_id']
                ]);
                
                $mensaje = 'Cuenta por pagar registrada exitosamente';
                $tipo_mensaje = 'success';
                
            } catch (PDOException $e) {
                $mensaje = 'Error al registrar cuenta por pagar: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Complete todos los campos obligatorios';
            $tipo_mensaje = 'warning';
        }
    }
    
    // Registrar pago
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['registrar_pago'])) {
        $cuenta_por_pagar_id = intval($_POST['cuenta_por_pagar_id']);
        $fecha_pago = $_POST['fecha_pago'];
        $monto_pago = floatval($_POST['monto_pago']);
        $metodo_pago = $_POST['metodo_pago'];
        $numero_comprobante = trim($_POST['numero_comprobante']);
        $banco = trim($_POST['banco']);
        $observaciones_pago = trim($_POST['observaciones_pago']);
        
        if ($cuenta_por_pagar_id > 0 && !empty($fecha_pago) && $monto_pago > 0) {
            try {
                $pdo->beginTransaction();
                
                // Verificar cuenta por pagar
                $stmt = $pdo->prepare("SELECT * FROM cuentas_por_pagar WHERE id = ? AND empresa_id = ?");
                $stmt->execute([$cuenta_por_pagar_id, $empresa_id]);
                $cuenta = $stmt->fetch();
                
                if (!$cuenta) {
                    throw new Exception('Cuenta por pagar no encontrada');
                }
                
                if ($monto_pago > $cuenta['monto_pendiente']) {
                    throw new Exception('El monto del pago no puede ser mayor al monto pendiente');
                }
                
                // Generar número de pago
                $stmt = $pdo->prepare("SELECT COALESCE(MAX(CAST(SUBSTRING(numero_pago, 2) AS UNSIGNED)), 0) + 1 as siguiente FROM pagos_proveedores WHERE empresa_id = ?");
                $stmt->execute([$empresa_id]);
                $numero_pago = 'P' . str_pad($stmt->fetch()['siguiente'], 6, '0', STR_PAD_LEFT);
                
                // Registrar pago
                $stmt = $pdo->prepare("
                    INSERT INTO pagos_proveedores 
                    (empresa_id, cuenta_por_pagar_id, numero_pago, fecha_pago, monto_pago, 
                     metodo_pago, numero_comprobante, banco, observaciones, usuario_registro) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $empresa_id, $cuenta_por_pagar_id, $numero_pago, $fecha_pago, $monto_pago,
                    $metodo_pago, $numero_comprobante, $banco, $observaciones_pago, $_SESSION['user_id']
                ]);
                
                // Actualizar monto pendiente
                $nuevo_monto_pendiente = $cuenta['monto_pendiente'] - $monto_pago;
                $nuevo_estado = $nuevo_monto_pendiente <= 0 ? 'pagado' : 'pagado_parcial';
                
                $stmt = $pdo->prepare("
                    UPDATE cuentas_por_pagar 
                    SET monto_pendiente = ?, estado = ? 
                    WHERE id = ?
                ");
                $stmt->execute([$nuevo_monto_pendiente, $nuevo_estado, $cuenta_por_pagar_id]);
                
                $pdo->commit();
                
                $mensaje = "Pago registrado exitosamente (N° $numero_pago)";
                $tipo_mensaje = 'success';
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $mensaje = 'Error al registrar pago: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Complete todos los campos obligatorios del pago';
            $tipo_mensaje = 'warning';
        }
    }
    
    // Obtener proveedores
    $stmt = $pdo->prepare("SELECT * FROM proveedores WHERE empresa_id = ? AND activo = 1 ORDER BY razon_social ASC");
    $stmt->execute([$empresa_id]);
    $proveedores = $stmt->fetchAll();
    
    // Obtener cuentas por pagar con filtros
    $filtro_estado = $_GET['estado'] ?? '';
    $filtro_proveedor = $_GET['proveedor'] ?? '';
    $filtro_vencimiento = $_GET['vencimiento'] ?? '';
    
    $where_conditions = ["cpp.empresa_id = ?"];
    $params = [$empresa_id];
    
    if (!empty($filtro_estado)) {
        $where_conditions[] = "cpp.estado = ?";
        $params[] = $filtro_estado;
    }
    
    if (!empty($filtro_proveedor)) {
        $where_conditions[] = "cpp.proveedor_id = ?";
        $params[] = $filtro_proveedor;
    }
    
    if ($filtro_vencimiento === 'vencidas') {
        $where_conditions[] = "cpp.fecha_vencimiento < CURDATE() AND cpp.estado IN ('pendiente', 'pagado_parcial')";
    } elseif ($filtro_vencimiento === 'por_vencer') {
        $where_conditions[] = "cpp.fecha_vencimiento <= DATE_ADD(CURDATE(), INTERVAL 7 DAY) AND cpp.estado IN ('pendiente', 'pagado_parcial')";
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    $stmt = $pdo->prepare("
        SELECT cpp.*, p.razon_social as proveedor_nombre, p.rut as proveedor_rut,
               COALESCE(SUM(pp.monto_pago), 0) as total_pagado,
               COUNT(pp.id) as num_pagos,
               CASE 
                   WHEN cpp.fecha_vencimiento < CURDATE() AND cpp.estado IN ('pendiente', 'pagado_parcial') THEN 'vencida'
                   WHEN cpp.fecha_vencimiento <= DATE_ADD(CURDATE(), INTERVAL 7 DAY) AND cpp.estado IN ('pendiente', 'pagado_parcial') THEN 'por_vencer'
                   ELSE cpp.estado
               END as estado_real
        FROM cuentas_por_pagar cpp
        JOIN proveedores p ON cpp.proveedor_id = p.id
        LEFT JOIN pagos_proveedores pp ON cpp.id = pp.cuenta_por_pagar_id
        WHERE $where_clause
        GROUP BY cpp.id
        ORDER BY cpp.fecha_vencimiento ASC, cpp.fecha_registro DESC
        LIMIT 100
    ");
    $stmt->execute($params);
    $cuentas_por_pagar = $stmt->fetchAll();
    
    // Estadísticas
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_cuentas,
            SUM(CASE WHEN estado IN ('pendiente', 'pagado_parcial') THEN monto_pendiente ELSE 0 END) as total_pendiente,
            SUM(CASE WHEN fecha_vencimiento < CURDATE() AND estado IN ('pendiente', 'pagado_parcial') THEN monto_pendiente ELSE 0 END) as total_vencido,
            COUNT(CASE WHEN fecha_vencimiento < CURDATE() AND estado IN ('pendiente', 'pagado_parcial') THEN 1 END) as cuentas_vencidas,
            COUNT(CASE WHEN fecha_vencimiento <= DATE_ADD(CURDATE(), INTERVAL 7 DAY) AND estado IN ('pendiente', 'pagado_parcial') THEN 1 END) as cuentas_por_vencer
        FROM cuentas_por_pagar 
        WHERE empresa_id = ?
    ");
    $stmt->execute([$empresa_id]);
    $estadisticas = $stmt->fetch();
    
} catch (PDOException $e) {
    die('Error de conexión: ' . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cuentas por Pagar - SISCONECTA ERP</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .main-container {
            padding: 20px;
            max-width: 1600px;
            margin: 0 auto;
        }
        
        .header-section {
            background: linear-gradient(45deg, #2c3e50, #34495e);
            color: white;
            padding: 25px 30px;
            border-radius: 15px;
            margin-bottom: 25px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }
        
        .header-title {
            font-size: 1.9rem;
            font-weight: 700;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
        }
        
        .btn-header {
            background: rgba(255,255,255,0.15);
            border: 2px solid rgba(255,255,255,0.3);
            color: white;
            border-radius: 10px;
            padding: 12px 20px;
            text-decoration: none;
            transition: all 0.3s ease;
            font-weight: 600;
        }
        
        .btn-header:hover {
            background: rgba(255,255,255,0.25);
            color: white;
            transform: translateY(-2px);
        }
        
        .stats-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border-left: 5px solid;
            transition: all 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.15);
        }
        
        .stat-card.primary { border-left-color: #3498db; }
        .stat-card.danger { border-left-color: #e74c3c; }
        .stat-card.warning { border-left-color: #f39c12; }
        .stat-card.success { border-left-color: #27ae60; }
        
        .stat-value {
            font-size: 2.2rem;
            font-weight: 800;
            color: #2c3e50;
            margin-bottom: 8px;
        }
        
        .stat-label {
            color: #666;
            font-weight: 600;
            font-size: 0.95rem;
        }
        
        .stat-icon {
            float: right;
            font-size: 2.5rem;
            opacity: 0.2;
        }
        
        .content-card {
            background: white;
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            margin-bottom: 25px;
        }
        
        .nav-pills .nav-link {
            border-radius: 10px;
            font-weight: 600;
            margin-right: 10px;
            padding: 12px 20px;
        }
        
        .nav-pills .nav-link.active {
            background: linear-gradient(135deg, #3498db, #2980b9);
        }
        
        .table {
            border-radius: 15px;
            overflow: hidden;
        }
        
        .table th {
            background: linear-gradient(135deg, #34495e, #2c3e50);
            color: white;
            border: none;
            font-weight: 700;
            padding: 15px 12px;
        }
        
        .table td {
            padding: 15px 12px;
            vertical-align: middle;
            border-top: 1px solid #f1f3f4;
        }
        
        .table tbody tr:hover {
            background: #f8f9fa;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 12px 15px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #3498db;
            box-shadow: 0 0 0 0.25rem rgba(52,152,219,0.25);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #3498db, #2980b9);
            border: none;
            border-radius: 10px;
            padding: 12px 25px;
            font-weight: 600;
        }
        
        .badge {
            padding: 8px 12px;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .filters-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 15px;
            margin-bottom: 25px;
        }
        
        .proveedor-card {
            background: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
        }
        
        .proveedor-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.15);
        }
        
        .cuenta-vencida {
            background: #fee;
            border-left: 4px solid #e74c3c;
        }
        
        .cuenta-por-vencer {
            background: #fff3cd;
            border-left: 4px solid #f39c12;
        }
        
        .cuenta-pagada {
            background: #d4edda;
            border-left: 4px solid #27ae60;
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header -->
        <div class="header-section">
            <h1 class="header-title">
                <i class="fas fa-money-bill-wave"></i>
                Cuentas por Pagar
            </h1>
            <div class="header-actions">
                <a href="dashboard.php" class="btn-header">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <a href="contabilidad.php" class="btn-header">
                    <i class="fas fa-calculator"></i> Contabilidad
                </a>
            </div>
        </div>

        <!-- Mensajes -->
        <?php if (!empty($mensaje)): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show">
                <i class="fas fa-<?php echo $tipo_mensaje === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                <?php echo htmlspecialchars($mensaje); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Estadísticas -->
        <div class="stats-cards">
            <div class="stat-card primary">
                <div class="stat-value"><?php echo number_format($estadisticas['total_cuentas'] ?? 0); ?></div>
                <div class="stat-label">Total Cuentas</div>
                <i class="fas fa-file-invoice stat-icon"></i>
            </div>
            
            <div class="stat-card danger">
                <div class="stat-value">$<?php echo number_format($estadisticas['total_pendiente'] ?? 0, 0); ?></div>
                <div class="stat-label">Total Pendiente</div>
                <i class="fas fa-exclamation-triangle stat-icon"></i>
            </div>
            
            <div class="stat-card warning">
                <div class="stat-value">$<?php echo number_format($estadisticas['total_vencido'] ?? 0, 0); ?></div>
                <div class="stat-label">Total Vencido</div>
                <i class="fas fa-clock stat-icon"></i>
            </div>
            
            <div class="stat-card success">
                <div class="stat-value"><?php echo number_format($estadisticas['cuentas_por_vencer'] ?? 0); ?></div>
                <div class="stat-label">Por Vencer (7 días)</div>
                <i class="fas fa-calendar-alt stat-icon"></i>
            </div>
        </div>

        <!-- Navegación -->
        <div class="content-card">
            <ul class="nav nav-pills mb-4">
                <li class="nav-item">
                    <a class="nav-link <?php echo $accion === 'dashboard' ? 'active' : ''; ?>" href="?accion=dashboard">
                        <i class="fas fa-tachometer-alt"></i> Dashboard
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?php echo $accion === 'cuentas' ? 'active' : ''; ?>" href="?accion=cuentas">
                        <i class="fas fa-list"></i> Cuentas por Pagar
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?php echo $accion === 'proveedores' ? 'active' : ''; ?>" href="?accion=proveedores">
                        <i class="fas fa-users"></i> Proveedores
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?php echo $accion === 'pagos' ? 'active' : ''; ?>" href="?accion=pagos">
                        <i class="fas fa-credit-card"></i> Registrar Pago
                    </a>
                </li>
            </ul>

            <?php if ($accion === 'dashboard'): ?>
                <!-- Dashboard de Cuentas por Pagar -->
                <h4><i class="fas fa-chart-pie"></i> Resumen de Cuentas por Pagar</h4>
                
                <!-- Filtros rápidos -->
                <div class="filters-section">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <a href="?estado=pendiente" class="btn btn-outline-warning w-100">
                                <i class="fas fa-clock"></i> Pendientes
                            </a>
                        </div>
                        <div class="col-md-3">
                            <a href="?vencimiento=vencidas" class="btn btn-outline-danger w-100">
                                <i class="fas fa-exclamation-triangle"></i> Vencidas
                            </a>
                        </div>
                        <div class="col-md-3">
                            <a href="?vencimiento=por_vencer" class="btn btn-outline-warning w-100">
                                <i class="fas fa-calendar-alt"></i> Por Vencer
                            </a>
                        </div>
                        <div class="col-md-3">
                            <a href="?estado=pagado" class="btn btn-outline-success w-100">
                                <i class="fas fa-check"></i> Pagadas
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Últimas cuentas por pagar -->
                <h5><i class="fas fa-clock"></i> Últimas Cuentas Registradas</h5>
                <?php if (empty($cuentas_por_pagar)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-money-bill-wave fa-3x text-muted mb-3"></i>
                        <h5>No hay cuentas por pagar registradas</h5>
                        <p>Comience registrando su primera cuenta por pagar</p>
                        <a href="?accion=cuentas" class="btn btn-primary">
                            <i class="fas fa-plus"></i> Registrar Primera Cuenta
                        </a>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Proveedor</th>
                                    <th>N° Factura</th>
                                    <th>Fecha Venc.</th>
                                    <th>Monto Original</th>
                                    <th>Pendiente</th>
                                    <th>Estado</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach (array_slice($cuentas_por_pagar, 0, 10) as $cuenta): ?>
                                    <tr class="<?php 
                                        if ($cuenta['estado_real'] === 'vencida') echo 'cuenta-vencida';
                                        elseif ($cuenta['estado_real'] === 'por_vencer') echo 'cuenta-por-vencer';
                                        elseif ($cuenta['estado'] === 'pagado') echo 'cuenta-pagada';
                                    ?>">
                                        <td>
                                            <strong><?php echo htmlspecialchars($cuenta['proveedor_nombre']); ?></strong>
                                            <br><small class="text-muted"><?php echo htmlspecialchars($cuenta['proveedor_rut']); ?></small>
                                        </td>
                                        <td><?php echo htmlspecialchars($cuenta['numero_factura']); ?></td>
                                        <td>
                                            <?php echo date('d/m/Y', strtotime($cuenta['fecha_vencimiento'])); ?>
                                            <?php if ($cuenta['estado_real'] === 'vencida'): ?>
                                                <br><small class="text-danger">Vencida</small>
                                            <?php elseif ($cuenta['estado_real'] === 'por_vencer'): ?>
                                                <br><small class="text-warning">Por vencer</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>$<?php echo number_format($cuenta['monto_original'], 0); ?></td>
                                        <td>
                                            <strong class="<?php echo $cuenta['monto_pendiente'] > 0 ? 'text-danger' : 'text-success'; ?>">
                                                $<?php echo number_format($cuenta['monto_pendiente'], 0); ?>
                                            </strong>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php 
                                                switch($cuenta['estado']) {
                                                    case 'pendiente': echo 'warning'; break;
                                                    case 'pagado_parcial': echo 'info'; break;
                                                    case 'pagado': echo 'success'; break;
                                                    case 'vencido': echo 'danger'; break;
                                                    default: echo 'secondary'; break;
                                                }
                                            ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $cuenta['estado'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($cuenta['estado'] !== 'pagado'): ?>
                                                <button class="btn btn-sm btn-success" onclick="registrarPago(<?php echo $cuenta['id']; ?>, '<?php echo htmlspecialchars($cuenta['proveedor_nombre']); ?>', <?php echo $cuenta['monto_pendiente']; ?>)">
                                                    <i class="fas fa-credit-card"></i> Pagar
                                                </button>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>

            <?php elseif ($accion === 'cuentas'): ?>
                <!-- Gestión de Cuentas por Pagar -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h4><i class="fas fa-list"></i> Gestión de Cuentas por Pagar</h4>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalNuevaCuenta">
                        <i class="fas fa-plus"></i> Nueva Cuenta por Pagar
                    </button>
                </div>

                <!-- Filtros -->
                <div class="filters-section">
                    <form method="GET" class="row g-3">
                        <input type="hidden" name="accion" value="cuentas">
                        <div class="col-md-3">
                            <select name="estado" class="form-control">
                                <option value="">Todos los estados</option>
                                <option value="pendiente" <?php echo $filtro_estado === 'pendiente' ? 'selected' : ''; ?>>Pendiente</option>
                                <option value="pagado_parcial" <?php echo $filtro_estado === 'pagado_parcial' ? 'selected' : ''; ?>>Pagado Parcial</option>
                                <option value="pagado" <?php echo $filtro_estado === 'pagado' ? 'selected' : ''; ?>>Pagado</option>
                                <option value="vencido" <?php echo $filtro_estado === 'vencido' ? 'selected' : ''; ?>>Vencido</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <select name="proveedor" class="form-control">
                                <option value="">Todos los proveedores</option>
                                <?php foreach ($proveedores as $proveedor): ?>
                                    <option value="<?php echo $proveedor['id']; ?>" <?php echo $filtro_proveedor == $proveedor['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($proveedor['razon_social']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <select name="vencimiento" class="form-control">
                                <option value="">Todos los vencimientos</option>
                                <option value="vencidas" <?php echo $filtro_vencimiento === 'vencidas' ? 'selected' : ''; ?>>Vencidas</option>
                                <option value="por_vencer" <?php echo $filtro_vencimiento === 'por_vencer' ? 'selected' : ''; ?>>Por vencer (7 días)</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <button type="submit" class="btn btn-primary w-100">
                                <i class="fas fa-filter"></i> Filtrar
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Lista completa de cuentas -->
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Proveedor</th>
                                <th>N° Factura</th>
                                <th>Fecha Factura</th>
                                <th>Fecha Vencimiento</th>
                                <th>Descripción</th>
                                <th>Monto Original</th>
                                <th>Monto Pendiente</th>
                                <th>Pagos</th>
                                <th>Estado</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($cuentas_por_pagar as $cuenta): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo htmlspecialchars($cuenta['proveedor_nombre']); ?></strong>
                                        <br><small class="text-muted"><?php echo htmlspecialchars($cuenta['proveedor_rut']); ?></small>
                                    </td>
                                    <td><?php echo htmlspecialchars($cuenta['numero_factura']); ?></td>
                                    <td><?php echo date('d/m/Y', strtotime($cuenta['fecha_factura'])); ?></td>
                                    <td>
                                        <?php echo date('d/m/Y', strtotime($cuenta['fecha_vencimiento'])); ?>
                                        <?php 
                                        $dias_vencimiento = (new DateTime())->diff(new DateTime($cuenta['fecha_vencimiento']))->days;
                                        $vencida = new DateTime() > new DateTime($cuenta['fecha_vencimiento']);
                                        ?>
                                        <?php if ($vencida && $cuenta['estado'] !== 'pagado'): ?>
                                            <br><small class="text-danger">Vencida hace <?php echo $dias_vencimiento; ?> días</small>
                                        <?php elseif (!$vencida && $dias_vencimiento <= 7 && $cuenta['estado'] !== 'pagado'): ?>
                                            <br><small class="text-warning">Vence en <?php echo $dias_vencimiento; ?> días</small>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($cuenta['descripcion']); ?></td>
                                    <td>$<?php echo number_format($cuenta['monto_original'], 0); ?></td>
                                    <td>
                                        <strong class="<?php echo $cuenta['monto_pendiente'] > 0 ? 'text-danger' : 'text-success'; ?>">
                                            $<?php echo number_format($cuenta['monto_pendiente'], 0); ?>
                                        </strong>
                                    </td>
                                    <td>
                                        <span class="badge bg-info"><?php echo $cuenta['num_pagos']; ?></span>
                                        <?php if ($cuenta['total_pagado'] > 0): ?>
                                            <br><small class="text-success">$<?php echo number_format($cuenta['total_pagado'], 0); ?></small>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?php 
                                            switch($cuenta['estado']) {
                                                case 'pendiente': echo 'warning'; break;
                                                case 'pagado_parcial': echo 'info'; break;
                                                case 'pagado': echo 'success'; break;
                                                case 'vencido': echo 'danger'; break;
                                                default: echo 'secondary'; break;
                                            }
                                        ?>">
                                            <?php echo ucfirst(str_replace('_', ' ', $cuenta['estado'])); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($cuenta['estado'] !== 'pagado'): ?>
                                            <button class="btn btn-sm btn-success" onclick="registrarPago(<?php echo $cuenta['id']; ?>, '<?php echo htmlspecialchars($cuenta['proveedor_nombre']); ?>', <?php echo $cuenta['monto_pendiente']; ?>)">
                                                <i class="fas fa-credit-card"></i>
                                            </button>
                                        <?php endif; ?>
                                        <button class="btn btn-sm btn-info" onclick="verDetalles(<?php echo $cuenta['id']; ?>)">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

            <?php elseif ($accion === 'proveedores'): ?>
                <!-- Gestión de Proveedores -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h4><i class="fas fa-users"></i> Gestión de Proveedores</h4>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalNuevoProveedor">
                        <i class="fas fa-user-plus"></i> Nuevo Proveedor
                    </button>
                </div>

                <div class="row">
                    <?php foreach ($proveedores as $proveedor): ?>
                        <div class="col-md-6 col-lg-4">
                            <div class="proveedor-card">
                                <h6><?php echo htmlspecialchars($proveedor['razon_social']); ?></h6>
                                <p class="text-muted mb-2">RUT: <?php echo htmlspecialchars($proveedor['rut']); ?></p>
                                <p class="mb-2">
                                    <i class="fas fa-phone"></i> <?php echo htmlspecialchars($proveedor['telefono'] ?: 'No especificado'); ?>
                                    <br><i class="fas fa-envelope"></i> <?php echo htmlspecialchars($proveedor['email'] ?: 'No especificado'); ?>
                                </p>
                                <p class="mb-3">
                                    <span class="badge bg-info"><?php echo ucfirst($proveedor['condiciones_pago']); ?></span>
                                    <?php if ($proveedor['dias_credito'] > 0): ?>
                                        <span class="badge bg-warning"><?php echo $proveedor['dias_credito']; ?> días</span>
                                    <?php endif; ?>
                                </p>
                                <a href="?accion=cuentas&proveedor=<?php echo $proveedor['id']; ?>" class="btn btn-sm btn-outline-primary">
                                    <i class="fas fa-list"></i> Ver Cuentas
                                </a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Modal Nueva Cuenta por Pagar -->
    <div class="modal fade" id="modalNuevaCuenta" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Nueva Cuenta por Pagar</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Proveedor *</label>
                                <select name="proveedor_id" class="form-control" required>
                                    <option value="">Seleccionar proveedor...</option>
                                    <?php foreach ($proveedores as $proveedor): ?>
                                        <option value="<?php echo $proveedor['id']; ?>">
                                            <?php echo htmlspecialchars($proveedor['razon_social']); ?> - <?php echo htmlspecialchars($proveedor['rut']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Número de Factura *</label>
                                <input type="text" name="numero_factura" class="form-control" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Fecha de Factura *</label>
                                <input type="date" name="fecha_factura" class="form-control" value="<?php echo date('Y-m-d'); ?>" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Fecha de Vencimiento *</label>
                                <input type="date" name="fecha_vencimiento" class="form-control" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Monto *</label>
                                <input type="number" name="monto_original" class="form-control" step="0.01" min="0" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Moneda</label>
                                <select name="moneda" class="form-control">
                                    <option value="CLP">CLP - Peso Chileno</option>
                                    <option value="USD">USD - Dólar</option>
                                    <option value="EUR">EUR - Euro</option>
                                </select>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Categoría</label>
                                <input type="text" name="categoria" class="form-control" placeholder="Ej: Servicios, Materiales">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Centro de Costo</label>
                                <input type="text" name="centro_costo" class="form-control" placeholder="Ej: Administración, Ventas">
                            </div>
                            
                            <div class="col-12">
                                <label class="form-label">Descripción *</label>
                                <textarea name="descripcion" class="form-control" rows="3" required></textarea>
                            </div>
                            
                            <div class="col-12">
                                <label class="form-label">Observaciones</label>
                                <textarea name="observaciones" class="form-control" rows="2"></textarea>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" name="crear_cuenta_pagar" class="btn btn-primary">Registrar Cuenta</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Nuevo Proveedor -->
    <div class="modal fade" id="modalNuevoProveedor" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Nuevo Proveedor</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">RUT *</label>
                                <input type="text" name="rut" class="form-control" placeholder="12.345.678-9" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Razón Social *</label>
                                <input type="text" name="razon_social" class="form-control" required>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Nombre de Contacto</label>
                                <input type="text" name="nombre_contacto" class="form-control">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Teléfono</label>
                                <input type="tel" name="telefono" class="form-control">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Email</label>
                                <input type="email" name="email" class="form-control">
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Giro</label>
                                <input type="text" name="giro" class="form-control">
                            </div>
                            
                            <div class="col-12">
                                <label class="form-label">Dirección</label>
                                <textarea name="direccion" class="form-control" rows="2"></textarea>
                            </div>
                            
                            <div class="col-md-4">
                                <label class="form-label">Condiciones de Pago</label>
                                <select name="condiciones_pago" class="form-control">
                                    <option value="contado">Contado</option>
                                    <option value="credito">Crédito</option>
                                </select>
                            </div>
                            
                            <div class="col-md-4">
                                <label class="form-label">Días de Crédito</label>
                                <input type="number" name="dias_credito" class="form-control" value="0" min="0">
                            </div>
                            
                            <div class="col-md-4">
                                <label class="form-label">Límite de Crédito</label>
                                <input type="number" name="limite_credito" class="form-control" value="0" min="0" step="0.01">
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" name="crear_proveedor" class="btn btn-primary">Crear Proveedor</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Registrar Pago -->
    <div class="modal fade" id="modalRegistrarPago" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Registrar Pago</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" id="formPago">
                    <div class="modal-body">
                        <input type="hidden" name="cuenta_por_pagar_id" id="cuentaPagarId">
                        
                        <div class="alert alert-info">
                            <strong>Proveedor:</strong> <span id="proveedorNombre"></span><br>
                            <strong>Monto Pendiente:</strong> $<span id="montoPendiente"></span>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Fecha de Pago *</label>
                            <input type="date" name="fecha_pago" class="form-control" value="<?php echo date('Y-m-d'); ?>" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Monto del Pago *</label>
                            <input type="number" name="monto_pago" class="form-control" step="0.01" min="0.01" required id="montoPagoInput">
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Método de Pago *</label>
                            <select name="metodo_pago" class="form-control" required>
                                <option value="">Seleccionar método...</option>
                                <option value="efectivo">Efectivo</option>
                                <option value="transferencia">Transferencia Bancaria</option>
                                <option value="cheque">Cheque</option>
                                <option value="debito">Tarjeta de Débito</option>
                                <option value="credito">Tarjeta de Crédito</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Número de Comprobante</label>
                            <input type="text" name="numero_comprobante" class="form-control" placeholder="Número de transferencia, cheque, etc.">
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Banco</label>
                            <input type="text" name="banco" class="form-control" placeholder="Nombre del banco">
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Observaciones del Pago</label>
                            <textarea name="observaciones_pago" class="form-control" rows="2"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" name="registrar_pago" class="btn btn-success">Registrar Pago</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function registrarPago(cuentaId, proveedorNombre, montoPendiente) {
            document.getElementById('cuentaPagarId').value = cuentaId;
            document.getElementById('proveedorNombre').textContent = proveedorNombre;
            document.getElementById('montoPendiente').textContent = montoPendiente.toLocaleString();
            document.getElementById('montoPagoInput').max = montoPendiente;
            
            const modal = new bootstrap.Modal(document.getElementById('modalRegistrarPago'));
            modal.show();
        }
        
        function verDetalles(cuentaId) {
            alert('Función de ver detalles en desarrollo. ID: ' + cuentaId);
        }
        
        // Formatear RUT mientras se escribe
        document.addEventListener('DOMContentLoaded', function() {
            const rutInputs = document.querySelectorAll('input[name="rut"]');
            rutInputs.forEach(input => {
                input.addEventListener('input', function(e) {
                    let value = e.target.value.replace(/[^0-9kK]/g, '');
                    if (value.length > 1) {
                        const numero = value.slice(0, -1);
                        const dv = value.slice(-1).toUpperCase();
                        if (numero.length > 0) {
                            const formatted = numero.replace(/\B(?=(\d{3})+(?!\d))/g, '.') + '-' + dv;
                            e.target.value = formatted;
                        }
                    }
                });
            });
        });
        
        // Auto-calcular fecha de vencimiento basada en días de crédito del proveedor
        document.querySelector('select[name="proveedor_id"]')?.addEventListener('change', function() {
            // Esta funcionalidad se puede implementar con AJAX para obtener los días de crédito del proveedor
        });
        
        console.log('Módulo Cuentas por Pagar - SISCONECTA ERP');
    </script>
</body>
</html>