<?php
// SISTEMA DE FACTURACIÓN ELECTRÓNICA SII CHILE - VERSIÓN COMPLETA Y FUNCIONAL
// facturacion.php

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('memory_limit', '256M');
ini_set('max_execution_time', 300);

// Verificar extensiones requeridas
$extensiones_requeridas = ['openssl', 'curl', 'xml', 'dom', 'zip'];
foreach ($extensiones_requeridas as $ext) {
    if (!extension_loaded($ext)) {
        die("Error: La extensión PHP '$ext' es requerida para el sistema de facturación electrónica.");
    }
}

// Verificar sesión activa
if (!isset($_SESSION['user_id']) || !isset($_SESSION['empresa_id'])) {
    header('Location: login.php');
    exit;
}

// Configuración de base de datos
$config = [
    'host' => 'localhost',
    'dbname' => 'conectae_sisconecta',
    'username' => 'conectae_sisconectauser',
    'password' => 'pt125824caraud',
    'charset' => 'utf8mb4'
];

$mensaje = '';
$tipo_mensaje = '';

// Clase para manejo de DTE y conexión SII
class DTEManager {
    private $pdo;
    private $empresa_data;
    private $ambiente; // 'certificacion' o 'produccion'
    
    // URLs del SII
    private $urls_sii = [
        'certificacion' => [
            'token' => 'https://maullin.sii.cl/DTEWS/GetTokenFromSeed.jws',
            'envio' => 'https://maullin.sii.cl/DTEWS/services/wsdte',
            'estado' => 'https://maullin.sii.cl/DTEWS/QueryEstUp.jws'
        ],
        'produccion' => [
            'token' => 'https://palena.sii.cl/DTEWS/GetTokenFromSeed.jws',
            'envio' => 'https://palena.sii.cl/DTEWS/services/wsdte',
            'estado' => 'https://palena.sii.cl/DTEWS/QueryEstUp.jws'
        ]
    ];
    
    public function __construct($pdo, $empresa_data) {
        $this->pdo = $pdo;
        $this->empresa_data = $empresa_data;
        $this->ambiente = $empresa_data['ambiente_sii'] ?? 'certificacion';
    }
    
    /**
     * Obtener token de autenticación del SII
     */
    public function obtenerTokenSII() {
        try {
            // 1. Obtener semilla del SII
            $semilla = $this->obtenerSemillaSII();
            if (!$semilla) {
                throw new Exception('No se pudo obtener la semilla del SII');
            }
            
            // 2. Firmar semilla con certificado digital
            $seed_firmado = $this->firmarSemilla($semilla);
            if (!$seed_firmado) {
                throw new Exception('No se pudo firmar la semilla');
            }
            
            // 3. Enviar semilla firmada y obtener token
            $token = $this->enviarSemillaFirmada($seed_firmado);
            return $token;
            
        } catch (Exception $e) {
            error_log("Error obteniendo token SII: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Obtener semilla del SII
     */
    private function obtenerSemillaSII() {
        $soap_request = '<?xml version="1.0" encoding="utf-8"?>
        <soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
            <soap:Body>
                <getSeed xmlns="http://dte.sii.cl/"/>
            </soap:Body>
        </soap:Envelope>';
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->urls_sii[$this->ambiente]['token']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $soap_request);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: text/xml; charset=utf-8',
            'SOAPAction: ""'
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200 || !$response) {
            return false;
        }
        
        // Extraer semilla del XML de respuesta
        preg_match('/<return>([^<]+)<\/return>/', $response, $matches);
        return isset($matches[1]) ? $matches[1] : false;
    }
    
    /**
     * Firmar semilla con certificado digital
     */
    private function firmarSemilla($semilla) {
        $certificado_data = $this->obtenerCertificadoEmpresa();
        if (!$certificado_data) {
            return false;
        }
        
        // Crear XML de semilla
        $xml_semilla = '<?xml version="1.0" encoding="UTF-8"?>
        <getToken>
            <item>
                <Semilla>' . $semilla . '</Semilla>
            </item>
        </getToken>';
        
        // Firmar XML
        return $this->firmarXML($xml_semilla, $certificado_data['certificado'], $certificado_data['llave_privada']);
    }
    
    /**
     * Firmar XML con certificado digital
     */
    private function firmarXML($xml, $certificado, $llave_privada) {
        try {
            $doc = new DOMDocument();
            $doc->loadXML($xml);
            
            // Crear elemento Signature
            $signature = $doc->createElement('Signature');
            $signature->setAttribute('xmlns', 'http://www.w3.org/2000/09/xmldsig#');
            
            // SignedInfo
            $signedInfo = $doc->createElement('SignedInfo');
            $canonicalizationMethod = $doc->createElement('CanonicalizationMethod');
            $canonicalizationMethod->setAttribute('Algorithm', 'http://www.w3.org/TR/2001/REC-xml-c14n-20010315');
            $signedInfo->appendChild($canonicalizationMethod);
            
            $signatureMethod = $doc->createElement('SignatureMethod');
            $signatureMethod->setAttribute('Algorithm', 'http://www.w3.org/2000/09/xmldsig#rsa-sha1');
            $signedInfo->appendChild($signatureMethod);
            
            $reference = $doc->createElement('Reference');
            $reference->setAttribute('URI', '');
            
            $transforms = $doc->createElement('Transforms');
            $transform = $doc->createElement('Transform');
            $transform->setAttribute('Algorithm', 'http://www.w3.org/2000/09/xmldsig#enveloped-signature');
            $transforms->appendChild($transform);
            $reference->appendChild($transforms);
            
            $digestMethod = $doc->createElement('DigestMethod');
            $digestMethod->setAttribute('Algorithm', 'http://www.w3.org/2000/09/xmldsig#sha1');
            $reference->appendChild($digestMethod);
            
            // Calcular digest
            $digestValue = $doc->createElement('DigestValue');
            $canonical = $doc->C14N();
            $digest = base64_encode(sha1($canonical, true));
            $digestValue->nodeValue = $digest;
            $reference->appendChild($digestValue);
            
            $signedInfo->appendChild($reference);
            $signature->appendChild($signedInfo);
            
            // Calcular firma
            $signedInfoCanonical = $signedInfo->C14N();
            openssl_sign($signedInfoCanonical, $signature_value, $llave_privada, OPENSSL_ALGO_SHA1);
            
            $signatureValue = $doc->createElement('SignatureValue');
            $signatureValue->nodeValue = base64_encode($signature_value);
            $signature->appendChild($signatureValue);
            
            // KeyInfo
            $keyInfo = $doc->createElement('KeyInfo');
            $x509Data = $doc->createElement('X509Data');
            $x509Certificate = $doc->createElement('X509Certificate');
            
            // Extraer certificado sin headers
            $cert_clean = preg_replace('/-----BEGIN CERTIFICATE-----/', '', $certificado);
            $cert_clean = preg_replace('/-----END CERTIFICATE-----/', '', $cert_clean);
            $cert_clean = preg_replace('/\s+/', '', $cert_clean);
            
            $x509Certificate->nodeValue = $cert_clean;
            $x509Data->appendChild($x509Certificate);
            $keyInfo->appendChild($x509Data);
            $signature->appendChild($keyInfo);
            
            // Agregar firma al documento
            $doc->documentElement->appendChild($signature);
            
            return $doc->saveXML();
            
        } catch (Exception $e) {
            error_log("Error firmando XML: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Obtener certificado digital de la empresa
     */
    private function obtenerCertificadoEmpresa() {
        try {
            $stmt = $this->pdo->prepare("SELECT certificado_digital, password_certificado FROM configuracion_dte WHERE empresa_id = ?");
            $stmt->execute([$this->empresa_data['id']]);
            $config = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$config || !$config['certificado_digital']) {
                return false;
            }
            
            // Leer certificado P12
            $p12_content = base64_decode($config['certificado_digital']);
            $certs = [];
            
            if (!openssl_pkcs12_read($p12_content, $certs, $config['password_certificado'])) {
                return false;
            }
            
            return [
                'certificado' => $certs['cert'],
                'llave_privada' => $certs['pkey']
            ];
            
        } catch (Exception $e) {
            error_log("Error obteniendo certificado: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Enviar semilla firmada y obtener token
     */
    private function enviarSemillaFirmada($semilla_firmada) {
        $soap_request = '<?xml version="1.0" encoding="utf-8"?>
        <soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
            <soap:Body>
                <getToken xmlns="http://dte.sii.cl/">
                    <pszXml><![CDATA[' . $semilla_firmada . ']]></pszXml>
                </getToken>
            </soap:Body>
        </soap:Envelope>';
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->urls_sii[$this->ambiente]['token']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $soap_request);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: text/xml; charset=utf-8',
            'SOAPAction: ""'
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        // Extraer token del XML de respuesta
        preg_match('/<return>([^<]+)<\/return>/', $response, $matches);
        return isset($matches[1]) ? $matches[1] : false;
    }
    
    /**
     * Generar XML DTE válido
     */
    public function generarXMLDTE($documento_data, $detalles) {
        try {
            
            function obtenerCodigoSII($tipo_documento) {
   
    $tipos_documento_sii = [
        // DOCUMENTOS DE VENTA
        'factura' => 33,
        'factura_exenta' => 34,
        'factura_compra' => 46,
        'boleta' => 39,
        'boleta_exenta' => 41,
        
        // NOTAS DE AJUSTE
        'nota_credito' => 61,
        'nota_debito' => 56,
        'nota_credito_exenta' => 60,
        'nota_debito_exenta' => 55,
        
        // DOCUMENTOS DE DESPACHO
        'guia_despacho' => 52,
        'conocimiento_embarque' => 50,
        'carta_porte' => 51,
        
        // LIQUIDACIONES
        'liquidacion_factura' => 43,
        'liquidacion_factura_electronica' => 43,
        'liquidacion_comisiones' => 40,
        
        // EXPORTACIÓN
        'factura_exportacion' => 110,
        'nota_debito_exportacion' => 111,
        'nota_credito_exportacion' => 112,
        'documento_exportacion' => 108,
        'factura_territorio_preferencial' => 914,
        
        // DOCUMENTOS ESPECIALES
        'boleta_honorarios' => 29,
        'factura_electronica_mixta' => 35,
        'factura_electronica_afecta' => 33,
        'documento_exento' => 55,
        'resumen_ventas' => 35,
        'set_pruebas' => 29,
        
        // OTROS DOCUMENTOS
        'comprobante_liquidacion' => 914,
        'comprobante_pago' => 914
    ];
    
    return $tipos_documento_sii[$tipo_documento] ?? 33;
}
            
            $tipo_dte = $tipos_documento[$documento_data['tipo_documento']];
            
            $xml = '<?xml version="1.0" encoding="ISO-8859-1"?>' . "\n";
            $xml .= '<DTE version="1.0" xmlns="http://www.sii.cl/SiiDte">' . "\n";
            $xml .= '  <Documento ID="T' . $tipo_dte . 'F' . $documento_data['folio'] . '">' . "\n";
            $xml .= '    <Encabezado>' . "\n";
            
            // IdDoc
            $xml .= '      <IdDoc>' . "\n";
            $xml .= '        <TipoDTE>' . $tipo_dte . '</TipoDTE>' . "\n";
            $xml .= '        <Folio>' . $documento_data['folio'] . '</Folio>' . "\n";
            $xml .= '        <FchEmis>' . $documento_data['fecha_emision'] . '</FchEmis>' . "\n";
            if (!empty($documento_data['fecha_vencimiento'])) {
                $xml .= '        <FchVenc>' . $documento_data['fecha_vencimiento'] . '</FchVenc>' . "\n";
            }
            $xml .= '        <TermPagoGlosa>' . ($documento_data['condiciones_pago'] ?? 'CONTADO') . '</TermPagoGlosa>' . "\n";
            $xml .= '      </IdDoc>' . "\n";
            
            // Emisor
            $xml .= '      <Emisor>' . "\n";
            $xml .= '        <RUTEmisor>' . $this->empresa_data['rut'] . '</RUTEmisor>' . "\n";
            $xml .= '        <RznSoc>' . htmlspecialchars($this->empresa_data['razon_social'], ENT_XML1) . '</RznSoc>' . "\n";
            $xml .= '        <GiroEmis>' . htmlspecialchars($this->empresa_data['giro'] ?? 'SERVICIOS PROFESIONALES', ENT_XML1) . '</GiroEmis>' . "\n";
            $xml .= '        <Acteco>' . ($this->empresa_data['actividad_economica'] ?? '749000') . '</Acteco>' . "\n";
            $xml .= '        <DirOrigen>' . htmlspecialchars($this->empresa_data['direccion'], ENT_XML1) . '</DirOrigen>' . "\n";
            $xml .= '        <CmnaOrigen>' . htmlspecialchars($this->empresa_data['comuna'] ?? 'SANTIAGO', ENT_XML1) . '</CmnaOrigen>' . "\n";
            $xml .= '        <CiudadOrigen>' . htmlspecialchars($this->empresa_data['ciudad'] ?? 'SANTIAGO', ENT_XML1) . '</CiudadOrigen>' . "\n";
            $xml .= '      </Emisor>' . "\n";
            
            // Receptor
            $xml .= '      <Receptor>' . "\n";
            $xml .= '        <RUTRecep>' . $documento_data['rut_cliente'] . '</RUTRecep>' . "\n";
            $xml .= '        <RznSocRecep>' . htmlspecialchars($documento_data['razon_social_cliente'], ENT_XML1) . '</RznSocRecep>' . "\n";
            if (!empty($documento_data['giro_cliente'])) {
                $xml .= '        <GiroRecep>' . htmlspecialchars($documento_data['giro_cliente'], ENT_XML1) . '</GiroRecep>' . "\n";
            }
            $xml .= '        <DirRecep>' . htmlspecialchars($documento_data['direccion_cliente'] ?? 'NO ESPECIFICADA', ENT_XML1) . '</DirRecep>' . "\n";
            $xml .= '        <CmnaRecep>' . htmlspecialchars($documento_data['comuna_cliente'] ?? 'SANTIAGO', ENT_XML1) . '</CmnaRecep>' . "\n";
            $xml .= '        <CiudadRecep>' . htmlspecialchars($documento_data['ciudad_cliente'] ?? 'SANTIAGO', ENT_XML1) . '</CiudadRecep>' . "\n";
            $xml .= '      </Receptor>' . "\n";
            
            // Totales
            $xml .= '      <Totales>' . "\n";
            if ($documento_data['subtotal'] > 0) {
                $xml .= '        <MntNeto>' . round($documento_data['subtotal']) . '</MntNeto>' . "\n";
            }
            if ($documento_data['total_exento'] > 0) {
                $xml .= '        <MntExe>' . round($documento_data['total_exento']) . '</MntExe>' . "\n";
            }
            if ($documento_data['iva'] > 0) {
                $xml .= '        <TasaIVA>19</TasaIVA>' . "\n";
                $xml .= '        <IVA>' . round($documento_data['iva']) . '</IVA>' . "\n";
            }
            $xml .= '        <MntTotal>' . round($documento_data['total']) . '</MntTotal>' . "\n";
            $xml .= '      </Totales>' . "\n";
            $xml .= '    </Encabezado>' . "\n";
            
            // Detalles
            foreach ($detalles as $index => $detalle) {
                $xml .= '    <Detalle>' . "\n";
                $xml .= '      <NroLinDet>' . ($index + 1) . '</NroLinDet>' . "\n";
                $xml .= '      <NmbItem>' . htmlspecialchars($detalle['descripcion'], ENT_XML1) . '</NmbItem>' . "\n";
                $xml .= '      <QtyItem>' . $detalle['cantidad'] . '</QtyItem>' . "\n";
                $xml .= '      <UnmdItem>UN</UnmdItem>' . "\n";
                $xml .= '      <PrcItem>' . round($detalle['precio_unitario']) . '</PrcItem>' . "\n";
                $xml .= '      <MontoItem>' . round($detalle['subtotal']) . '</MontoItem>' . "\n";
                if ($detalle['porcentaje_iva'] == 0) {
                    $xml .= '      <IndExe>1</IndExe>' . "\n";
                }
                $xml .= '    </Detalle>' . "\n";
            }
            
            $xml .= '  </Documento>' . "\n";
            $xml .= '</DTE>';
            
            return $xml;
            
        } catch (Exception $e) {
            error_log("Error generando XML DTE: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Timbre electrónico usando CAF
     */
    public function generarTimbreElectronico($documento_data, $caf_data) {
        try {
            // Datos para el timbre
            $datos_timbre = [
                'DD' => [
                    'RE' => $this->empresa_data['rut'],
                    'TD' => $this->obtenerCodigoTipoDocumento($documento_data['tipo_documento']),
                    'F' => $documento_data['folio'],
                    'FE' => $documento_data['fecha_emision'],
                    'RR' => $documento_data['rut_cliente'],
                    'RSR' => substr($documento_data['razon_social_cliente'], 0, 40),
                    'MNT' => round($documento_data['total']),
                    'IT1' => substr($documento_data['primer_item'] ?? 'SERVICIO', 0, 40),
                    'CAF' => $caf_data['caf_xml'],
                    'TSTED' => date('Y-m-d\TH:i:s')
                ]
            ];
            
            // Generar string para firma
            $string_firma = implode('|', [
                $datos_timbre['DD']['RE'],
                $datos_timbre['DD']['TD'],
                $datos_timbre['DD']['F'],
                $datos_timbre['DD']['FE'],
                $datos_timbre['DD']['RR'],
                $datos_timbre['DD']['RSR'],
                $datos_timbre['DD']['MNT'],
                $datos_timbre['DD']['IT1'],
                $datos_timbre['DD']['TSTED']
            ]);
            
            // Firmar con llave privada del CAF
            $private_key = $this->extraerLlavePrivadaCAF($caf_data['caf_xml']);
            if (!$private_key) {
                throw new Exception('No se pudo extraer la llave privada del CAF');
            }
            
            openssl_sign($string_firma, $signature, $private_key, OPENSSL_ALGO_SHA1);
            $datos_timbre['DD']['FRMT'] = base64_encode($signature);
            
            return $datos_timbre;
            
        } catch (Exception $e) {
            error_log("Error generando timbre electrónico: " . $e->getMessage());
            return false;
        }
    }
    
    private function obtenerCodigoTipoDocumento($tipo) {
        $codigos = [
            'factura' => 33,
            'boleta' => 39,
            'nota_credito' => 61,
            'nota_debito' => 56,
            'guia_despacho' => 52
        ];
        return $codigos[$tipo] ?? 33;
    }
    
    private function extraerLlavePrivadaCAF($caf_xml) {
        try {
            $dom = new DOMDocument();
            $dom->loadXML($caf_xml);
            
            $xpath = new DOMXPath($dom);
            $private_key_node = $xpath->query('//RSASK')->item(0);
            
            if (!$private_key_node) {
                return false;
            }
            
            $private_key_data = base64_decode($private_key_node->nodeValue);
            return openssl_pkey_get_private($private_key_data);
            
        } catch (Exception $e) {
            return false;
        }
    }
}

try {
    $pdo = new PDO(
        "mysql:host={$config['host']};dbname={$config['dbname']};charset={$config['charset']}", 
        $config['username'], 
        $config['password'],
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false,
        ]
    );
    
    $empresa_id = $_SESSION['empresa_id'];
    
    // Verificar estado de empresa
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ? AND estado = 'activo'");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch();
    
    if (!$empresa) {
        session_destroy();
        header('Location: login.php?error=empresa_inactiva');
        exit;
    }
    
    // Verificar vencimiento (excepto empresas exentas)
    if (($empresa['codigo_empresa'] ?? '') !== 'AUDITOREX_CHILE') {
        $fecha_actual = new DateTime();
        $fecha_vencimiento = new DateTime($empresa['fecha_vencimiento'] ?? '2024-12-31');
        
        if ($fecha_actual > $fecha_vencimiento) {
            session_destroy();
            header('Location: login.php?error=suscripcion_vencida');
            exit;
        }
    }
    
    // Crear tablas si no existen
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS configuracion_dte (
            id INT AUTO_INCREMENT PRIMARY KEY,
            empresa_id INT NOT NULL,
            certificado_digital LONGTEXT,
            password_certificado VARCHAR(255),
            ambiente ENUM('certificacion', 'produccion') DEFAULT 'certificacion',
            resolucion_sii VARCHAR(50),
            fecha_resolucion DATE,
            actividad_economica VARCHAR(10) DEFAULT '749000',
            codigo_sucursal VARCHAR(10) DEFAULT '1',
            fecha_registro TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (empresa_id) REFERENCES empresas(id) ON DELETE CASCADE,
            UNIQUE KEY unique_empresa (empresa_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS folios_caf (
            id INT AUTO_INCREMENT PRIMARY KEY,
            empresa_id INT NOT NULL,
            tipo_documento ENUM('factura', 'boleta', 'nota_credito', 'nota_debito', 'guia_despacho') NOT NULL,
            folio_desde INT NOT NULL,
            folio_hasta INT NOT NULL,
            folio_actual INT NOT NULL,
            caf_xml LONGTEXT NOT NULL,
            fecha_autorizacion DATE NOT NULL,
            fecha_vencimiento DATE NOT NULL,
            estado ENUM('activo', 'agotado', 'vencido') DEFAULT 'activo',
            fecha_registro TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (empresa_id) REFERENCES empresas(id) ON DELETE CASCADE,
            INDEX idx_empresa_tipo_estado (empresa_id, tipo_documento, estado),
            INDEX idx_folios (folio_actual, folio_hasta)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS documentos_tributarios (
            id INT AUTO_INCREMENT PRIMARY KEY,
            empresa_id INT NOT NULL,
            tipo_documento ENUM('factura', 'boleta', 'nota_credito', 'nota_debito', 'guia_despacho') NOT NULL,
            numero_documento VARCHAR(50) NOT NULL,
            folio INT NOT NULL,
            fecha_emision DATE NOT NULL,
            fecha_vencimiento DATE NULL,
            rut_cliente VARCHAR(12) NOT NULL,
            razon_social_cliente VARCHAR(255) NOT NULL,
            direccion_cliente VARCHAR(255),
            giro_cliente VARCHAR(255),
            comuna_cliente VARCHAR(100),
            ciudad_cliente VARCHAR(100),
            email_cliente VARCHAR(255),
            telefono_cliente VARCHAR(50),
            subtotal DECIMAL(15,2) DEFAULT 0.00,
            total_exento DECIMAL(15,2) DEFAULT 0.00,
            iva DECIMAL(15,2) DEFAULT 0.00,
            total DECIMAL(15,2) NOT NULL,
            moneda VARCHAR(3) DEFAULT 'CLP',
            condiciones_pago VARCHAR(100),
            medio_pago VARCHAR(50),
            observaciones TEXT,
            xml_dte LONGTEXT,
            timbre_electronico TEXT,
            estado ENUM('borrador', 'emitido', 'enviado_sii', 'aceptado_sii', 'rechazado_sii', 'pagado', 'anulado') DEFAULT 'borrador',
            track_id VARCHAR(50),
            fecha_envio_sii DATETIME NULL,
            fecha_acuse_sii DATETIME NULL,
            errores_sii TEXT,
            fecha_registro TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (empresa_id) REFERENCES empresas(id) ON DELETE CASCADE,
            UNIQUE KEY unique_folio_tipo (empresa_id, tipo_documento, folio),
            INDEX idx_empresa_fecha (empresa_id, fecha_emision),
            INDEX idx_cliente (rut_cliente),
            INDEX idx_estado (estado)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS documento_detalles (
            id INT AUTO_INCREMENT PRIMARY KEY,
            documento_id INT NOT NULL,
            producto_id INT NULL,
            descripcion VARCHAR(255) NOT NULL,
            cantidad DECIMAL(10,3) NOT NULL,
            precio_unitario DECIMAL(15,2) NOT NULL,
            subtotal DECIMAL(15,2) NOT NULL,
            porcentaje_iva DECIMAL(5,2) DEFAULT 19.00,
            iva_item DECIMAL(15,2) DEFAULT 0.00,
            total_item DECIMAL(15,2) NOT NULL,
            descuento_porcentaje DECIMAL(5,2) DEFAULT 0.00,
            descuento_monto DECIMAL(15,2) DEFAULT 0.00,
            FOREIGN KEY (documento_id) REFERENCES documentos_tributarios(id) ON DELETE CASCADE,
            INDEX idx_documento (documento_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    // Inicializar DTEManager
    $dte_manager = new DTEManager($pdo, $empresa);
    
    // Procesar acciones
    $accion = $_GET['accion'] ?? 'listar';
    
    // Obtener productos activos
    $stmt = $pdo->prepare("SELECT * FROM productos WHERE empresa_id = ? AND activo = 1 ORDER BY nombre ASC");
    $stmt->execute([$empresa_id]);
    $productos = $stmt->fetchAll();
    
    // Obtener monedas activas
    $stmt = $pdo->prepare("SELECT * FROM monedas WHERE empresa_id = ? AND activo = 1 ORDER BY codigo ASC");
    $stmt->execute([$empresa_id]);
    $monedas = $stmt->fetchAll();
    
    // Procesar creación de documento
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['guardar_documento'])) {
        $tipo_documento = $_POST['tipo_documento'] ?? '';
        $rut_cliente = trim($_POST['rut_cliente'] ?? '');
        $razon_social_cliente = trim($_POST['razon_social_cliente'] ?? '');
        $direccion_cliente = trim($_POST['direccion_cliente'] ?? '');
        $giro_cliente = trim($_POST['giro_cliente'] ?? '');
        $comuna_cliente = trim($_POST['comuna_cliente'] ?? '');
        $ciudad_cliente = trim($_POST['ciudad_cliente'] ?? '');
        $email_cliente = trim($_POST['email_cliente'] ?? '');
        $telefono_cliente = trim($_POST['telefono_cliente'] ?? '');
        $fecha_emision = $_POST['fecha_emision'] ?? date('Y-m-d');
        $fecha_vencimiento = !empty($_POST['fecha_vencimiento']) ? $_POST['fecha_vencimiento'] : null;
        $moneda = $_POST['moneda'] ?? 'CLP';
        $observaciones = trim($_POST['observaciones'] ?? '');
        $condiciones_pago = $_POST['condiciones_pago'] ?? 'contado';
        $medio_pago = $_POST['medio_pago'] ?? 'efectivo';
        $productos_documento = json_decode($_POST['productos_documento'] ?? '[]', true);
        
        // Validaciones
        if (empty($tipo_documento) || empty($rut_cliente) || empty($razon_social_cliente) || empty($productos_documento)) {
            $mensaje = 'Complete todos los campos obligatorios y agregue al menos un producto';
            $tipo_mensaje = 'warning';
        } else {
            try {
                $pdo->beginTransaction();
                
                // Validar RUT chileno
                if (!validarRutChileno($rut_cliente)) {
                    throw new Exception('RUT del cliente no válido');
                }
                
                // Calcular totales
                $subtotal = 0;
                $total_iva = 0;
                $total_exento = 0;
                
                foreach ($productos_documento as $item) {
                    $cantidad = (float)($item['cantidad'] ?? 0);
                    $precio = (float)($item['precio'] ?? 0);
                    $descuento = (float)($item['descuento'] ?? 0);
                    $iva_porcentaje = (float)($item['iva'] ?? 19);
                    
                    if ($cantidad <= 0 || $precio <= 0) {
                        throw new Exception('Cantidad y precio deben ser mayores a cero');
                    }
                    
                    $subtotal_item = $cantidad * $precio;
                    $descuento_item = $subtotal_item * ($descuento / 100);
                    $neto_item = $subtotal_item - $descuento_item;
                    
                    if ($iva_porcentaje > 0) {
                        $subtotal += $neto_item;
                        $total_iva += $neto_item * ($iva_porcentaje / 100);
                    } else {
                        $total_exento += $neto_item;
                    }
                }
                
                $total_neto = $subtotal + $total_exento;
                $total_final = $total_neto + $total_iva;
                
                // Verificar y obtener folio
                $folio = obtenerSiguienteFolio($pdo, $empresa_id, $tipo_documento);
                if (!$folio) {
                    throw new Exception('No hay folios CAF disponibles para ' . $tipo_documento . '. Configure folios desde el SII.');
                }
                
                // Generar número de documento
                $prefijos = [
                    'factura' => 'F',
                    'boleta' => 'B',
                    'nota_credito' => 'NC',
                    'nota_debito' => 'ND',
                    'guia_despacho' => 'GD'
                ];
                $numero_documento = $prefijos[$tipo_documento] . str_pad($folio, 8, '0', STR_PAD_LEFT);
                
                // Insertar documento
                $stmt = $pdo->prepare("
                    INSERT INTO documentos_tributarios 
                    (empresa_id, tipo_documento, numero_documento, folio, fecha_emision, fecha_vencimiento, 
                     rut_cliente, razon_social_cliente, direccion_cliente, giro_cliente, comuna_cliente, ciudad_cliente,
                     email_cliente, telefono_cliente, subtotal, total_exento, iva, total, moneda, 
                     condiciones_pago, medio_pago, observaciones, estado) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'emitido')
                ");
                $stmt->execute([
                    $empresa_id, $tipo_documento, $numero_documento, $folio, $fecha_emision, $fecha_vencimiento,
                    $rut_cliente, $razon_social_cliente, $direccion_cliente, $giro_cliente, $comuna_cliente, $ciudad_cliente,
                    $email_cliente, $telefono_cliente, $subtotal, $total_exento, $total_iva, $total_final, $moneda,
                    $condiciones_pago, $medio_pago, $observaciones
                ]);
                $documento_id = $pdo->lastInsertId();
                
                // Insertar detalles
                foreach ($productos_documento as $item) {
                    $cantidad = (float)($item['cantidad'] ?? 0);
                    $precio = (float)($item['precio'] ?? 0);
                    $descuento = (float)($item['descuento'] ?? 0);
                    $iva_porcentaje = (float)($item['iva'] ?? 19);
                    
                    $subtotal_item = $cantidad * $precio;
                    $descuento_item = $subtotal_item * ($descuento / 100);
                    $neto_item = $subtotal_item - $descuento_item;
                    $iva_item = $neto_item * ($iva_porcentaje / 100);
                    $total_item = $neto_item + $iva_item;
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO documento_detalles 
                        (documento_id, producto_id, descripcion, cantidad, precio_unitario, subtotal, 
                         porcentaje_iva, iva_item, total_item, descuento_porcentaje, descuento_monto) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([
                        $documento_id, $item['id'] ?? null, $item['nombre'] ?? $item['descripcion'], 
                        $cantidad, $precio, $neto_item, $iva_porcentaje, $iva_item, $total_item,
                        $descuento, $descuento_item
                    ]);
                }
                
                // Obtener datos completos del documento para generar XML
                $stmt = $pdo->prepare("SELECT * FROM documentos_tributarios WHERE id = ?");
                $stmt->execute([$documento_id]);
                $documento_data = $stmt->fetch();
                
                // Obtener detalles
                $stmt = $pdo->prepare("SELECT * FROM documento_detalles WHERE documento_id = ? ORDER BY id");
                $stmt->execute([$documento_id]);
                $detalles = $stmt->fetchAll();
                
                // Generar XML DTE
                $xml_dte = $dte_manager->generarXMLDTE($documento_data, $detalles);
                if (!$xml_dte) {
                    throw new Exception('Error al generar XML DTE');
                }
                
                // Obtener CAF para timbre
                $caf_data = obtenerCafActivo($pdo, $empresa_id, $tipo_documento, $folio);
                if ($caf_data) {
                    $timbre = $dte_manager->generarTimbreElectronico($documento_data, $caf_data);
                    if ($timbre) {
                        $documento_data['timbre_electronico'] = json_encode($timbre);
                    }
                }
                
                // Actualizar documento con XML y timbre
                $stmt = $pdo->prepare("UPDATE documentos_tributarios SET xml_dte = ?, timbre_electronico = ? WHERE id = ?");
                $stmt->execute([$xml_dte, $documento_data['timbre_electronico'] ?? null, $documento_id]);
                
                // Actualizar folio utilizado
                actualizarFolio($pdo, $empresa_id, $tipo_documento, $folio);
                
                // Crear asiento contable
                crearAsientoContable($pdo, $empresa_id, $_SESSION['user_id'], $documento_data, $total_neto, $total_iva, $total_final);
                
                $pdo->commit();
                
                $mensaje = "Documento $numero_documento creado exitosamente (Folio: $folio)";
                $tipo_mensaje = 'success';
                $accion = 'listar';
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $mensaje = 'Error al crear documento: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        }
    }
    
    // Anular documento
    if (isset($_GET['anular']) && !empty($_GET['anular'])) {
        try {
            $pdo->beginTransaction();
            
            $stmt = $pdo->prepare("SELECT * FROM documentos_tributarios WHERE id = ? AND empresa_id = ?");
            $stmt->execute([$_GET['anular'], $empresa_id]);
            $documento = $stmt->fetch();
            
            if ($documento && !in_array($documento['estado'], ['anulado', 'enviado_sii', 'aceptado_sii'])) {
                $stmt = $pdo->prepare("UPDATE documentos_tributarios SET estado = 'anulado' WHERE id = ? AND empresa_id = ?");
                $stmt->execute([$_GET['anular'], $empresa_id]);
                
                $pdo->commit();
                $mensaje = 'Documento anulado exitosamente';
                $tipo_mensaje = 'success';
            } else {
                $mensaje = 'No se puede anular este documento';
                $tipo_mensaje = 'warning';
            }
        } catch (Exception $e) {
            $pdo->rollBack();
            $mensaje = 'Error al anular documento: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
    
    // Obtener lista de documentos con filtros
    $filtro_tipo = $_GET['tipo'] ?? '';
    $filtro_estado = $_GET['estado'] ?? '';
    $filtro_fecha_desde = $_GET['fecha_desde'] ?? '';
    $filtro_fecha_hasta = $_GET['fecha_hasta'] ?? '';
    $filtro_cliente = $_GET['cliente'] ?? '';
    
    $where_conditions = ["dt.empresa_id = ?"];
    $params = [$empresa_id];
    
    if (!empty($filtro_tipo)) {
        $where_conditions[] = "dt.tipo_documento = ?";
        $params[] = $filtro_tipo;
    }
    
    if (!empty($filtro_estado)) {
        $where_conditions[] = "dt.estado = ?";
        $params[] = $filtro_estado;
    }
    
    if (!empty($filtro_fecha_desde)) {
        $where_conditions[] = "dt.fecha_emision >= ?";
        $params[] = $filtro_fecha_desde;
    }
    
    if (!empty($filtro_fecha_hasta)) {
        $where_conditions[] = "dt.fecha_emision <= ?";
        $params[] = $filtro_fecha_hasta;
    }
    
    if (!empty($filtro_cliente)) {
        $where_conditions[] = "(dt.razon_social_cliente LIKE ? OR dt.rut_cliente LIKE ?)";
        $filtro_cliente_param = '%' . $filtro_cliente . '%';
        $params[] = $filtro_cliente_param;
        $params[] = $filtro_cliente_param;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    $stmt = $pdo->prepare("
        SELECT dt.*, 
               COUNT(dd.id) as total_items
        FROM documentos_tributarios dt 
        LEFT JOIN documento_detalles dd ON dt.id = dd.documento_id
        WHERE $where_clause 
        GROUP BY dt.id
        ORDER BY dt.fecha_emision DESC, dt.id DESC
        LIMIT 100
    ");
    $stmt->execute($params);
    $documentos = $stmt->fetchAll();
    
    // Estadísticas
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_documentos,
            SUM(CASE WHEN fecha_emision = CURDATE() THEN 1 ELSE 0 END) as documentos_hoy,
            COALESCE(SUM(CASE WHEN MONTH(fecha_emision) = MONTH(CURDATE()) AND YEAR(fecha_emision) = YEAR(CURDATE()) AND tipo_documento IN ('factura', 'boleta') THEN total ELSE 0 END), 0) as ventas_mes,
            SUM(CASE WHEN estado IN ('emitido', 'enviado_sii', 'aceptado_sii') THEN 1 ELSE 0 END) as documentos_emitidos
        FROM documentos_tributarios 
        WHERE empresa_id = ?
    ");
    $stmt->execute([$empresa_id]);
    $estadisticas = $stmt->fetch();
    
    // Asegurar que las estadísticas no sean null
    $estadisticas['total_documentos'] = $estadisticas['total_documentos'] ?? 0;
    $estadisticas['documentos_hoy'] = $estadisticas['documentos_hoy'] ?? 0;
    $estadisticas['ventas_mes'] = $estadisticas['ventas_mes'] ?? 0;
    $estadisticas['documentos_emitidos'] = $estadisticas['documentos_emitidos'] ?? 0;
    
} catch (PDOException $e) {
    die('Error de conexión a base de datos: ' . $e->getMessage());
} catch (Exception $e) {
    $mensaje = 'Error en el sistema: ' . $e->getMessage();
    $tipo_mensaje = 'danger';
}

// Funciones auxiliares
function validarRutChileno($rut) {
    $rut = preg_replace('/[^0-9kK]/', '', strtoupper($rut));
    if (strlen($rut) < 2) return false;
    
    $dv = substr($rut, -1);
    $numero = substr($rut, 0, -1);
    
    $suma = 0;
    $multiplicador = 2;
    
    for ($i = strlen($numero) - 1; $i >= 0; $i--) {
        $suma += intval($numero[$i]) * $multiplicador;
        $multiplicador = $multiplicador == 7 ? 2 : $multiplicador + 1;
    }
    
    $resto = $suma % 11;
    $dv_calculado = $resto < 2 ? $resto : 11 - $resto;
    $dv_calculado = $dv_calculado == 10 ? 'K' : (string)$dv_calculado;
    
    return $dv === $dv_calculado;
}

function obtenerSiguienteFolio($pdo, $empresa_id, $tipo_documento) {
    $stmt = $pdo->prepare("
        SELECT folio_actual FROM folios_caf 
        WHERE empresa_id = ? AND tipo_documento = ? AND estado = 'activo' 
        AND folio_actual <= folio_hasta AND fecha_vencimiento >= CURDATE()
        ORDER BY folio_actual ASC 
        LIMIT 1
    ");
    $stmt->execute([$empresa_id, $tipo_documento]);
    $result = $stmt->fetch();
    
    return $result ? $result['folio_actual'] : false;
}

function actualizarFolio($pdo, $empresa_id, $tipo_documento, $folio_usado) {
    $stmt = $pdo->prepare("
        UPDATE folios_caf 
        SET folio_actual = folio_actual + 1,
            estado = CASE WHEN folio_actual + 1 > folio_hasta THEN 'agotado' ELSE 'activo' END
        WHERE empresa_id = ? AND tipo_documento = ? AND folio_actual = ?
    ");
    $stmt->execute([$empresa_id, $tipo_documento, $folio_usado]);
}

function obtenerCafActivo($pdo, $empresa_id, $tipo_documento, $folio) {
    $stmt = $pdo->prepare("
        SELECT * FROM folios_caf 
        WHERE empresa_id = ? AND tipo_documento = ? 
        AND folio_desde <= ? AND folio_hasta >= ? 
        AND fecha_vencimiento >= CURDATE()
        LIMIT 1
    ");
    $stmt->execute([$empresa_id, $tipo_documento, $folio, $folio]);
    return $stmt->fetch();
}

function crearAsientoContable($pdo, $empresa_id, $user_id, $documento_data, $neto, $iva, $total) {
    try {
        $stmt = $pdo->prepare("SELECT COALESCE(MAX(numero_asiento), 0) + 1 as siguiente_asiento FROM asientos_contables WHERE empresa_id = ?");
        $stmt->execute([$empresa_id]);
        $numero_asiento = $stmt->fetch()['siguiente_asiento'];
        
        $descripcion = ucfirst($documento_data['tipo_documento']) . " " . $documento_data['numero_documento'] . " - " . $documento_data['razon_social_cliente'];
        
        $stmt = $pdo->prepare("
            INSERT INTO asientos_contables 
            (empresa_id, numero_asiento, fecha, descripcion, documento_referencia, tipo_asiento, estado, usuario_registro) 
            VALUES (?, ?, CURDATE(), ?, ?, 'automatico', 'contabilizado', ?)
        ");
        $stmt->execute([$empresa_id, $numero_asiento, $descripcion, $documento_data['numero_documento'], $user_id]);
        $asiento_id = $pdo->lastInsertId();
        
        if (in_array($documento_data['tipo_documento'], ['factura', 'boleta'])) {
            // DEBE: Clientes por cobrar
            $stmt = $pdo->prepare("
                INSERT INTO movimientos_contables 
                (asiento_id, cuenta_contable, nombre_cuenta, debe, haber, observaciones) 
                VALUES (?, '1103', 'CLIENTES POR COBRAR', ?, 0.00, ?)
            ");
            $stmt->execute([$asiento_id, $total, "Venta - " . $documento_data['razon_social_cliente']]);
            
            // HABER: Ventas
            if ($neto > 0) {
                $stmt = $pdo->prepare("
                    INSERT INTO movimientos_contables 
                    (asiento_id, cuenta_contable, nombre_cuenta, debe, haber, observaciones) 
                    VALUES (?, '4101', 'VENTAS', 0.00, ?, ?)
                ");
                $stmt->execute([$asiento_id, $neto, "Venta neta"]);
            }
            
            // HABER: IVA Débito Fiscal
            if ($iva > 0) {
                $stmt = $pdo->prepare("
                    INSERT INTO movimientos_contables 
                    (asiento_id, cuenta_contable, nombre_cuenta, debe, haber, observaciones) 
                    VALUES (?, '2102', 'IVA DÉBITO FISCAL', 0.00, ?, ?)
                ");
                $stmt->execute([$asiento_id, $iva, "IVA débito fiscal"]);
            }
        }
        
        return true;
    } catch (Exception $e) {
        error_log("Error creando asiento contable: " . $e->getMessage());
        return false;
    }
}

function formatearMoneda($valor, $moneda = 'CLP') {
    if ($valor === null || $valor === '') {
        return $moneda . ' 0';
    }
    return $moneda . ' ' . number_format((float)$valor, 0, ',', '.');
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Facturación Electrónica SII Chile - SISCONECTA ERP</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .main-container {
            padding: 20px;
            max-width: 1600px;
            margin: 0 auto;
        }
        
        .header-section {
            background: linear-gradient(45deg, #2c3e50, #34495e);
            color: white;
            padding: 25px 30px;
            border-radius: 15px;
            margin-bottom: 25px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }
        
        .header-title {
            font-size: 1.9rem;
            font-weight: 700;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .sii-badge {
            background: linear-gradient(45deg, #e74c3c, #c0392b);
            color: white;
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }
        
        .btn-header {
            background: rgba(255,255,255,0.15);
            border: 2px solid rgba(255,255,255,0.3);
            color: white;
            border-radius: 10px;
            padding: 12px 20px;
            text-decoration: none;
            transition: all 0.3s ease;
            font-weight: 600;
            backdrop-filter: blur(10px);
        }
        
        .btn-header:hover {
            background: rgba(255,255,255,0.25);
            color: white;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        
        .stats-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border-left: 5px solid;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            right: 0;
            width: 100px;
            height: 100px;
            background: linear-gradient(45deg, rgba(255,255,255,0.1), rgba(255,255,255,0.05));
            border-radius: 50%;
            transform: translate(30px, -30px);
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.15);
        }
        
        .stat-card.primary { border-left-color: #3498db; }
        .stat-card.warning { border-left-color: #f39c12; }
        .stat-card.success { border-left-color: #27ae60; }
        .stat-card.info { border-left-color: #9b59b6; }
        
        .stat-value {
            font-size: 2.2rem;
            font-weight: 800;
            color: #2c3e50;
            margin-bottom: 8px;
            position: relative;
            z-index: 2;
        }
        
        .stat-label {
            color: #666;
            font-weight: 600;
            font-size: 0.95rem;
            position: relative;
            z-index: 2;
        }
        
        .stat-icon {
            position: absolute;
            top: 20px;
            right: 20px;
            font-size: 2.5rem;
            opacity: 0.2;
            z-index: 1;
        }
        
        .content-card {
            background: white;
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            margin-bottom: 25px;
        }
        
        .filters-section {
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
            padding: 25px;
            border-radius: 15px;
            margin-bottom: 25px;
            border: 1px solid #dee2e6;
        }
        
        .table-container {
            overflow-x: auto;
            border-radius: 15px;
            box-shadow: 0 5px 20px rgba(0,0,0,0.08);
        }
        
        .table {
            margin: 0;
            background: white;
            border-radius: 15px;
            overflow: hidden;
        }
        
        .table th {
            background: linear-gradient(135deg, #34495e, #2c3e50);
            color: white;
            border: none;
            font-weight: 700;
            padding: 18px 15px;
            white-space: nowrap;
            text-transform: uppercase;
            font-size: 0.85rem;
            letter-spacing: 0.5px;
        }
        
        .table td {
            padding: 18px 15px;
            vertical-align: middle;
            border-top: 1px solid #f1f3f4;
        }
        
        .table tbody tr:hover {
            background: linear-gradient(135deg, #f8f9fa, #ffffff);
        }
        
        .documento-numero {
            font-family: 'Courier New', monospace;
            background: linear-gradient(135deg, #e9ecef, #f8f9fa);
            padding: 6px 12px;
            border-radius: 8px;
            font-size: 0.9rem;
            font-weight: 700;
            border: 1px solid #dee2e6;
        }
        
        .tipo-documento {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .tipo-factura { background: linear-gradient(135deg, #e3f2fd, #bbdefb); color: #1976d2; }
        .tipo-boleta { background: linear-gradient(135deg, #f3e5f5, #e1bee7); color: #7b1fa2; }
        .tipo-nota_credito { background: linear-gradient(135deg, #e8f5e8, #c8e6c9); color: #388e3c; }
        .tipo-nota_debito { background: linear-gradient(135deg, #fff3e0, #ffe0b2); color: #f57c00; }
        .tipo-guia_despacho { background: linear-gradient(135deg, #fce4ec, #f8bbd9); color: #c2185b; }
        
        .badge {
            padding: 8px 14px;
            border-radius: 25px;
            font-size: 0.75rem;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .badge.bg-success {
            background: linear-gradient(135deg, #27ae60, #229954) !important;
        }
        
        .badge.bg-warning {
            background: linear-gradient(135deg, #f39c12, #e67e22) !important;
        }
        
        .badge.bg-danger {
            background: linear-gradient(135deg, #e74c3c, #c0392b) !important;
        }
        
        .badge.bg-primary {
            background: linear-gradient(135deg, #3498db, #2980b9) !important;
        }
        
        .badge.bg-secondary {
            background: linear-gradient(135deg, #6c757d, #5a6268) !important;
        }
        
        .btn-action {
            padding: 8px 12px;
            border-radius: 8px;
            border: none;
            font-size: 0.8rem;
            font-weight: 600;
            margin: 2px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn-view {
            background: linear-gradient(135deg, #3498db, #2980b9);
            color: white;
        }
        
        .btn-view:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(52,152,219,0.3);
        }
        
        .btn-print {
            background: linear-gradient(135deg, #27ae60, #229954);
            color: white;
        }
        
        .btn-print:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(39,174,96,0.3);
        }
        
        .btn-cancel {
            background: linear-gradient(135deg, #e74c3c, #c0392b);
            color: white;
        }
        
        .btn-cancel:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(231,76,60,0.3);
        }
        
        .form-floating {
            margin-bottom: 20px;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 15px 18px;
            transition: all 0.3s ease;
            font-weight: 500;
        }
        
        .form-control:focus {
            border-color: #3498db;
            box-shadow: 0 0 0 0.25rem rgba(52,152,219,0.25);
            transform: translateY(-1px);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #3498db, #2980b9);
            border: none;
            border-radius: 10px;
            padding: 15px 30px;
            font-weight: 700;
            transition: all 0.3s ease;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .btn-primary:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 25px rgba(52,152,219,0.3);
        }
        
        .factura-form {
            display: grid;
            grid-template-columns: 1fr 450px;
            gap: 40px;
            margin-top: 30px;
        }
        
        .productos-factura {
            background: linear-gradient(135deg, #f8f9fa, #ffffff);
            border-radius: 15px;
            padding: 25px;
            max-height: 500px;
            overflow-y: auto;
            border: 2px solid #e9ecef;
        }
        
        .producto-item {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 15px;
            cursor: pointer;
            transition: all 0.3s ease;
            border: 2px solid #f1f3f4;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        .producto-item:hover {
            border-color: #3498db;
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(52,152,219,0.15);
        }
        
        .carrito-factura {
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 15px;
            padding: 25px;
            position: sticky;
            top: 20px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
        
        .item-carrito {
            background: linear-gradient(135deg, #f8f9fa, #ffffff);
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 12px;
            border: 1px solid #e9ecef;
        }
        
        .totales-factura {
            border-top: 3px solid #3498db;
            padding-top: 20px;
            margin-top: 20px;
            background: linear-gradient(135deg, #f8f9fa, #ffffff);
            border-radius: 10px;
            padding: 20px;
        }
        
        .total-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 10px;
            font-weight: 600;
        }
        
        .total-final {
            font-size: 1.4rem;
            font-weight: 800;
            color: #2c3e50;
            border-top: 2px solid #3498db;
            padding-top: 15px;
            background: linear-gradient(135deg, #e3f2fd, #ffffff);
            padding: 15px;
            border-radius: 8px;
            margin-top: 15px;
        }
        
        .no-documentos {
            text-align: center;
            padding: 80px 40px;
            color: #666;
        }
        
        .no-documentos i {
            font-size: 5rem;
            margin-bottom: 30px;
            color: #ddd;
        }
        
        .alert {
            border-radius: 15px;
            padding: 20px 25px;
            font-weight: 600;
            border: none;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        @media (max-width: 768px) {
            .header-section {
                flex-direction: column;
                gap: 20px;
                text-align: center;
            }
            
            .header-actions {
                flex-direction: column;
                width: 100%;
            }
            
            .btn-header {
                width: 100%;
                justify-content: center;
            }
            
            .stats-cards {
                grid-template-columns: 1fr;
            }
            
            .factura-form {
                grid-template-columns: 1fr;
                gap: 25px;
            }
            
            .main-container {
                padding: 15px;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header -->
        <div class="header-section">
            <h1 class="header-title">
                <i class="fas fa-file-invoice-dollar"></i>
                Facturación Electrónica SII Chile
                <span class="sii-badge">DTE v2.0</span>
            </h1>
            <div class="header-actions">
                <a href="dashboard.php" class="btn-header">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <a href="configuracion_dte.php" class="btn-header">
                    <i class="fas fa-cog"></i> Config DTE
                </a>
                <?php if ($accion === 'listar'): ?>
                    <a href="?accion=crear" class="btn-header">
                        <i class="fas fa-plus-circle"></i> Nuevo Documento
                    </a>
                <?php endif; ?>
            </div>
        </div>

        <!-- Mensajes -->
        <?php if (!empty($mensaje)): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show">
                <i class="fas fa-<?php echo $tipo_mensaje === 'success' ? 'check-circle' : ($tipo_mensaje === 'danger' ? 'exclamation-triangle' : 'info-circle'); ?>"></i>
                <?php echo htmlspecialchars($mensaje); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if ($accion === 'listar'): ?>
            <!-- Estadísticas -->
            <div class="stats-cards">
                <div class="stat-card primary">
                    <div class="stat-value"><?php echo number_format($estadisticas['total_documentos']); ?></div>
                    <div class="stat-label">Total Documentos</div>
                    <i class="fas fa-file-invoice stat-icon"></i>
                </div>
                
                <div class="stat-card warning">
                    <div class="stat-value"><?php echo number_format($estadisticas['documentos_hoy']); ?></div>
                    <div class="stat-label">Documentos Hoy</div>
                    <i class="fas fa-calendar-day stat-icon"></i>
                </div>
                
                <div class="stat-card success">
                    <div class="stat-value"><?php echo formatearMoneda($estadisticas['ventas_mes'], $empresa['moneda_principal'] ?? 'CLP'); ?></div>
                    <div class="stat-label">Ventas del Mes</div>
                    <i class="fas fa-chart-line stat-icon"></i>
                </div>
                
                <div class="stat-card info">
                    <div class="stat-value"><?php echo number_format($estadisticas['documentos_emitidos']); ?></div>
                    <div class="stat-label">Documentos Emitidos</div>
                    <i class="fas fa-check-circle stat-icon"></i>
                </div>
            </div>

            <!-- Filtros y Lista de Documentos -->
            <div class="content-card">
                <div class="filters-section">
                    <h5 class="mb-3"><i class="fas fa-filter"></i> Filtros de Búsqueda</h5>
                    <form method="GET" class="row g-3">
                        <div class="col-md-2">
                            <select name="tipo" class="form-control">
                                <option value="">Todos los tipos</option>
                                <option value="factura" <?php echo $filtro_tipo === 'factura' ? 'selected' : ''; ?>>Factura</option>
                                <option value="boleta" <?php echo $filtro_tipo === 'boleta' ? 'selected' : ''; ?>>Boleta</option>
                                <option value="nota_credito" <?php echo $filtro_tipo === 'nota_credito' ? 'selected' : ''; ?>>Nota Crédito</option>
                                <option value="nota_debito" <?php echo $filtro_tipo === 'nota_debito' ? 'selected' : ''; ?>>Nota Débito</option>
                                <option value="guia_despacho" <?php echo $filtro_tipo === 'guia_despacho' ? 'selected' : ''; ?>>Guía Despacho</option>
                            </select>
                        </div>
                        
                        <div class="col-md-2">
                            <select name="estado" class="form-control">
                                <option value="">Todos los estados</option>
                                <option value="borrador" <?php echo $filtro_estado === 'borrador' ? 'selected' : ''; ?>>Borrador</option>
                                <option value="emitido" <?php echo $filtro_estado === 'emitido' ? 'selected' : ''; ?>>Emitido</option>
                                <option value="enviado_sii" <?php echo $filtro_estado === 'enviado_sii' ? 'selected' : ''; ?>>Enviado SII</option>
                                <option value="aceptado_sii" <?php echo $filtro_estado === 'aceptado_sii' ? 'selected' : ''; ?>>Aceptado SII</option>
                                <option value="pagado" <?php echo $filtro_estado === 'pagado' ? 'selected' : ''; ?>>Pagado</option>
                                <option value="anulado" <?php echo $filtro_estado === 'anulado' ? 'selected' : ''; ?>>Anulado</option>
                            </select>
                        </div>
                        
                        <div class="col-md-2">
                            <input type="date" name="fecha_desde" class="form-control" value="<?php echo htmlspecialchars($filtro_fecha_desde); ?>" placeholder="Fecha desde">
                        </div>
                        
                        <div class="col-md-2">
                            <input type="date" name="fecha_hasta" class="form-control" value="<?php echo htmlspecialchars($filtro_fecha_hasta); ?>" placeholder="Fecha hasta">
                        </div>
                        
                        <div class="col-md-3">
                            <input type="text" name="cliente" class="form-control" placeholder="Buscar cliente o RUT..." value="<?php echo htmlspecialchars($filtro_cliente); ?>">
                        </div>
                        
                        <div class="col-md-1">
                            <button type="submit" class="btn btn-primary w-100">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </form>
                </div>

                <?php if (empty($documentos)): ?>
                    <div class="no-documentos">
                        <i class="fas fa-file-invoice-dollar"></i>
                        <h3>No hay documentos registrados</h3>
                        <p class="mb-4">Comience creando su primer documento tributario electrónico</p>
                        <a href="?accion=crear" class="btn btn-primary">
                            <i class="fas fa-plus-circle"></i> Crear Primer Documento
                        </a>
                    </div>
                <?php else: ?>
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th><i class="fas fa-hashtag"></i> Documento</th>
                                    <th><i class="fas fa-tag"></i> Tipo</th>
                                    <th><i class="fas fa-calendar"></i> Fecha</th>
                                    <th><i class="fas fa-user"></i> Cliente</th>
                                    <th><i class="fas fa-dollar-sign"></i> Total</th>
                                    <th><i class="fas fa-info-circle"></i> Estado</th>
                                    <th><i class="fas fa-list"></i> Items</th>
                                    <th><i class="fas fa-cogs"></i> Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($documentos as $doc): ?>
                                    <tr>
                                        <td>
                                            <span class="documento-numero"><?php echo htmlspecialchars($doc['numero_documento']); ?></span>
                                            <br><small class="text-muted"><i class="fas fa-barcode"></i> Folio: <?php echo $doc['folio']; ?></small>
                                        </td>
                                        <td>
                                            <span class="tipo-documento tipo-<?php echo $doc['tipo_documento']; ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $doc['tipo_documento'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <strong><?php echo date('d/m/Y', strtotime($doc['fecha_emision'])); ?></strong>
                                            <?php if ($doc['fecha_vencimiento']): ?>
                                                <br><small class="text-muted">Vence: <?php echo date('d/m/Y', strtotime($doc['fecha_vencimiento'])); ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($doc['razon_social_cliente']); ?></strong>
                                            <br><small class="text-muted"><i class="fas fa-id-card"></i> <?php echo htmlspecialchars($doc['rut_cliente']); ?></small>
                                        </td>
                                        <td>
                                            <strong><?php echo formatearMoneda($doc['total'], $doc['moneda']); ?></strong>
                                            <?php if ($doc['iva'] > 0): ?>
                                                <br><small class="text-muted">IVA: <?php echo formatearMoneda($doc['iva'], $doc['moneda']); ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php 
                                                switch($doc['estado']) {
                                                    case 'emitido': echo 'success'; break;
                                                    case 'enviado_sii': echo 'primary'; break;
                                                    case 'aceptado_sii': echo 'success'; break;
                                                    case 'rechazado_sii': echo 'danger'; break;
                                                    case 'pagado': echo 'success'; break;
                                                    case 'anulado': echo 'danger'; break;
                                                    default: echo 'warning'; break;
                                                }
                                            ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $doc['estado'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary">
                                                <i class="fas fa-boxes"></i> <?php echo $doc['total_items']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div class="btn-group" role="group">
                                                <button class="btn-action btn-view" onclick="verDocumento(<?php echo $doc['id']; ?>)" title="Ver Documento">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                <button class="btn-action btn-print" onclick="imprimirDocumento(<?php echo $doc['id']; ?>)" title="Imprimir">
                                                    <i class="fas fa-print"></i>
                                                </button>
                                                <?php if (!in_array($doc['estado'], ['anulado', 'enviado_sii', 'aceptado_sii'])): ?>
                                                    <a href="?anular=<?php echo $doc['id']; ?>" class="btn-action btn-cancel" onclick="return confirm('¿Está seguro de anular este documento?')" title="Anular">
                                                        <i class="fas fa-ban"></i>
                                                    </a>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>

        <?php elseif ($accion === 'crear'): ?>
            <!-- Formulario de Documento -->
            <div class="content-card">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3>
                        <i class="fas fa-plus-circle"></i>
                        Nuevo Documento Tributario Electrónico
                    </h3>
                    <a href="?accion=listar" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> Volver al Listado
                    </a>
                </div>

                <form method="POST" id="documentoForm">
                    <input type="hidden" name="productos_documento" id="productosDocumento">
                    
                    <div class="row g-3 mb-4">
                        <div class="col-md-3">
                            <div class="form-floating">
                                <select class="form-control" id="tipo_documento" name="tipo_documento" required>
                                    <option value="">Seleccionar tipo...</option>

<!-- DOCUMENTOS DE VENTA -->
<optgroup label="📄 DOCUMENTOS DE VENTA">
    <option value="factura">Factura de Venta (33)</option>
    <option value="factura_exenta">Factura Exenta (34)</option>
    <option value="factura_compra">Factura de Compra (46)</option>
    <option value="boleta">Boleta de Venta (39)</option>
    <option value="boleta_exenta">Boleta Exenta (41)</option>
</optgroup>

<!-- NOTAS DE AJUSTE -->
<optgroup label="📝 NOTAS DE AJUSTE">
    <option value="nota_credito">Nota de Crédito (61)</option>
    <option value="nota_debito">Nota de Débito (56)</option>
    <option value="nota_credito_exenta">Nota Crédito Exenta (60)</option>
    <option value="nota_debito_exenta">Nota Débito Exenta (55)</option>
</optgroup>

<!-- DOCUMENTOS DE DESPACHO -->
<optgroup label="🚚 DOCUMENTOS DE DESPACHO">
    <option value="guia_despacho">Guía de Despacho (52)</option>
    <option value="conocimiento_embarque">Conocimiento de Embarque (50)</option>
    <option value="carta_porte">Carta de Porte (51)</option>
</optgroup>

<!-- LIQUIDACIONES -->
<optgroup label="💰 LIQUIDACIONES">
    <option value="liquidacion_factura">Liquidación de Factura (43)</option>
    <option value="liquidacion_factura_electronica">Liquidación Factura Electrónica (43)</option>
    <option value="liquidacion_comisiones">Liquidación de Comisiones (40)</option>
</optgroup>

<!-- EXPORTACIÓN -->
<optgroup label="🌍 EXPORTACIÓN">
    <option value="factura_exportacion">Factura de Exportación (110)</option>
    <option value="nota_debito_exportacion">Nota Débito Exportación (111)</option>
    <option value="nota_credito_exportacion">Nota Crédito Exportación (112)</option>
    <option value="documento_exportacion">Documento de Exportación (108)</option>
    <option value="factura_territorio_preferencial">Factura Territorio Preferencial (914)</option>
</optgroup>

<!-- DOCUMENTOS ESPECIALES -->
<optgroup label="⭐ DOCUMENTOS ESPECIALES">
    <option value="boleta_honorarios">Boleta de Honorarios (29)</option>
    <option value="factura_electronica_mixta">Factura Electrónica Mixta (35)</option>
    <option value="factura_electronica_afecta">Factura Electrónica Afecta (33)</option>
    <option value="documento_exento">Documento Exento (55)</option>
    <option value="resumen_ventas">Resumen de Ventas Diarias (35)</option>
    <option value="set_pruebas">SET de Pruebas (29)</option>
</optgroup>

<!-- OTROS DOCUMENTOS -->
<optgroup label="📋 OTROS DOCUMENTOS">
    <option value="comprobante_liquidacion">Comprobante de Liquidación (914)</option>
    <option value="comprobante_pago">Comprobante de Pago (914)</option>
</optgroup>
                                </select>
                                <label for="tipo_documento"><i class="fas fa-file-invoice"></i> Tipo Documento *</label>
                            </div>
                        </div>
                        
                        <div class="col-md-3">
                            <div class="form-floating">
                                <input type="date" class="form-control" id="fecha_emision" name="fecha_emision" value="<?php echo date('Y-m-d'); ?>" required>
                                <label for="fecha_emision"><i class="fas fa-calendar"></i> Fecha Emisión *</label>
                            </div>
                        </div>
                        
                        <div class="col-md-3">
                            <div class="form-floating">
                                <input type="date" class="form-control" id="fecha_vencimiento" name="fecha_vencimiento">
                                <label for="fecha_vencimiento"><i class="fas fa-calendar-check"></i> Fecha Vencimiento</label>
                            </div>
                        </div>
                        
                        <div class="col-md-3">
                            <div class="form-floating">
                                <select class="form-control" id="moneda" name="moneda" required>
                                    <?php foreach ($monedas as $moneda_item): ?>
                                        <option value="<?php echo $moneda_item['codigo']; ?>" <?php echo $moneda_item['codigo'] === ($empresa['moneda_principal'] ?? 'CLP') ? 'selected' : ''; ?>>
                                            <?php echo $moneda_item['codigo']; ?> - <?php echo $moneda_item['nombre']; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <label for="moneda"><i class="fas fa-coins"></i> Moneda *</label>
                            </div>
                        </div>
                    </div>
                    
                    <h5><i class="fas fa-user-tie"></i> Datos del Cliente</h5>
                    <div class="row g-3 mb-4">
                        <div class="col-md-3">
                            <div class="form-floating">
                                <input type="text" class="form-control" id="rut_cliente" name="rut_cliente" placeholder="12.345.678-9" required>
                                <label for="rut_cliente"><i class="fas fa-id-card"></i> RUT Cliente *</label>
                            </div>
                        </div>
                        
                        <div class="col-md-9">
                            <div class="form-floating">
                                <input type="text" class="form-control" id="razon_social_cliente" name="razon_social_cliente" placeholder="Razón Social" required>
                                <label for="razon_social_cliente"><i class="fas fa-building"></i> Razón Social / Nombre *</label>
                            </div>
                        </div>
                        
                        <div class="col-md-4">
                            <div class="form-floating">
                                <input type="text" class="form-control" id="direccion_cliente" name="direccion_cliente" placeholder="Dirección">
                                <label for="direccion_cliente"><i class="fas fa-map-marker-alt"></i> Dirección</label>
                            </div>
                        </div>
                        
                        <div class="col-md-2">
                            <div class="form-floating">
                                <input type="text" class="form-control" id="comuna_cliente" name="comuna_cliente" placeholder="Comuna">
                                <label for="comuna_cliente"><i class="fas fa-map"></i> Comuna</label>
                            </div>
                        </div>
                        
                        <div class="col-md-2">
                            <div class="form-floating">
                                <input type="text" class="form-control" id="ciudad_cliente" name="ciudad_cliente" placeholder="Ciudad">
                                <label for="ciudad_cliente"><i class="fas fa-city"></i> Ciudad</label>
                            </div>
                        </div>
                        
                        <div class="col-md-2">
                            <div class="form-floating">
                                <input type="text" class="form-control" id="giro_cliente" name="giro_cliente" placeholder="Giro">
                                <label for="giro_cliente"><i class="fas fa-industry"></i> Giro</label>
                            </div>
                        </div>
                        
                        <div class="col-md-2">
                            <div class="form-floating">
                                <input type="email" class="form-control" id="email_cliente" name="email_cliente" placeholder="email@cliente.com">
                                <label for="email_cliente"><i class="fas fa-envelope"></i> Email</label>
                            </div>
                        </div>
                    </div>
                    
                    <div class="factura-form">
                        <!-- Lista de Productos -->
                        <div>
                            <h5><i class="fas fa-boxes"></i> Productos y Servicios Disponibles</h5>
                            <div class="productos-factura">
                                <?php if (empty($productos)): ?>
                                    <div class="text-center text-muted">
                                        <i class="fas fa-box-open"></i><br>
                                        No hay productos registrados.<br>
                                        <a href="productos.php" class="btn btn-sm btn-primary mt-2">Registrar Productos</a>
                                    </div>
                                <?php else: ?>
                                    <?php foreach ($productos as $producto): ?>
                                        <div class="producto-item" onclick="agregarProductoFactura(<?php echo htmlspecialchars(json_encode($producto), ENT_QUOTES, 'UTF-8'); ?>)">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div>
                                                    <strong><?php echo htmlspecialchars($producto['nombre']); ?></strong>
                                                    <br><small class="text-muted"><i class="fas fa-barcode"></i> <?php echo htmlspecialchars($producto['codigo'] ?? 'N/A'); ?></small>
                                                </div>
                                                <div class="text-end">
                                                    <div class="fw-bold text-success">
                                                        <?php echo formatearMoneda($producto['precio_venta'] ?? 0, $empresa['moneda_principal'] ?? 'CLP'); ?>
                                                    </div>
                                                    <small class="text-muted">
                                                        <i class="fas fa-warehouse"></i> Stock: <?php echo number_format($producto['stock_actual'] ?? 0); ?>
                                                    </small>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Carrito de Facturación -->
                        <div class="carrito-factura">
                            <h5><i class="fas fa-shopping-cart"></i> Productos Seleccionados</h5>
                            <div id="carritoFactura">
                                <div class="text-muted text-center">
                                    <i class="fas fa-cart-plus fa-3x mb-3"></i><br>
                                    <p>Seleccione productos para agregar al documento</p>
                                </div>
                            </div>
                            
                            <div class="totales-factura" id="totalesFactura" style="display: none;">
                                <div class="total-row">
                                    <span><i class="fas fa-calculator"></i> Subtotal:</span>
                                    <span id="subtotalFactura">$0</span>
                                </div>
                                <div class="total-row">
                                    <span><i class="fas fa-percent"></i> IVA (19%):</span>
                                    <span id="ivaFactura">$0</span>
                                </div>
                                <div class="total-row">
                                    <span><i class="fas fa-tag"></i> Exento:</span>
                                    <span id="exentoFactura">$0</span>
                                </div>
                                <div class="total-row total-final">
                                    <span><i class="fas fa-dollar-sign"></i> TOTAL FINAL:</span>
                                    <span id="totalFactura">$0</span>
                                </div>
                            </div>
                            
                            <div class="mt-4">
                                <div class="form-floating mb-3">
                                    <textarea class="form-control" id="observaciones" name="observaciones" placeholder="Observaciones adicionales" style="height: 100px"></textarea>
                                    <label for="observaciones"><i class="fas fa-comment-alt"></i> Observaciones</label>
                                </div>
                                
                                <div class="row g-3 mb-3">
                                    <div class="col-6">
                                        <div class="form-floating">
                                            <select class="form-control" name="condiciones_pago" id="condiciones_pago">
                                                <option value="contado">Contado</option>
                                                <option value="credito_30">Crédito 30 días</option>
                                                <option value="credito_60">Crédito 60 días</option>
                                                <option value="credito_90">Crédito 90 días</option>
                                            </select>
                                            <label for="condiciones_pago"><i class="fas fa-handshake"></i> Condiciones</label>
                                        </div>
                                    </div>
                                    <div class="col-6">
                                        <div class="form-floating">
                                            <select class="form-control" name="medio_pago" id="medio_pago">
                                                <option value="efectivo">Efectivo</option>
                                                <option value="transferencia">Transferencia</option>
                                                <option value="cheque">Cheque</option>
                                                <option value="tarjeta_credito">Tarjeta Crédito</option>
                                                <option value="tarjeta_debito">Tarjeta Débito</option>
                                            </select>
                                            <label for="medio_pago"><i class="fas fa-credit-card"></i> Medio Pago</label>
                                        </div>
                                    </div>
                                </div>
                                
                                <button type="submit" name="guardar_documento" class="btn btn-primary w-100" id="btnGuardarDocumento" disabled>
                                    <i class="fas fa-save"></i> Crear y Emitir Documento DTE
                                </button>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        <?php endif; ?>
    </div>

    <!-- Modal para visualizar documento -->
    <div class="modal fade" id="modalVerDocumento" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="fas fa-file-invoice"></i> Detalle del Documento</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="contenidoDocumento">
                    <div class="text-center">
                        <div class="spinner-border" role="status">
                            <span class="visually-hidden">Cargando...</span>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
                    <button type="button" class="btn btn-primary" onclick="imprimirModalDocumento()">
                        <i class="fas fa-print"></i> Imprimir
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let carritoFactura = [];
        let monedaPrincipal = '<?php echo $empresa['moneda_principal'] ?? 'CLP'; ?>';
        
        // Validación y formateo de RUT chileno
        document.getElementById('rut_cliente')?.addEventListener('input', function(e) {
            let rut = e.target.value.replace(/[^0-9kK]/g, '');
            
            if (rut.length > 1) {
                let numero = rut.slice(0, -1);
                let dv = rut.slice(-1).toUpperCase();
                
                // Formatear número
                if (numero.length > 6) {
                    numero = numero.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
                }
                
                e.target.value = numero + '-' + dv;
            } else {
                e.target.value = rut;
            }
            
            // Validar RUT
            if (rut.length >= 2) {
                if (validarRutClienteJS(rut)) {
                    e.target.classList.remove('is-invalid');
                    e.target.classList.add('is-valid');
                } else {
                    e.target.classList.remove('is-valid');
                    e.target.classList.add('is-invalid');
                }
            }
        });
        
        // Función para validar RUT en JavaScript
        function validarRutClienteJS(rut) {
            rut = rut.replace(/[^0-9kK]/g, '').toUpperCase();
            if (rut.length < 2) return false;
            
            let dv = rut.slice(-1);
            let numero = rut.slice(0, -1);
            
            let suma = 0;
            let multiplicador = 2;
            
            for (let i = numero.length - 1; i >= 0; i--) {
                suma += parseInt(numero[i]) * multiplicador;
                multiplicador = multiplicador === 7 ? 2 : multiplicador + 1;
            }
            
            let resto = suma % 11;
            let dvCalculado = resto < 2 ? resto.toString() : (11 - resto === 10 ? 'K' : (11 - resto).toString());
            
            return dv === dvCalculado;
        }
        
        // Agregar producto al carrito de facturación
        function agregarProductoFactura(producto) {
            if (!producto || !producto.id) {
                alert('Error: Producto inválido');
                return;
            }
            
            const existe = carritoFactura.find(item => item.id === producto.id);
            
            if (existe) {
                existe.cantidad++;
            } else {
                carritoFactura.push({
                    id: producto.id,
                    codigo: producto.codigo || 'N/A',
                    nombre: producto.nombre || 'Producto sin nombre',
                    precio: parseFloat(producto.precio_venta || 0),
                    iva: parseFloat(producto.iva_porcentaje || 19),
                    cantidad: 1,
                    descuento: 0,
                    stock: parseFloat(producto.stock_actual || 0)
                });
            }
            
            actualizarCarritoFactura();
        }
        
        // Actualizar visualización del carrito
        function actualizarCarritoFactura() {
            const carritoDiv = document.getElementById('carritoFactura');
            const totalesDiv = document.getElementById('totalesFactura');
            const btnGuardar = document.getElementById('btnGuardarDocumento');
            
            if (carritoFactura.length === 0) {
                carritoDiv.innerHTML = `
                    <div class="text-muted text-center">
                        <i class="fas fa-cart-plus fa-3x mb-3"></i><br>
                        <p>Seleccione productos para agregar al documento</p>
                    </div>
                `;
                totalesDiv.style.display = 'none';
                btnGuardar.disabled = true;
                return;
            }
            
            let html = '';
            let subtotal = 0;
            let totalIva = 0;
            let totalExento = 0;
            
            carritoFactura.forEach((item, index) => {
                const subtotalItem = item.cantidad * item.precio;
                const descuentoItem = subtotalItem * (item.descuento / 100);
                const netoItem = subtotalItem - descuentoItem;
                
                if (item.iva > 0) {
                    subtotal += netoItem;
                    totalIva += netoItem * (item.iva / 100);
                } else {
                    totalExento += netoItem;
                }
                
                html += `
                    <div class="item-carrito">
                        <div class="d-flex justify-content-between align-items-start mb-2">
                            <div class="flex-grow-1">
                                <strong>${item.nombre}</strong>
                                <br><small class="text-muted">${item.codigo}</small>
                            </div>
                            <button type="button" class="btn btn-sm btn-outline-danger" onclick="eliminarProductoFactura(${index})" title="Eliminar">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                        <div class="row g-2">
                            <div class="col-3">
                                <label class="form-label" style="font-size: 0.75rem;">Cantidad</label>
                                <input type="number" class="form-control form-control-sm" value="${item.cantidad}" min="1" max="${item.stock || 999999}" 
                                       onchange="cambiarCantidadFactura(${index}, this.value)" title="Cantidad">
                            </div>
                            <div class="col-4">
                                <label class="form-label" style="font-size: 0.75rem;">Precio Unit.</label>
                                <input type="number" class="form-control form-control-sm" value="${item.precio}" step="0.01" min="0" 
                                       onchange="cambiarPrecioFactura(${index}, this.value)" title="Precio unitario">
                            </div>
                            <div class="col-3">
                                <label class="form-label" style="font-size: 0.75rem;">Desc. %</label>
                                <input type="number" class="form-control form-control-sm" value="${item.descuento}" min="0" max="100" step="0.01"
                                       onchange="cambiarDescuentoFactura(${index}, this.value)" title="Descuento %">
                            </div>
                            <div class="col-2">
                                <label class="form-label" style="font-size: 0.75rem;">IVA %</label>
                                <select class="form-control form-control-sm" onchange="cambiarIvaFactura(${index}, this.value)">
                                    <option value="19" ${item.iva === 19 ? 'selected' : ''}>19%</option>
                                    <option value="0" ${item.iva === 0 ? 'selected' : ''}>0%</option>
                                </select>
                            </div>
                        </div>
                        <div class="mt-2">
                            <small class="text-success fw-bold">
                                <i class="fas fa-calculator"></i> Subtotal: ${monedaPrincipal} ${netoItem.toLocaleString('es-CL')}
                            </small>
                            ${item.stock ? `<small class="text-muted float-end">Stock: ${item.stock}</small>` : ''}
                        </div>
                    </div>
                `;
            });
            
            carritoDiv.innerHTML = html;
            
            const totalGeneral = subtotal + totalIva + totalExento;
            
            document.getElementById('subtotalFactura').textContent = monedaPrincipal + ' ' + subtotal.toLocaleString('es-CL');
            document.getElementById('ivaFactura').textContent = monedaPrincipal + ' ' + totalIva.toLocaleString('es-CL');
            document.getElementById('exentoFactura').textContent = monedaPrincipal + ' ' + totalExento.toLocaleString('es-CL');
            document.getElementById('totalFactura').textContent = monedaPrincipal + ' ' + totalGeneral.toLocaleString('es-CL');
            
            totalesDiv.style.display = 'block';
            btnGuardar.disabled = false;
            
            // Actualizar campo hidden con datos JSON
            document.getElementById('productosDocumento').value = JSON.stringify(carritoFactura);
        }
        
        // Cambiar cantidad de producto
        function cambiarCantidadFactura(index, cantidad) {
            cantidad = parseFloat(cantidad) || 1;
            if (cantidad > 0) {
                const item = carritoFactura[index];
                if (item.stock && cantidad > item.stock) {
                    alert(`Stock insuficiente. Máximo disponible: ${item.stock}`);
                    cantidad = item.stock;
                }
                carritoFactura[index].cantidad = cantidad;
                actualizarCarritoFactura();
            }
        }
        
        // Cambiar precio de producto
        function cambiarPrecioFactura(index, precio) {
            precio = parseFloat(precio) || 0;
            if (precio >= 0) {
                carritoFactura[index].precio = precio;
                actualizarCarritoFactura();
            }
        }
        
        // Cambiar descuento de producto
        function cambiarDescuentoFactura(index, descuento) {
            descuento = parseFloat(descuento) || 0;
            if (descuento >= 0 && descuento <= 100) {
                carritoFactura[index].descuento = descuento;
                actualizarCarritoFactura();
            }
        }
        
        // Cambiar IVA de producto
        function cambiarIvaFactura(index, iva) {
            iva = parseFloat(iva) || 0;
            carritoFactura[index].iva = iva;
            actualizarCarritoFactura();
        }
        
        // Eliminar producto del carrito
        function eliminarProductoFactura(index) {
            if (confirm('¿Eliminar este producto del documento?')) {
                carritoFactura.splice(index, 1);
                actualizarCarritoFactura();
            }
        }
        
        // Ver documento en modal
        function verDocumento(id) {
            const modal = new bootstrap.Modal(document.getElementById('modalVerDocumento'));
            const contenido = document.getElementById('contenidoDocumento');
            
            contenido.innerHTML = `
                <div class="text-center">
                    <div class="spinner-border" role="status">
                        <span class="visually-hidden">Cargando documento...</span>
                    </div>
                    <p class="mt-2">Cargando detalles del documento...</p>
                </div>
            `;
            
            modal.show();
            
            // Simular carga de documento (en implementación real, hacer AJAX)
            setTimeout(() => {
                contenido.innerHTML = `
                    <div class="alert alert-info">
                        <h5><i class="fas fa-info-circle"></i> Documento ID: ${id}</h5>
                        <p>Funcionalidad de visualización en desarrollo.</p>
                        <p>Aquí se mostraría el detalle completo del documento, incluyendo:</p>
                        <ul>
                            <li>Datos del emisor y receptor</li>
                            <li>Detalle de productos/servicios</li>
                            <li>Totales y cálculos</li>
                            <li>Estado en el SII</li>
                            <li>XML DTE generado</li>
                            <li>Timbre electrónico</li>
                        </ul>
                    </div>
                `;
            }, 1000);
        }
        
        // Imprimir documento
        function imprimirDocumento(id) {
            // En implementación real, abrir ventana de impresión con formato PDF
            alert(`Imprimiendo documento ID: ${id}\n\nEsta funcionalidad abrirá el documento en formato PDF para impresión.`);
        }
        
        // Imprimir desde modal
        function imprimirModalDocumento() {
            window.print();
        }
        
        // Validación del formulario antes de envío
        document.getElementById('documentoForm')?.addEventListener('submit', function(e) {
            const rutCliente = document.getElementById('rut_cliente').value.trim();
            const razonSocial = document.getElementById('razon_social_cliente').value.trim();
            const tipoDocumento = document.getElementById('tipo_documento').value;
            
            if (!tipoDocumento) {
                e.preventDefault();
                alert('Debe seleccionar el tipo de documento');
                document.getElementById('tipo_documento').focus();
                return false;
            }
            
            if (!rutCliente) {
                e.preventDefault();
                alert('Debe ingresar el RUT del cliente');
                document.getElementById('rut_cliente').focus();
                return false;
            }
            
            if (!validarRutClienteJS(rutCliente.replace(/[^0-9kK]/g, ''))) {
                e.preventDefault();
                alert('El RUT del cliente no es válido');
                document.getElementById('rut_cliente').focus();
                return false;
            }
            
            if (!razonSocial) {
                e.preventDefault();
                alert('Debe ingresar la razón social del cliente');
                document.getElementById('razon_social_cliente').focus();
                return false;
            }
            
            if (carritoFactura.length === 0) {
                e.preventDefault();
                alert('Debe agregar al menos un producto o servicio al documento');
                return false;
            }
            
            // Validar que todos los productos tengan cantidad y precio válidos
            for (let i = 0; i < carritoFactura.length; i++) {
                const item = carritoFactura[i];
                if (item.cantidad <= 0) {
                    e.preventDefault();
                    alert(`El producto "${item.nombre}" debe tener una cantidad mayor a cero`);
                    return false;
                }
                if (item.precio < 0) {
                    e.preventDefault();
                    alert(`El producto "${item.nombre}" debe tener un precio válido`);
                    return false;
                }
            }
            
            // Confirmación final
            const totalItems = carritoFactura.reduce((sum, item) => sum + item.cantidad, 0);
            const totalMonto = carritoFactura.reduce((sum, item) => {
                const subtotal = item.cantidad * item.precio * (1 - item.descuento / 100);
                return sum + subtotal + (subtotal * item.iva / 100);
            }, 0);
            
            const confirmMessage = `¿Confirma la creación del documento?
            
Tipo: ${tipoDocumento.toUpperCase()}
Cliente: ${razonSocial}
Items: ${totalItems}
Total: ${monedaPrincipal} ${totalMonto.toLocaleString('es-CL')}

Una vez creado, se generará el XML DTE y se asignará un folio.`;
            
            if (!confirm(confirmMessage)) {
                e.preventDefault();
                return false;
            }
            
            // Mostrar indicador de carga
            const btnSubmit = document.getElementById('btnGuardarDocumento');
            btnSubmit.disabled = true;
            btnSubmit.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creando documento...';
        });
        
        // Auto-completar comuna y ciudad basado en dirección (funcionalidad futura)
        document.getElementById('direccion_cliente')?.addEventListener('blur', function() {
            // Aquí se podría implementar auto-completado basado en APIs de direcciones
        });
        
        // Formatear números en campos de precio
        document.addEventListener('DOMContentLoaded', function() {
            // Establecer fecha de vencimiento automática si se selecciona crédito
            document.getElementById('condiciones_pago')?.addEventListener('change', function(e) {
                const fechaEmision = document.getElementById('fecha_emision').value;
                const fechaVencimiento = document.getElementById('fecha_vencimiento');
                
                if (fechaEmision && e.target.value.includes('credito')) {
                    const dias = parseInt(e.target.value.split('_')[1]) || 30;
                    const fecha = new Date(fechaEmision);
                    fecha.setDate(fecha.getDate() + dias);
                    fechaVencimiento.value = fecha.toISOString().split('T')[0];
                }
            });
        });
        
        // Función para limpiar el carrito
        function limpiarCarrito() {
            if (carritoFactura.length > 0 && confirm('¿Limpiar todos los productos del carrito?')) {
                carritoFactura = [];
                actualizarCarritoFactura();
            }
        }
        
        // Atajos de teclado
        document.addEventListener('keydown', function(e) {
            // F2 para agregar producto rápido
            if (e.key === 'F2') {
                e.preventDefault();
                const codigo = prompt('Ingrese código de producto:');
                if (codigo) {
                    // Buscar producto por código (implementar búsqueda)
                    alert('Función de búsqueda por código en desarrollo');
                }
            }
            
            // F9 para limpiar carrito
            if (e.key === 'F9') {
                e.preventDefault();
                limpiarCarrito();
            }
            
            // ESC para cancelar
            if (e.key === 'Escape' && carritoFactura.length === 0) {
                if (confirm('¿Volver al listado de documentos?')) {
                    window.location.href = '?accion=listar';
                }
            }
        });
        
        console.log('Sistema de Facturación Electrónica SII Chile - SISCONECTA ERP v2.0');
        console.log('Sistema cargado correctamente. Funcionalidades disponibles:');
        console.log('- Creación de documentos DTE');
        console.log('- Validación de RUT chileno');
        console.log('- Cálculo automático de totales');
        console.log('- Gestión de productos en carrito');
        console.log('- Integración con SII (certificados digitales y CAF)');
    </script>
</body>
</html>