<?php
// MÓDULO DE COTIZACIONES - SISCONECTA ERP
// cotizaciones.php

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Verificar sesión activa
if (!isset($_SESSION['user_id']) || !isset($_SESSION['empresa_id'])) {
    header('Location: login.php');
    exit;
}

// Configuración de base de datos
$host = 'localhost';
$dbname = 'conectae_sisconecta';
$username = 'conectae_sisconectauser';
$password = 'pt125824caraud';

$mensaje = '';
$tipo_mensaje = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $empresa_id = $_SESSION['empresa_id'];
    
    // Verificar estado de empresa
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ? AND estado = 'activo'");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch();
    
    if (!$empresa) {
        session_destroy();
        header('Location: login.php?error=empresa_inactiva');
        exit;
    }
    
    // Crear tabla de cotizaciones si no existe
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS cotizaciones (
            id INT AUTO_INCREMENT PRIMARY KEY,
            empresa_id INT NOT NULL,
            numero_cotizacion VARCHAR(50) NOT NULL,
            fecha_cotizacion DATE NOT NULL,
            fecha_vencimiento DATE NOT NULL,
            cliente_rut VARCHAR(20) NOT NULL,
            cliente_razon_social VARCHAR(255) NOT NULL,
            cliente_direccion TEXT,
            cliente_telefono VARCHAR(50),
            cliente_email VARCHAR(100),
            cliente_contacto VARCHAR(100),
            subtotal DECIMAL(15,2) DEFAULT 0.00,
            descuento_global DECIMAL(15,2) DEFAULT 0.00,
            iva DECIMAL(15,2) DEFAULT 0.00,
            total DECIMAL(15,2) NOT NULL,
            moneda VARCHAR(3) DEFAULT 'CLP',
            condiciones_comerciales TEXT,
            forma_pago VARCHAR(100),
            tiempo_entrega VARCHAR(100),
            validez_oferta INT DEFAULT 30,
            observaciones TEXT,
            estado ENUM('borrador', 'enviada', 'aprobada', 'rechazada', 'vencida', 'convertida') DEFAULT 'borrador',
            usuario_creacion INT NOT NULL,
            fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (empresa_id) REFERENCES empresas(id) ON DELETE CASCADE,
            UNIQUE KEY unique_numero (empresa_id, numero_cotizacion),
            INDEX idx_cliente (cliente_rut),
            INDEX idx_fecha (fecha_cotizacion),
            INDEX idx_estado (estado)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    
    // Crear tabla de detalles de cotizaciones
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS cotizacion_detalles (
            id INT AUTO_INCREMENT PRIMARY KEY,
            cotizacion_id INT NOT NULL,
            producto_id INT NULL,
            descripcion VARCHAR(255) NOT NULL,
            cantidad DECIMAL(10,3) NOT NULL,
            precio_unitario DECIMAL(15,2) NOT NULL,
            descuento_porcentaje DECIMAL(5,2) DEFAULT 0.00,
            subtotal DECIMAL(15,2) NOT NULL,
            observaciones TEXT,
            FOREIGN KEY (cotizacion_id) REFERENCES cotizaciones(id) ON DELETE CASCADE,
            INDEX idx_cotizacion (cotizacion_id),
            INDEX idx_producto (producto_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
    
    $accion = $_GET['accion'] ?? 'listar';
    
    // Obtener productos activos
    $stmt = $pdo->prepare("SELECT * FROM productos WHERE empresa_id = ? AND activo = 1 ORDER BY nombre ASC");
    $stmt->execute([$empresa_id]);
    $productos = $stmt->fetchAll();
    
    // Crear nueva cotización
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['crear_cotizacion'])) {
        $cliente_rut = trim($_POST['cliente_rut']);
        $cliente_razon_social = trim($_POST['cliente_razon_social']);
        $cliente_direccion = trim($_POST['cliente_direccion']);
        $cliente_telefono = trim($_POST['cliente_telefono']);
        $cliente_email = trim($_POST['cliente_email']);
        $cliente_contacto = trim($_POST['cliente_contacto']);
        $fecha_cotizacion = $_POST['fecha_cotizacion'];
        $fecha_vencimiento = $_POST['fecha_vencimiento'];
        $condiciones_comerciales = trim($_POST['condiciones_comerciales']);
        $forma_pago = $_POST['forma_pago'];
        $tiempo_entrega = trim($_POST['tiempo_entrega']);
        $validez_oferta = intval($_POST['validez_oferta']);
        $observaciones = trim($_POST['observaciones']);
        $productos_cotizacion = json_decode($_POST['productos_cotizacion'], true);
        
        if (!empty($cliente_rut) && !empty($cliente_razon_social) && !empty($productos_cotizacion)) {
            try {
                $pdo->beginTransaction();
                
                // Generar número de cotización
                $stmt = $pdo->prepare("SELECT MAX(CAST(SUBSTRING(numero_cotizacion, 4) AS UNSIGNED)) as ultimo FROM cotizaciones WHERE empresa_id = ? AND numero_cotizacion LIKE 'COT%'");
                $stmt->execute([$empresa_id]);
                $ultimo_numero = $stmt->fetch()['ultimo'] ?? 0;
                $numero_cotizacion = 'COT' . str_pad($ultimo_numero + 1, 6, '0', STR_PAD_LEFT);
                
                // Calcular totales
                $subtotal = 0;
                $total_iva = 0;
                
                foreach ($productos_cotizacion as $item) {
                    $cantidad = floatval($item['cantidad']);
                    $precio = floatval($item['precio']);
                    $descuento = floatval($item['descuento'] ?? 0);
                    
                    $subtotal_item = $cantidad * $precio;
                    $descuento_item = $subtotal_item * ($descuento / 100);
                    $neto_item = $subtotal_item - $descuento_item;
                    $subtotal += $neto_item;
                }
                
                $total_iva = $subtotal * 0.19; // IVA 19%
                $total_final = $subtotal + $total_iva;
                
                // Insertar cotización
                $stmt = $pdo->prepare("
                    INSERT INTO cotizaciones 
                    (empresa_id, numero_cotizacion, fecha_cotizacion, fecha_vencimiento, cliente_rut, 
                     cliente_razon_social, cliente_direccion, cliente_telefono, cliente_email, cliente_contacto,
                     subtotal, iva, total, condiciones_comerciales, forma_pago, tiempo_entrega, 
                     validez_oferta, observaciones, usuario_creacion, moneda) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $empresa_id, $numero_cotizacion, $fecha_cotizacion, $fecha_vencimiento, $cliente_rut,
                    $cliente_razon_social, $cliente_direccion, $cliente_telefono, $cliente_email, $cliente_contacto,
                    $subtotal, $total_iva, $total_final, $condiciones_comerciales, $forma_pago, $tiempo_entrega,
                    $validez_oferta, $observaciones, $_SESSION['user_id'], $empresa['moneda_principal'] ?? 'CLP'
                ]);
                $cotizacion_id = $pdo->lastInsertId();
                
                // Insertar detalles
                foreach ($productos_cotizacion as $item) {
                    $cantidad = floatval($item['cantidad']);
                    $precio = floatval($item['precio']);
                    $descuento = floatval($item['descuento'] ?? 0);
                    $subtotal_item = ($cantidad * $precio) * (1 - $descuento / 100);
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO cotizacion_detalles 
                        (cotizacion_id, producto_id, descripcion, cantidad, precio_unitario, descuento_porcentaje, subtotal) 
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([
                        $cotizacion_id, $item['id'] ?? null, $item['descripcion'], 
                        $cantidad, $precio, $descuento, $subtotal_item
                    ]);
                }
                
                $pdo->commit();
                
                $mensaje = "Cotización $numero_cotizacion creada exitosamente";
                $tipo_mensaje = 'success';
                $accion = 'listar';
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $mensaje = 'Error al crear cotización: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Complete todos los campos obligatorios y agregue productos';
            $tipo_mensaje = 'warning';
        }
    }
    
    // Cambiar estado de cotización
    if (isset($_GET['cambiar_estado'])) {
        $cotizacion_id = $_GET['cambiar_estado'];
        $nuevo_estado = $_GET['estado'];
        
        try {
            $stmt = $pdo->prepare("UPDATE cotizaciones SET estado = ? WHERE id = ? AND empresa_id = ?");
            $stmt->execute([$nuevo_estado, $cotizacion_id, $empresa_id]);
            
            $mensaje = 'Estado de cotización actualizado';
            $tipo_mensaje = 'success';
        } catch (PDOException $e) {
            $mensaje = 'Error al cambiar estado: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
    
    // Convertir cotización a factura
    if (isset($_GET['convertir_factura'])) {
        $cotizacion_id = $_GET['convertir_factura'];
        
        try {
            $pdo->beginTransaction();
            
            // Obtener datos de la cotización
            $stmt = $pdo->prepare("
                SELECT c.*, cd.* 
                FROM cotizaciones c 
                LEFT JOIN cotizacion_detalles cd ON c.id = cd.cotizacion_id 
                WHERE c.id = ? AND c.empresa_id = ? AND c.estado = 'aprobada'
            ");
            $stmt->execute([$cotizacion_id, $empresa_id]);
            $datos_cotizacion = $stmt->fetchAll();
            
            if (empty($datos_cotizacion)) {
                throw new Exception('Cotización no encontrada o no está aprobada');
            }
            
            $cotizacion = $datos_cotizacion[0];
            
            // Generar folio para factura (simplificado)
            $stmt = $pdo->prepare("SELECT MAX(folio) as ultimo_folio FROM documentos_tributarios WHERE empresa_id = ? AND tipo_documento = 'factura'");
            $stmt->execute([$empresa_id]);
            $ultimo_folio = $stmt->fetch()['ultimo_folio'] ?? 0;
            $nuevo_folio = $ultimo_folio + 1;
            $numero_documento = 'F' . str_pad($nuevo_folio, 8, '0', STR_PAD_LEFT);
            
            // Crear documento tributario
            $stmt = $pdo->prepare("
                INSERT INTO documentos_tributarios 
                (empresa_id, tipo_documento, numero_documento, folio, fecha_emision, rut_cliente, 
                 razon_social_cliente, direccion_cliente, telefono_cliente, email_cliente,
                 subtotal, iva, total, moneda, observaciones, estado) 
                VALUES (?, 'factura', ?, ?, CURDATE(), ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'emitido')
            ");
            $stmt->execute([
                $empresa_id, $numero_documento, $nuevo_folio, $cotizacion['cliente_rut'],
                $cotizacion['cliente_razon_social'], $cotizacion['cliente_direccion'], 
                $cotizacion['cliente_telefono'], $cotizacion['cliente_email'],
                $cotizacion['subtotal'], $cotizacion['iva'], $cotizacion['total'], 
                $cotizacion['moneda'], "Convertida desde cotización {$cotizacion['numero_cotizacion']}"
            ]);
            $documento_id = $pdo->lastInsertId();
            
            // Agregar detalles del documento
            foreach ($datos_cotizacion as $detalle) {
                if (!empty($detalle['descripcion'])) {
                    $stmt = $pdo->prepare("
                        INSERT INTO documento_detalles 
                        (documento_id, producto_id, descripcion, cantidad, precio_unitario, subtotal, porcentaje_iva) 
                        VALUES (?, ?, ?, ?, ?, ?, 19.00)
                    ");
                    $stmt->execute([
                        $documento_id, $detalle['producto_id'], $detalle['descripcion'],
                        $detalle['cantidad'], $detalle['precio_unitario'], $detalle['subtotal']
                    ]);
                }
            }
            
            // Cambiar estado de cotización a convertida
            $stmt = $pdo->prepare("UPDATE cotizaciones SET estado = 'convertida' WHERE id = ?");
            $stmt->execute([$cotizacion_id]);
            
            $pdo->commit();
            
            $mensaje = "Cotización convertida a factura $numero_documento exitosamente";
            $tipo_mensaje = 'success';
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $mensaje = 'Error al convertir cotización: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
    
    // Obtener cotizaciones con filtros
    $filtro_estado = $_GET['estado'] ?? '';
    $filtro_cliente = $_GET['cliente'] ?? '';
    $filtro_fecha_desde = $_GET['fecha_desde'] ?? '';
    $filtro_fecha_hasta = $_GET['fecha_hasta'] ?? '';
    
    $where_conditions = ["c.empresa_id = ?"];
    $params = [$empresa_id];
    
    if (!empty($filtro_estado)) {
        $where_conditions[] = "c.estado = ?";
        $params[] = $filtro_estado;
    }
    
    if (!empty($filtro_cliente)) {
        $where_conditions[] = "(c.cliente_razon_social LIKE ? OR c.cliente_rut LIKE ?)";
        $filtro_cliente_param = '%' . $filtro_cliente . '%';
        $params[] = $filtro_cliente_param;
        $params[] = $filtro_cliente_param;
    }
    
    if (!empty($filtro_fecha_desde)) {
        $where_conditions[] = "c.fecha_cotizacion >= ?";
        $params[] = $filtro_fecha_desde;
    }
    
    if (!empty($filtro_fecha_hasta)) {
        $where_conditions[] = "c.fecha_cotizacion <= ?";
        $params[] = $filtro_fecha_hasta;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Actualizar estados vencidos
    $pdo->exec("UPDATE cotizaciones SET estado = 'vencida' WHERE estado IN ('enviada', 'borrador') AND fecha_vencimiento < CURDATE()");
    
    $stmt = $pdo->prepare("
        SELECT c.*, u.nombre as usuario_nombre,
               COUNT(cd.id) as total_items
        FROM cotizaciones c 
        LEFT JOIN usuarios u ON c.usuario_creacion = u.id
        LEFT JOIN cotizacion_detalles cd ON c.id = cd.cotizacion_id
        WHERE $where_clause 
        GROUP BY c.id
        ORDER BY c.fecha_creacion DESC
        LIMIT 100
    ");
    $stmt->execute($params);
    $cotizaciones = $stmt->fetchAll();
    
    // Estadísticas
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_cotizaciones,
            SUM(CASE WHEN estado = 'aprobada' THEN 1 ELSE 0 END) as aprobadas,
            SUM(CASE WHEN estado = 'aprobada' THEN total ELSE 0 END) as valor_aprobadas,
            SUM(CASE WHEN estado = 'enviada' THEN 1 ELSE 0 END) as pendientes,
            SUM(CASE WHEN MONTH(fecha_cotizacion) = MONTH(CURDATE()) THEN 1 ELSE 0 END) as cotizaciones_mes
        FROM cotizaciones 
        WHERE empresa_id = ?
    ");
    $stmt->execute([$empresa_id]);
    $estadisticas = $stmt->fetch();
    
} catch (PDOException $e) {
    die('Error de conexión: ' . $e->getMessage());
}

// Función para validar RUT chileno
function validarRutChileno($rut) {
    $rut = preg_replace('/[^0-9kK]/', '', strtoupper($rut));
    if (strlen($rut) < 2) return false;
    
    $dv = substr($rut, -1);
    $numero = substr($rut, 0, -1);
    
    $suma = 0;
    $multiplicador = 2;
    
    for ($i = strlen($numero) - 1; $i >= 0; $i--) {
        $suma += intval($numero[$i]) * $multiplicador;
        $multiplicador = $multiplicador == 7 ? 2 : $multiplicador + 1;
    }
    
    $resto = $suma % 11;
    $dv_calculado = $resto < 2 ? $resto : 11 - $resto;
    $dv_calculado = $dv_calculado == 10 ? 'K' : (string)$dv_calculado;
    
    return $dv === $dv_calculado;
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cotizaciones - SISCONECTA ERP</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .main-container {
            padding: 20px;
            max-width: 1600px;
            margin: 0 auto;
        }
        
        .header-section {
            background: linear-gradient(45deg, #2c3e50, #34495e);
            color: white;
            padding: 25px 30px;
            border-radius: 15px;
            margin-bottom: 25px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }
        
        .header-title {
            font-size: 1.9rem;
            font-weight: 700;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
        }
        
        .btn-header {
            background: rgba(255,255,255,0.15);
            border: 2px solid rgba(255,255,255,0.3);
            color: white;
            border-radius: 10px;
            padding: 12px 20px;
            text-decoration: none;
            transition: all 0.3s ease;
            font-weight: 600;
        }
        
        .btn-header:hover {
            background: rgba(255,255,255,0.25);
            color: white;
            transform: translateY(-2px);
        }
        
        .stats-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: linear-gradient(135deg, #ffffff, #f8f9fa);
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border-left: 5px solid;
            transition: all 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.15);
        }
        
        .stat-card.primary { border-left-color: #3498db; }
        .stat-card.success { border-left-color: #27ae60; }
        .stat-card.warning { border-left-color: #f39c12; }
        .stat-card.info { border-left-color: #9b59b6; }
        .stat-card.danger { border-left-color: #e74c3c; }
        
        .stat-value {
            font-size: 2.2rem;
            font-weight: 800;
            color: #2c3e50;
            margin-bottom: 8px;
        }
        
        .stat-label {
            color: #666;
            font-weight: 600;
            font-size: 0.95rem;
        }
        
        .stat-icon {
            float: right;
            font-size: 2.5rem;
            opacity: 0.2;
        }
        
        .content-card {
            background: white;
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            margin-bottom: 25px;
        }
        
        .filters-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 15px;
            margin-bottom: 25px;
        }
        
        .table {
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        
        .table th {
            background: linear-gradient(135deg, #34495e, #2c3e50);
            color: white;
            border: none;
            font-weight: 700;
            padding: 15px 12px;
        }
        
        .table td {
            padding: 15px 12px;
            vertical-align: middle;
            border-top: 1px solid #f1f3f4;
        }
        
        .table tbody tr:hover {
            background: #f8f9fa;
            transform: scale(1.01);
            transition: all 0.2s ease;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 12px 15px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #3498db;
            box-shadow: 0 0 0 0.25rem rgba(52,152,219,0.25);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #3498db, #2980b9);
            border: none;
            border-radius: 10px;
            padding: 12px 25px;
            font-weight: 600;
        }
        
        .badge {
            padding: 8px 12px;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .cotizacion-form {
            display: grid;
            grid-template-columns: 1fr 400px;
            gap: 30px;
            margin-top: 20px;
        }
        
        .productos-cotizacion {
            background: #f8f9fa;
            border-radius: 15px;
            padding: 20px;
            max-height: 500px;
            overflow-y: auto;
        }
        
        .producto-item {
            background: white;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 10px;
            cursor: pointer;
            transition: all 0.3s ease;
            border: 2px solid transparent;
        }
        
        .producto-item:hover {
            border-color: #3498db;
            transform: translateY(-2px);
        }
        
        .carrito-cotizacion {
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 15px;
            padding: 25px;
            position: sticky;
            top: 20px;
        }
        
        .item-carrito {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 12px;
            margin-bottom: 10px;
        }
        
        .totales-cotizacion {
            border-top: 2px solid #3498db;
            padding-top: 15px;
            margin-top: 15px;
        }
        
        .total-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
        }
        
        .total-final {
            font-size: 1.3rem;
            font-weight: 700;
            color: #2c3e50;
            border-top: 1px solid #dee2e6;
            padding-top: 8px;
        }
        
        .estado-badge {
            padding: 6px 12px;
            border-radius: 15px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .estado-borrador { background: #f8f9fa; color: #666; }
        .estado-enviada { background: #fff3cd; color: #856404; }
        .estado-aprobada { background: #d4edda; color: #155724; }
        .estado-rechazada { background: #f8d7da; color: #721c24; }
        .estado-vencida { background: #e2e3e5; color: #495057; }
        .estado-convertida { background: #d1ecf1; color: #0c5460; }
        
        .btn-action {
            padding: 6px 12px;
            border-radius: 6px;
            border: none;
            font-size: 0.8rem;
            font-weight: 500;
            margin: 2px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn-view { background: #3498db; color: white; }
        .btn-approve { background: #27ae60; color: white; }
        .btn-reject { background: #e74c3c; color: white; }
        .btn-convert { background: #9b59b6; color: white; }
        .btn-send { background: #f39c12; color: white; }
        
        .btn-action:hover {
            transform: translateY(-1px);
            box-shadow: 0 3px 10px rgba(0,0,0,0.2);
        }
        
        @media (max-width: 768px) {
            .cotizacion-form {
                grid-template-columns: 1fr;
                gap: 20px;
            }
            
            .stats-cards {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header -->
        <div class="header-section">
            <h1 class="header-title">
                <i class="fas fa-file-contract"></i>
                Módulo de Cotizaciones
                <span class="badge bg-info">PRO</span>
            </h1>
            <div class="header-actions">
                <a href="dashboard.php" class="btn-header">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <?php if ($accion === 'listar'): ?>
                    <a href="?accion=crear" class="btn-header">
                        <i class="fas fa-plus"></i> Nueva Cotización
                    </a>
                <?php endif; ?>
            </div>
        </div>

        <!-- Mensajes -->
        <?php if (!empty($mensaje)): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show">
                <i class="fas fa-<?php echo $tipo_mensaje === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                <?php echo htmlspecialchars($mensaje); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if ($accion === 'listar'): ?>
            <!-- Estadísticas -->
            <div class="stats-cards">
                <div class="stat-card primary">
                    <div class="stat-value"><?php echo number_format($estadisticas['total_cotizaciones'] ?? 0); ?></div>
                    <div class="stat-label">Total Cotizaciones</div>
                    <i class="fas fa-file-contract stat-icon"></i>
                </div>
                
                <div class="stat-card success">
                    <div class="stat-value"><?php echo number_format($estadisticas['aprobadas'] ?? 0); ?></div>
                    <div class="stat-label">Cotizaciones Aprobadas</div>
                    <i class="fas fa-check-circle stat-icon"></i>
                </div>
                
                <div class="stat-card info">
                    <div class="stat-value"><?php echo $empresa['moneda_principal'] ?? 'CLP'; ?> <?php echo number_format($estadisticas['valor_aprobadas'] ?? 0, 0); ?></div>
                    <div class="stat-label">Valor Aprobadas</div>
                    <i class="fas fa-dollar-sign stat-icon"></i>
                </div>
                
                <div class="stat-card warning">
                    <div class="stat-value"><?php echo number_format($estadisticas['pendientes'] ?? 0); ?></div>
                    <div class="stat-label">Pendientes Respuesta</div>
                    <i class="fas fa-clock stat-icon"></i>
                </div>
                
                <div class="stat-card danger">
                    <div class="stat-value"><?php echo number_format($estadisticas['cotizaciones_mes'] ?? 0); ?></div>
                    <div class="stat-label">Cotizaciones del Mes</div>
                    <i class="fas fa-calendar stat-icon"></i>
                </div>
            </div>

            <!-- Filtros y Lista -->
            <div class="content-card">
                <div class="filters-section">
                    <h5><i class="fas fa-filter"></i> Filtros de Búsqueda</h5>
                    <form method="GET" class="row g-3">
                        <div class="col-md-2">
                            <select name="estado" class="form-control">
                                <option value="">Todos los estados</option>
                                <option value="borrador" <?php echo $filtro_estado === 'borrador' ? 'selected' : ''; ?>>Borrador</option>
                                <option value="enviada" <?php echo $filtro_estado === 'enviada' ? 'selected' : ''; ?>>Enviada</option>
                                <option value="aprobada" <?php echo $filtro_estado === 'aprobada' ? 'selected' : ''; ?>>Aprobada</option>
                                <option value="rechazada" <?php echo $filtro_estado === 'rechazada' ? 'selected' : ''; ?>>Rechazada</option>
                                <option value="vencida" <?php echo $filtro_estado === 'vencida' ? 'selected' : ''; ?>>Vencida</option>
                                <option value="convertida" <?php echo $filtro_estado === 'convertida' ? 'selected' : ''; ?>>Convertida</option>
                            </select>
                        </div>
                        
                        <div class="col-md-2">
                            <input type="date" name="fecha_desde" class="form-control" value="<?php echo $filtro_fecha_desde; ?>">
                        </div>
                        
                        <div class="col-md-2">
                            <input type="date" name="fecha_hasta" class="form-control" value="<?php echo $filtro_fecha_hasta; ?>">
                        </div>
                        
                        <div class="col-md-5">
                            <input type="text" name="cliente" class="form-control" placeholder="Buscar cliente..." value="<?php echo htmlspecialchars($filtro_cliente); ?>">
                        </div>
                        
                        <div class="col-md-1">
                            <button type="submit" class="btn btn-primary w-100">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </form>
                </div>

                <?php if (empty($cotizaciones)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-file-contract fa-3x text-muted mb-3"></i>
                        <h4>No hay cotizaciones registradas</h4>
                        <p class="text-muted">Comience creando su primera cotización</p>
                        <a href="?accion=crear" class="btn btn-primary">
                            <i class="fas fa-plus"></i> Crear Primera Cotización
                        </a>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>N° Cotización</th>
                                    <th>Fecha</th>
                                    <th>Cliente</th>
                                    <th>Total</th>
                                    <th>Vencimiento</th>
                                    <th>Estado</th>
                                    <th>Items</th>
                                    <th>Usuario</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($cotizaciones as $cotizacion): ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($cotizacion['numero_cotizacion']); ?></strong></td>
                                        <td><?php echo date('d/m/Y', strtotime($cotizacion['fecha_cotizacion'])); ?></td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($cotizacion['cliente_razon_social']); ?></strong>
                                            <br><small class="text-muted"><?php echo htmlspecialchars($cotizacion['cliente_rut']); ?></small>
                                        </td>
                                        <td><strong><?php echo $cotizacion['moneda']; ?> <?php echo number_format($cotizacion['total'], 0); ?></strong></td>
                                        <td>
                                            <?php 
                                            $vencimiento = new DateTime($cotizacion['fecha_vencimiento']);
                                            $hoy = new DateTime();
                                            $vencido = $vencimiento < $hoy;
                                            ?>
                                            <span class="<?php echo $vencido ? 'text-danger' : ''; ?>">
                                                <?php echo date('d/m/Y', strtotime($cotizacion['fecha_vencimiento'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="estado-badge estado-<?php echo $cotizacion['estado']; ?>">
                                                <?php echo ucfirst($cotizacion['estado']); ?>
                                            </span>
                                        </td>
                                        <td><span class="badge bg-info"><?php echo $cotizacion['total_items']; ?></span></td>
                                        <td><?php echo htmlspecialchars($cotizacion['usuario_nombre'] ?? 'Sistema'); ?></td>
                                        <td>
                                            <div class="btn-group" role="group">
                                                <button class="btn-action btn-view" onclick="verCotizacion(<?php echo $cotizacion['id']; ?>)" title="Ver">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                
                                                <?php if ($cotizacion['estado'] === 'borrador'): ?>
                                                    <a href="?cambiar_estado=<?php echo $cotizacion['id']; ?>&estado=enviada" class="btn-action btn-send" title="Enviar">
                                                        <i class="fas fa-paper-plane"></i>
                                                    </a>
                                                <?php endif; ?>
                                                
                                                <?php if ($cotizacion['estado'] === 'enviada'): ?>
                                                    <a href="?cambiar_estado=<?php echo $cotizacion['id']; ?>&estado=aprobada" class="btn-action btn-approve" title="Aprobar">
                                                        <i class="fas fa-check"></i>
                                                    </a>
                                                    <a href="?cambiar_estado=<?php echo $cotizacion['id']; ?>&estado=rechazada" class="btn-action btn-reject" title="Rechazar">
                                                        <i class="fas fa-times"></i>
                                                    </a>
                                                <?php endif; ?>
                                                
                                                <?php if ($cotizacion['estado'] === 'aprobada'): ?>
                                                    <a href="?convertir_factura=<?php echo $cotizacion['id']; ?>" class="btn-action btn-convert" 
                                                       onclick="return confirm('¿Convertir esta cotización a factura?')" title="Convertir a Factura">
                                                        <i class="fas fa-file-invoice"></i>
                                                    </a>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>

        <?php elseif ($accion === 'crear'): ?>
            <!-- Crear Nueva Cotización -->
            <div class="content-card">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3><i class="fas fa-plus"></i> Nueva Cotización</h3>
                    <a href="?accion=listar" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> Volver al Listado
                    </a>
                </div>

                <form method="POST" id="cotizacionForm">
                    <input type="hidden" name="productos_cotizacion" id="productosCotizacion">
                    
                    <!-- Datos de la Cotización -->
                    <div class="row g-3 mb-4">
                        <div class="col-md-3">
                            <label class="form-label">Fecha Cotización</label>
                            <input type="date" name="fecha_cotizacion" class="form-control" value="<?php echo date('Y-m-d'); ?>" required>
                        </div>
                        
                        <div class="col-md-3">
                            <label class="form-label">Fecha Vencimiento</label>
                            <input type="date" name="fecha_vencimiento" class="form-control" value="<?php echo date('Y-m-d', strtotime('+30 days')); ?>" required>
                        </div>
                        
                        <div class="col-md-3">
                            <label class="form-label">Validez Oferta (días)</label>
                            <input type="number" name="validez_oferta" class="form-control" value="30" min="1" required>
                        </div>
                        
                        <div class="col-md-3">
                            <label class="form-label">Forma de Pago</label>
                            <select name="forma_pago" class="form-control" required>
                                <option value="contado">Contado</option>
                                <option value="credito_30">Crédito 30 días</option>
                                <option value="credito_60">Crédito 60 días</option>
                                <option value="transferencia">Transferencia</option>
                                <option value="cheque">Cheque</option>
                            </select>
                        </div>
                    </div>
                    
                    <!-- Datos del Cliente -->
                    <h5><i class="fas fa-user-tie"></i> Datos del Cliente</h5>
                    <div class="row g-3 mb-4">
                        <div class="col-md-3">
                            <label class="form-label">RUT Cliente *</label>
                            <input type="text" name="cliente_rut" class="form-control" placeholder="12.345.678-9" required>
                        </div>
                        
                        <div class="col-md-9">
                            <label class="form-label">Razón Social *</label>
                            <input type="text" name="cliente_razon_social" class="form-control" placeholder="Razón Social del Cliente" required>
                        </div>
                        
                        <div class="col-md-6">
                            <label class="form-label">Dirección</label>
                            <input type="text" name="cliente_direccion" class="form-control" placeholder="Dirección del Cliente">
                        </div>
                        
                        <div class="col-md-3">
                            <label class="form-label">Teléfono</label>
                            <input type="tel" name="cliente_telefono" class="form-control" placeholder="+56 9 1234 5678">
                        </div>
                        
                        <div class="col-md-3">
                            <label class="form-label">Email</label>
                            <input type="email" name="cliente_email" class="form-control" placeholder="cliente@email.com">
                        </div>
                        
                        <div class="col-md-6">
                            <label class="form-label">Persona de Contacto</label>
                            <input type="text" name="cliente_contacto" class="form-control" placeholder="Nombre del contacto">
                        </div>
                        
                        <div class="col-md-6">
                            <label class="form-label">Tiempo de Entrega</label>
                            <input type="text" name="tiempo_entrega" class="form-control" placeholder="Ej: 15 días hábiles">
                        </div>
                    </div>
                    
                    <div class="cotizacion-form">
                        <!-- Lista de Productos -->
                        <div>
                            <h5><i class="fas fa-boxes"></i> Productos y Servicios</h5>
                            <div class="productos-cotizacion">
                                <?php if (empty($productos)): ?>
                                    <div class="text-center text-muted">
                                        <i class="fas fa-box-open"></i><br>
                                        No hay productos registrados
                                    </div>
                                <?php else: ?>
                                    <?php foreach ($productos as $producto): ?>
                                        <div class="producto-item" onclick="agregarProductoCotizacion(<?php echo htmlspecialchars(json_encode($producto), ENT_QUOTES, 'UTF-8'); ?>)">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <div>
                                                    <strong><?php echo htmlspecialchars($producto['nombre']); ?></strong>
                                                    <br><small class="text-muted"><?php echo htmlspecialchars($producto['codigo'] ?? 'N/A'); ?></small>
                                                </div>
                                                <div class="text-end">
                                                    <div class="fw-bold text-success">
                                                        <?php echo $empresa['moneda_principal'] ?? 'CLP'; ?> <?php echo number_format($producto['precio_venta'] ?? 0, 0); ?>
                                                    </div>
                                                    <small class="text-muted">Stock: <?php echo number_format($producto['stock_actual'] ?? 0); ?></small>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Carrito de Cotización -->
                        <div class="carrito-cotizacion">
                            <h5><i class="fas fa-shopping-cart"></i> Productos Cotizados</h5>
                            <div id="carritoCotizacion">
                                <div class="text-muted text-center">
                                    <i class="fas fa-cart-plus fa-2x mb-3"></i><br>
                                    Agregue productos a la cotización
                                </div>
                            </div>
                            
                            <div class="totales-cotizacion" id="totalesCotizacion" style="display: none;">
                                <div class="total-row">
                                    <span>Subtotal:</span>
                                    <span id="subtotalCotizacion">$0</span>
                                </div>
                                <div class="total-row">
                                    <span>IVA (19%):</span>
                                    <span id="ivaCotizacion">$0</span>
                                </div>
                                <div class="total-row total-final">
                                    <span>TOTAL:</span>
                                    <span id="totalCotizacion">$0</span>
                                </div>
                            </div>
                            
                            <div class="mt-3">
                                <label class="form-label">Condiciones Comerciales</label>
                                <textarea name="condiciones_comerciales" class="form-control mb-3" rows="3" 
                                          placeholder="Condiciones específicas de la cotización"></textarea>
                                
                                <label class="form-label">Observaciones</label>
                                <textarea name="observaciones" class="form-control mb-3" rows="3" 
                                          placeholder="Observaciones adicionales"></textarea>
                                
                                <button type="submit" name="crear_cotizacion" class="btn btn-primary w-100" id="btnCrearCotizacion" disabled>
                                    <i class="fas fa-save"></i> Crear Cotización
                                </button>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let carritoCotizacion = [];
        let monedaPrincipal = '<?php echo $empresa['moneda_principal'] ?? 'CLP'; ?>';
        
        // Validación y formateo de RUT
        document.addEventListener('DOMContentLoaded', function() {
            const rutInput = document.querySelector('input[name="cliente_rut"]');
            if (rutInput) {
                rutInput.addEventListener('input', function(e) {
                    let rut = e.target.value.replace(/[^0-9kK]/g, '');
                    if (rut.length > 1) {
                        let numero = rut.slice(0, -1);
                        let dv = rut.slice(-1).toUpperCase();
                        if (numero.length > 0) {
                            numero = numero.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
                            e.target.value = numero + '-' + dv;
                        }
                    }
                });
            }
        });
        
        // Agregar producto a cotización
        function agregarProductoCotizacion(producto) {
            if (!producto || !producto.id) {
                alert('Error: Producto inválido');
                return;
            }
            
            const existe = carritoCotizacion.find(item => item.id === producto.id);
            
            if (existe) {
                existe.cantidad++;
            } else {
                carritoCotizacion.push({
                    id: producto.id,
                    codigo: producto.codigo || 'N/A',
                    descripcion: producto.nombre || 'Producto sin nombre',
                    precio: parseFloat(producto.precio_venta || 0),
                    cantidad: 1,
                    descuento: 0
                });
            }
            
            actualizarCarritoCotizacion();
        }
        
        // Actualizar carrito de cotización
        function actualizarCarritoCotizacion() {
            const carritoDiv = document.getElementById('carritoCotizacion');
            const totalesDiv = document.getElementById('totalesCotizacion');
            const btnCrear = document.getElementById('btnCrearCotizacion');
            
            if (carritoCotizacion.length === 0) {
                carritoDiv.innerHTML = `
                    <div class="text-muted text-center">
                        <i class="fas fa-cart-plus fa-2x mb-3"></i><br>
                        Agregue productos a la cotización
                    </div>
                `;
                totalesDiv.style.display = 'none';
                if (btnCrear) btnCrear.disabled = true;
                return;
            }
            
            let html = '';
            let subtotal = 0;
            
            carritoCotizacion.forEach((item, index) => {
                const subtotalItem = item.cantidad * item.precio * (1 - item.descuento / 100);
                subtotal += subtotalItem;
                
                html += `
                    <div class="item-carrito">
                        <div class="d-flex justify-content-between align-items-start mb-2">
                            <strong>${item.descripcion}</strong>
                            <button type="button" class="btn btn-sm btn-outline-danger" onclick="eliminarProductoCotizacion(${index})">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                        <div class="row g-2">
                            <div class="col-4">
                                <input type="number" class="form-control form-control-sm" value="${item.cantidad}" min="1" 
                                       onchange="cambiarCantidadCotizacion(${index}, this.value)">
                            </div>
                            <div class="col-4">
                                <input type="number" class="form-control form-control-sm" value="${item.precio}" step="0.01" 
                                       onchange="cambiarPrecioCotizacion(${index}, this.value)">
                            </div>
                            <div class="col-4">
                                <input type="number" class="form-control form-control-sm" value="${item.descuento}" min="0" max="100" 
                                       onchange="cambiarDescuentoCotizacion(${index}, this.value)" placeholder="Desc %">
                            </div>
                        </div>
                        <small class="text-muted">Subtotal: ${monedaPrincipal} ${subtotalItem.toLocaleString()}</small>
                    </div>
                `;
            });
            
            carritoDiv.innerHTML = html;
            
            const iva = subtotal * 0.19;
            const total = subtotal + iva;
            
            document.getElementById('subtotalCotizacion').textContent = monedaPrincipal + ' ' + subtotal.toLocaleString();
            document.getElementById('ivaCotizacion').textContent = monedaPrincipal + ' ' + iva.toLocaleString();
            document.getElementById('totalCotizacion').textContent = monedaPrincipal + ' ' + total.toLocaleString();
            
            totalesDiv.style.display = 'block';
            if (btnCrear) btnCrear.disabled = false;
            
            document.getElementById('productosCotizacion').value = JSON.stringify(carritoCotizacion);
        }
        
        // Cambiar cantidad
        function cambiarCantidadCotizacion(index, cantidad) {
            cantidad = parseInt(cantidad) || 1;
            if (cantidad > 0) {
                carritoCotizacion[index].cantidad = cantidad;
                actualizarCarritoCotizacion();
            }
        }
        
        // Cambiar precio
        function cambiarPrecioCotizacion(index, precio) {
            precio = parseFloat(precio) || 0;
            if (precio >= 0) {
                carritoCotizacion[index].precio = precio;
                actualizarCarritoCotizacion();
            }
        }
        
        // Cambiar descuento
        function cambiarDescuentoCotizacion(index, descuento) {
            descuento = parseFloat(descuento) || 0;
            if (descuento >= 0 && descuento <= 100) {
                carritoCotizacion[index].descuento = descuento;
                actualizarCarritoCotizacion();
            }
        }
        
        // Eliminar producto
        function eliminarProductoCotizacion(index) {
            carritoCotizacion.splice(index, 1);
            actualizarCarritoCotizacion();
        }
        
        // Ver cotización
        function verCotizacion(id) {
            alert('Funcionalidad de vista de cotización en desarrollo. ID: ' + id);
        }
        
        // Validación del formulario
        document.getElementById('cotizacionForm')?.addEventListener('submit', function(e) {
            if (carritoCotizacion.length === 0) {
                e.preventDefault();
                alert('Debe agregar al menos un producto a la cotización');
                return false;
            }
        });
        
        console.log('Módulo de Cotizaciones - SISCONECTA ERP');
        console.log('Sistema de gestión de cotizaciones profesional');
    </script>
</body>
</html>