<?php
// MÓDULO DE CUENTAS POR COBRAR - SISCONECTA ERP
// cuentas_por_cobrar.php

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Verificar sesión activa
if (!isset($_SESSION['user_id']) || !isset($_SESSION['empresa_id'])) {
    header('Location: login.php');
    exit;
}

// Configuración de base de datos
$host = 'localhost';
$dbname = 'conectae_sisconecta';
$username = 'conectae_sisconectauser';
$password = 'pt125824caraud';

$mensaje = '';
$tipo_mensaje = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $empresa_id = $_SESSION['empresa_id'];
    
    // Verificar estado de empresa
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ? AND estado = 'activo'");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch();
    
    if (!$empresa) {
        session_destroy();
        header('Location: login.php?error=empresa_inactiva');
        exit;
    }
    
    // Verificar vencimiento (excepto AUDITOREX CHILE)
    if (($empresa['codigo_empresa'] ?? '') !== 'AUDITOREX_CHILE') {
        $fecha_actual = new DateTime();
        $fecha_vencimiento = new DateTime($empresa['fecha_vencimiento'] ?? '2024-12-31');
        
        if ($fecha_actual > $fecha_vencimiento) {
            session_destroy();
            header('Location: login.php?error=suscripcion_vencida');
            exit;
        }
    }
    
    // Crear tabla de pagos si no existe
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS pagos_recibidos (
            id INT AUTO_INCREMENT PRIMARY KEY,
            empresa_id INT NOT NULL,
            documento_id INT NOT NULL,
            monto_pago DECIMAL(15,2) NOT NULL,
            fecha_pago DATE NOT NULL,
            metodo_pago ENUM('efectivo', 'transferencia', 'cheque', 'tarjeta_credito', 'tarjeta_debito') NOT NULL,
            numero_referencia VARCHAR(100),
            observaciones TEXT,
            fecha_registro TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            usuario_registro INT,
            FOREIGN KEY (empresa_id) REFERENCES empresas(id) ON DELETE CASCADE,
            FOREIGN KEY (documento_id) REFERENCES documentos_tributarios(id) ON DELETE CASCADE,
            FOREIGN KEY (usuario_registro) REFERENCES usuarios(id),
            INDEX idx_empresa_fecha (empresa_id, fecha_pago),
            INDEX idx_documento (documento_id)
        ) ENGINE=InnoDB
    ");
    
    $accion = $_GET['accion'] ?? 'dashboard';
    
    // Registrar pago
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['registrar_pago'])) {
        $documento_id = $_POST['documento_id'];
        $monto_pago = floatval($_POST['monto_pago']);
        $fecha_pago = $_POST['fecha_pago'];
        $metodo_pago = $_POST['metodo_pago'];
        $numero_referencia = trim($_POST['numero_referencia'] ?? '');
        $observaciones = trim($_POST['observaciones'] ?? '');
        
        if ($documento_id && $monto_pago > 0 && !empty($fecha_pago) && !empty($metodo_pago)) {
            try {
                $pdo->beginTransaction();
                
                // Obtener datos del documento
                $stmt = $pdo->prepare("SELECT * FROM documentos_tributarios WHERE id = ? AND empresa_id = ?");
                $stmt->execute([$documento_id, $empresa_id]);
                $documento = $stmt->fetch();
                
                if (!$documento) {
                    throw new Exception('Documento no encontrado');
                }
                
                // Calcular total pagado hasta ahora
                $stmt = $pdo->prepare("SELECT COALESCE(SUM(monto_pago), 0) as total_pagado FROM pagos_recibidos WHERE documento_id = ?");
                $stmt->execute([$documento_id]);
                $total_pagado = $stmt->fetch()['total_pagado'];
                
                // Validar que no se exceda el monto del documento
                if (($total_pagado + $monto_pago) > $documento['total']) {
                    throw new Exception('El monto del pago excede el saldo pendiente del documento');
                }
                
                // Registrar el pago
                $stmt = $pdo->prepare("
                    INSERT INTO pagos_recibidos 
                    (empresa_id, documento_id, monto_pago, fecha_pago, metodo_pago, numero_referencia, observaciones, usuario_registro) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$empresa_id, $documento_id, $monto_pago, $fecha_pago, $metodo_pago, $numero_referencia, $observaciones, $_SESSION['user_id']]);
                
                // Actualizar estado del documento si está completamente pagado
                $nuevo_total_pagado = $total_pagado + $monto_pago;
                if ($nuevo_total_pagado >= $documento['total']) {
                    $stmt = $pdo->prepare("UPDATE documentos_tributarios SET estado = 'pagado' WHERE id = ?");
                    $stmt->execute([$documento_id]);
                }
                
                // Crear asiento contable del pago
                $stmt = $pdo->prepare("SELECT COALESCE(MAX(numero_asiento), 0) + 1 as siguiente_asiento FROM asientos_contables WHERE empresa_id = ?");
                $stmt->execute([$empresa_id]);
                $numero_asiento = $stmt->fetch()['siguiente_asiento'];
                
                $descripcion = "Pago recibido - " . $documento['tipo_documento'] . " " . $documento['numero_documento'] . " - " . $documento['razon_social_cliente'];
                
                $stmt = $pdo->prepare("
                    INSERT INTO asientos_contables 
                    (empresa_id, numero_asiento, fecha, descripcion, tipo_asiento, estado, usuario_registro) 
                    VALUES (?, ?, ?, ?, 'automatico', 'contabilizado', ?)
                ");
                $stmt->execute([$empresa_id, $numero_asiento, $fecha_pago, $descripcion, $_SESSION['user_id']]);
                $asiento_id = $pdo->lastInsertId();
                
                // DEBE: Caja/Banco (según método de pago)
                $cuenta_caja = $metodo_pago === 'efectivo' ? '110101' : '110102';
                $nombre_cuenta = $metodo_pago === 'efectivo' ? 'CAJA' : 'BANCOS';
                
                $stmt = $pdo->prepare("
                    INSERT INTO movimientos_contables 
                    (asiento_id, cuenta_contable, nombre_cuenta, debe, haber, observaciones) 
                    VALUES (?, ?, ?, ?, 0.00, ?)
                ");
                $stmt->execute([$asiento_id, $cuenta_caja, $nombre_cuenta, $monto_pago, "Pago recibido - {$metodo_pago}"]);
                
                // HABER: Clientes por Cobrar
                $stmt = $pdo->prepare("
                    INSERT INTO movimientos_contables 
                    (asiento_id, cuenta_contable, nombre_cuenta, debe, haber, observaciones) 
                    VALUES (?, '110201', 'CLIENTES POR COBRAR', 0.00, ?, ?)
                ");
                $stmt->execute([$asiento_id, $monto_pago, "Cobro de " . $documento['razon_social_cliente']]);
                
                $pdo->commit();
                
                $mensaje = "Pago registrado exitosamente por $" . number_format($monto_pago, 2);
                $tipo_mensaje = 'success';
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $mensaje = 'Error al registrar pago: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Complete todos los campos obligatorios';
            $tipo_mensaje = 'warning';
        }
    }
    
    // Obtener facturas por cobrar
    $filtro_estado = $_GET['estado'] ?? '';
    $filtro_cliente = $_GET['cliente'] ?? '';
    $filtro_vencimiento = $_GET['vencimiento'] ?? '';
    
    $where_conditions = ["dt.empresa_id = ?", "dt.tipo_documento IN ('factura', 'boleta')", "dt.estado != 'anulado'"];
    $params = [$empresa_id];
    
    if (!empty($filtro_estado)) {
        if ($filtro_estado === 'pendiente') {
            $where_conditions[] = "dt.estado != 'pagado'";
        } else {
            $where_conditions[] = "dt.estado = ?";
            $params[] = $filtro_estado;
        }
    }
    
    if (!empty($filtro_cliente)) {
        $where_conditions[] = "(dt.razon_social_cliente LIKE ? OR dt.rut_cliente LIKE ?)";
        $filtro_cliente_param = '%' . $filtro_cliente . '%';
        $params[] = $filtro_cliente_param;
        $params[] = $filtro_cliente_param;
    }
    
    if (!empty($filtro_vencimiento)) {
        switch ($filtro_vencimiento) {
            case 'vencidas':
                $where_conditions[] = "dt.fecha_vencimiento < CURDATE()";
                break;
            case 'por_vencer':
                $where_conditions[] = "dt.fecha_vencimiento BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)";
                break;
            case 'vigentes':
                $where_conditions[] = "dt.fecha_vencimiento >= CURDATE()";
                break;
        }
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    $stmt = $pdo->prepare("
        SELECT dt.*,
               COALESCE(SUM(pr.monto_pago), 0) as total_pagado,
               (dt.total - COALESCE(SUM(pr.monto_pago), 0)) as saldo_pendiente,
               CASE 
                   WHEN dt.fecha_vencimiento < CURDATE() AND dt.estado != 'pagado' THEN 'vencida'
                   WHEN dt.fecha_vencimiento BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY) AND dt.estado != 'pagado' THEN 'por_vencer'
                   WHEN dt.estado = 'pagado' THEN 'pagada'
                   ELSE 'vigente'
               END as estado_cobranza,
               DATEDIFF(CURDATE(), dt.fecha_vencimiento) as dias_vencimiento
        FROM documentos_tributarios dt
        LEFT JOIN pagos_recibidos pr ON dt.id = pr.documento_id
        WHERE $where_clause
        GROUP BY dt.id
        HAVING saldo_pendiente > 0 OR dt.estado = 'pagado'
        ORDER BY dt.fecha_vencimiento ASC, dt.fecha_emision DESC
        LIMIT 100
    ");
    $stmt->execute($params);
    $documentos_cobrar = $stmt->fetchAll();
    
    // Estadísticas de cuentas por cobrar
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(DISTINCT dt.id) as total_documentos,
            SUM(dt.total - COALESCE(pr.total_pagado, 0)) as total_por_cobrar,
            SUM(CASE WHEN dt.fecha_vencimiento < CURDATE() AND dt.estado != 'pagado' THEN (dt.total - COALESCE(pr.total_pagado, 0)) ELSE 0 END) as total_vencido,
            COUNT(CASE WHEN dt.fecha_vencimiento < CURDATE() AND dt.estado != 'pagado' THEN 1 END) as documentos_vencidos,
            COUNT(CASE WHEN dt.estado = 'pagado' THEN 1 END) as documentos_pagados
        FROM documentos_tributarios dt
        LEFT JOIN (
            SELECT documento_id, SUM(monto_pago) as total_pagado 
            FROM pagos_recibidos 
            GROUP BY documento_id
        ) pr ON dt.id = pr.documento_id
        WHERE dt.empresa_id = ? 
        AND dt.tipo_documento IN ('factura', 'boleta')
        AND dt.estado != 'anulado'
        AND (dt.total - COALESCE(pr.total_pagado, 0) > 0 OR dt.estado = 'pagado')
    ");
    $stmt->execute([$empresa_id]);
    $estadisticas = $stmt->fetch();
    
    // Obtener histórico de pagos recientes
    $stmt = $pdo->prepare("
        SELECT pr.*, dt.numero_documento, dt.razon_social_cliente, dt.total as total_documento,
               u.nombre as usuario_nombre
        FROM pagos_recibidos pr
        JOIN documentos_tributarios dt ON pr.documento_id = dt.id
        LEFT JOIN usuarios u ON pr.usuario_registro = u.id
        WHERE pr.empresa_id = ?
        ORDER BY pr.fecha_registro DESC
        LIMIT 20
    ");
    $stmt->execute([$empresa_id]);
    $pagos_recientes = $stmt->fetchAll();
    
    // Antigüedad de saldos
    if ($accion === 'antiguedad') {
        $stmt = $pdo->prepare("
            SELECT 
                dt.rut_cliente,
                dt.razon_social_cliente,
                SUM(CASE WHEN DATEDIFF(CURDATE(), dt.fecha_vencimiento) <= 0 THEN (dt.total - COALESCE(pr.total_pagado, 0)) ELSE 0 END) as vigente,
                SUM(CASE WHEN DATEDIFF(CURDATE(), dt.fecha_vencimiento) BETWEEN 1 AND 30 THEN (dt.total - COALESCE(pr.total_pagado, 0)) ELSE 0 END) as vencido_30,
                SUM(CASE WHEN DATEDIFF(CURDATE(), dt.fecha_vencimiento) BETWEEN 31 AND 60 THEN (dt.total - COALESCE(pr.total_pagado, 0)) ELSE 0 END) as vencido_60,
                SUM(CASE WHEN DATEDIFF(CURDATE(), dt.fecha_vencimiento) BETWEEN 61 AND 90 THEN (dt.total - COALESCE(pr.total_pagado, 0)) ELSE 0 END) as vencido_90,
                SUM(CASE WHEN DATEDIFF(CURDATE(), dt.fecha_vencimiento) > 90 THEN (dt.total - COALESCE(pr.total_pagado, 0)) ELSE 0 END) as vencido_mas_90,
                SUM(dt.total - COALESCE(pr.total_pagado, 0)) as total_saldo
            FROM documentos_tributarios dt
            LEFT JOIN (
                SELECT documento_id, SUM(monto_pago) as total_pagado 
                FROM pagos_recibidos 
                GROUP BY documento_id
            ) pr ON dt.id = pr.documento_id
            WHERE dt.empresa_id = ? 
            AND dt.tipo_documento IN ('factura', 'boleta')
            AND dt.estado != 'anulado'
            AND (dt.total - COALESCE(pr.total_pagado, 0)) > 0
            GROUP BY dt.rut_cliente, dt.razon_social_cliente
            HAVING total_saldo > 0
            ORDER BY total_saldo DESC
        ");
        $stmt->execute([$empresa_id]);
        $antiguedad_saldos = $stmt->fetchAll();
    }
    
} catch (PDOException $e) {
    die('Error de conexión: ' . $e->getMessage());
}

function formatearMoneda($valor, $moneda = 'CLP') {
    return $moneda . ' ' . number_format((float)$valor, 0, ',', '.');
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cuentas por Cobrar - SISCONECTA ERP</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .main-container {
            padding: 20px;
            max-width: 1600px;
            margin: 0 auto;
        }
        
        .header-section {
            background: linear-gradient(45deg, #27ae60, #2ecc71);
            color: white;
            padding: 25px 30px;
            border-radius: 15px;
            margin-bottom: 25px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 10px 30px rgba(39,174,96,0.3);
        }
        
        .header-title {
            font-size: 1.9rem;
            font-weight: 700;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
        }
        
        .btn-header {
            background: rgba(255,255,255,0.15);
            border: 2px solid rgba(255,255,255,0.3);
            color: white;
            border-radius: 10px;
            padding: 12px 20px;
            text-decoration: none;
            transition: all 0.3s ease;
            font-weight: 600;
        }
        
        .btn-header:hover {
            background: rgba(255,255,255,0.25);
            color: white;
            transform: translateY(-2px);
        }
        
        .stats-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            border-left: 5px solid;
            transition: all 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.15);
        }
        
        .stat-card.success { border-left-color: #27ae60; }
        .stat-card.warning { border-left-color: #f39c12; }
        .stat-card.danger { border-left-color: #e74c3c; }
        .stat-card.primary { border-left-color: #3498db; }
        
        .stat-value {
            font-size: 2.2rem;
            font-weight: 800;
            color: #2c3e50;
            margin-bottom: 8px;
        }
        
        .stat-label {
            color: #666;
            font-weight: 600;
            font-size: 0.95rem;
        }
        
        .stat-icon {
            float: right;
            font-size: 2.5rem;
            opacity: 0.2;
        }
        
        .content-card {
            background: white;
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            margin-bottom: 25px;
        }
        
        .nav-pills .nav-link {
            border-radius: 10px;
            font-weight: 600;
            margin-right: 10px;
            padding: 12px 20px;
        }
        
        .nav-pills .nav-link.active {
            background: linear-gradient(135deg, #27ae60, #2ecc71);
        }
        
        .table {
            border-radius: 15px;
            overflow: hidden;
        }
        
        .table th {
            background: linear-gradient(135deg, #34495e, #2c3e50);
            color: white;
            border: none;
            font-weight: 700;
            padding: 15px 12px;
        }
        
        .table td {
            padding: 15px 12px;
            vertical-align: middle;
            border-top: 1px solid #f1f3f4;
        }
        
        .table tbody tr:hover {
            background: #f8f9fa;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 12px 15px;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #27ae60, #2ecc71);
            border: none;
            border-radius: 10px;
            padding: 12px 25px;
            font-weight: 600;
        }
        
        .badge {
            padding: 8px 12px;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .filters-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 15px;
            margin-bottom: 25px;
        }
        
        .documento-vencido { background-color: #ffeaea !important; }
        .documento-por-vencer { background-color: #fff3cd !important; }
        .documento-pagado { background-color: #d4edda !important; }
        
        .pago-item {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 10px;
            border-left: 4px solid #27ae60;
        }
        
        .antiguedad-table th {
            background: linear-gradient(135deg, #27ae60, #2ecc71);
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header -->
        <div class="header-section">
            <h1 class="header-title">
                <i class="fas fa-hand-holding-usd"></i>
                Cuentas por Cobrar
            </h1>
            <div class="header-actions">
                <a href="dashboard.php" class="btn-header">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <a href="contabilidad.php" class="btn-header">
                    <i class="fas fa-calculator"></i> Contabilidad
                </a>
            </div>
        </div>

        <!-- Mensajes -->
        <?php if (!empty($mensaje)): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show">
                <i class="fas fa-<?php echo $tipo_mensaje === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                <?php echo htmlspecialchars($mensaje); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if ($accion === 'dashboard'): ?>
            <!-- Estadísticas -->
            <div class="stats-cards">
                <div class="stat-card success">
                    <div class="stat-value"><?php echo formatearMoneda($estadisticas['total_por_cobrar'] ?? 0, $empresa['moneda_principal'] ?? 'CLP'); ?></div>
                    <div class="stat-label">Total por Cobrar</div>
                    <i class="fas fa-coins stat-icon"></i>
                </div>
                
                <div class="stat-card danger">
                    <div class="stat-value"><?php echo formatearMoneda($estadisticas['total_vencido'] ?? 0, $empresa['moneda_principal'] ?? 'CLP'); ?></div>
                    <div class="stat-label">Total Vencido</div>
                    <i class="fas fa-exclamation-triangle stat-icon"></i>
                </div>
                
                <div class="stat-card warning">
                    <div class="stat-value"><?php echo number_format($estadisticas['documentos_vencidos'] ?? 0); ?></div>
                    <div class="stat-label">Documentos Vencidos</div>
                    <i class="fas fa-clock stat-icon"></i>
                </div>
                
                <div class="stat-card primary">
                    <div class="stat-value"><?php echo number_format($estadisticas['documentos_pagados'] ?? 0); ?></div>
                    <div class="stat-label">Documentos Pagados</div>
                    <i class="fas fa-check-circle stat-icon"></i>
                </div>
            </div>

            <!-- Navegación -->
            <div class="content-card">
                <ul class="nav nav-pills mb-4">
                    <li class="nav-item">
                        <a class="nav-link active" href="?accion=dashboard">
                            <i class="fas fa-tachometer-alt"></i> Dashboard
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="?accion=documentos">
                            <i class="fas fa-file-invoice"></i> Documentos por Cobrar
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="?accion=pagos">
                            <i class="fas fa-money-bill-wave"></i> Registrar Pago
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="?accion=antiguedad">
                            <i class="fas fa-chart-bar"></i> Antigüedad de Saldos
                        </a>
                    </li>
                </ul>

                <!-- Pagos Recientes -->
                <h5><i class="fas fa-history"></i> Últimos Pagos Recibidos</h5>
                
                <?php if (empty($pagos_recientes)): ?>
                    <div class="text-center text-muted py-4">
                        <i class="fas fa-money-bill-wave fa-3x mb-3"></i>
                        <h6>No hay pagos registrados</h6>
                        <p>Los pagos recibidos aparecerán aquí</p>
                    </div>
                <?php else: ?>
                    <?php foreach (array_slice($pagos_recientes, 0, 5) as $pago): ?>
                        <div class="pago-item">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <strong><?php echo htmlspecialchars($pago['razon_social_cliente']); ?></strong>
                                    <span class="badge bg-info"><?php echo htmlspecialchars($pago['numero_documento']); ?></span>
                                    <br>
                                    <small class="text-muted">
                                        <?php echo date('d/m/Y', strtotime($pago['fecha_pago'])); ?> - 
                                        <?php echo ucfirst($pago['metodo_pago']); ?>
                                        <?php if ($pago['numero_referencia']): ?>
                                            - Ref: <?php echo htmlspecialchars($pago['numero_referencia']); ?>
                                        <?php endif; ?>
                                    </small>
                                </div>
                                <div class="text-end">
                                    <div class="h5 text-success mb-0">
                                        <?php echo formatearMoneda($pago['monto_pago'], $empresa['moneda_principal'] ?? 'CLP'); ?>
                                    </div>
                                    <small class="text-muted">Por: <?php echo htmlspecialchars($pago['usuario_nombre'] ?? 'Sistema'); ?></small>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    
                    <div class="text-center mt-3">
                        <a href="?accion=pagos" class="btn btn-primary">
                            <i class="fas fa-plus"></i> Registrar Nuevo Pago
                        </a>
                    </div>
                <?php endif; ?>
            </div>

        <?php elseif ($accion === 'documentos'): ?>
            <!-- Lista de Documentos por Cobrar -->
            <div class="content-card">
                <h3><i class="fas fa-file-invoice"></i> Documentos por Cobrar</h3>
                
                <!-- Filtros -->
                <div class="filters-section">
                    <form method="GET" class="row g-3">
                        <input type="hidden" name="accion" value="documentos">
                        <div class="col-md-3">
                            <select name="estado" class="form-control">
                                <option value="">Todos los estados</option>
                                <option value="pendiente" <?php echo $filtro_estado === 'pendiente' ? 'selected' : ''; ?>>Pendientes</option>
                                <option value="pagado" <?php echo $filtro_estado === 'pagado' ? 'selected' : ''; ?>>Pagados</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <select name="vencimiento" class="form-control">
                                <option value="">Todos los vencimientos</option>
                                <option value="vencidas" <?php echo $filtro_vencimiento === 'vencidas' ? 'selected' : ''; ?>>Vencidas</option>
                                <option value="por_vencer" <?php echo $filtro_vencimiento === 'por_vencer' ? 'selected' : ''; ?>>Por vencer (7 días)</option>
                                <option value="vigentes" <?php echo $filtro_vencimiento === 'vigentes' ? 'selected' : ''; ?>>Vigentes</option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <input type="text" name="cliente" class="form-control" placeholder="Buscar cliente..." value="<?php echo htmlspecialchars($filtro_cliente); ?>">
                        </div>
                        <div class="col-md-2">
                            <button type="submit" class="btn btn-primary w-100">
                                <i class="fas fa-filter"></i> Filtrar
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Tabla de documentos -->
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Documento</th>
                                <th>Fecha Emisión</th>
                                <th>Fecha Venc.</th>
                                <th>Cliente</th>
                                <th>Total</th>
                                <th>Pagado</th>
                                <th>Saldo</th>
                                <th>Estado</th>
                                <th>Días</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($documentos_cobrar as $doc): ?>
                                <tr class="<?php 
                                    switch($doc['estado_cobranza']) {
                                        case 'vencida': echo 'documento-vencido'; break;
                                        case 'por_vencer': echo 'documento-por-vencer'; break;
                                        case 'pagada': echo 'documento-pagado'; break;
                                    }
                                ?>">
                                    <td>
                                        <strong><?php echo htmlspecialchars($doc['numero_documento']); ?></strong>
                                        <br><small><?php echo ucfirst($doc['tipo_documento']); ?></small>
                                    </td>
                                    <td><?php echo date('d/m/Y', strtotime($doc['fecha_emision'])); ?></td>
                                    <td>
                                        <?php if ($doc['fecha_vencimiento']): ?>
                                            <?php echo date('d/m/Y', strtotime($doc['fecha_vencimiento'])); ?>
                                        <?php else: ?>
                                            <span class="text-muted">Sin fecha</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($doc['razon_social_cliente']); ?></strong>
                                        <br><small><?php echo htmlspecialchars($doc['rut_cliente']); ?></small>
                                    </td>
                                    <td><?php echo formatearMoneda($doc['total'], $doc['moneda']); ?></td>
                                    <td class="text-success"><?php echo formatearMoneda($doc['total_pagado'], $doc['moneda']); ?></td>
                                    <td class="text-danger"><strong><?php echo formatearMoneda($doc['saldo_pendiente'], $doc['moneda']); ?></strong></td>
                                    <td>
                                        <span class="badge bg-<?php 
                                            switch($doc['estado_cobranza']) {
                                                case 'vencida': echo 'danger'; break;
                                                case 'por_vencer': echo 'warning'; break;
                                                case 'pagada': echo 'success'; break;
                                                default: echo 'primary'; break;
                                            }
                                        ?>">
                                            <?php echo ucfirst(str_replace('_', ' ', $doc['estado_cobranza'])); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($doc['dias_vencimiento'] > 0 && $doc['estado_cobranza'] === 'vencida'): ?>
                                            <span class="text-danger">+<?php echo $doc['dias_vencimiento']; ?></span>
                                        <?php elseif ($doc['dias_vencimiento'] < 0): ?>
                                            <span class="text-success"><?php echo abs($doc['dias_vencimiento']); ?></span>
                                        <?php else: ?>
                                            <span class="text-muted">0</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($doc['saldo_pendiente'] > 0): ?>
                                            <button class="btn btn-sm btn-success" onclick="registrarPago(<?php echo $doc['id']; ?>, '<?php echo htmlspecialchars($doc['numero_documento']); ?>', '<?php echo htmlspecialchars($doc['razon_social_cliente']); ?>', <?php echo $doc['saldo_pendiente']; ?>)">
                                                <i class="fas fa-money-bill-wave"></i>
                                            </button>
                                        <?php else: ?>
                                            <span class="badge bg-success">Pagado</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

        <?php elseif ($accion === 'antiguedad'): ?>
            <!-- Antigüedad de Saldos -->
            <div class="content-card">
                <h3><i class="fas fa-chart-bar"></i> Antigüedad de Saldos</h3>
                <p class="text-muted">Análisis de cuentas por cobrar por antigüedad</p>
                
                <div class="table-responsive">
                    <table class="table antiguedad-table">
                        <thead>
                            <tr>
                                <th>Cliente</th>
                                <th>RUT</th>
                                <th>Vigente</th>
                                <th>1-30 días</th>
                                <th>31-60 días</th>
                                <th>61-90 días</th>
                                <th>+90 días</th>
                                <th>Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $totales = ['vigente' => 0, 'vencido_30' => 0, 'vencido_60' => 0, 'vencido_90' => 0, 'vencido_mas_90' => 0, 'total_saldo' => 0];
                            foreach ($antiguedad_saldos as $cliente): 
                                $totales['vigente'] += $cliente['vigente'];
                                $totales['vencido_30'] += $cliente['vencido_30'];
                                $totales['vencido_60'] += $cliente['vencido_60'];
                                $totales['vencido_90'] += $cliente['vencido_90'];
                                $totales['vencido_mas_90'] += $cliente['vencido_mas_90'];
                                $totales['total_saldo'] += $cliente['total_saldo'];
                            ?>
                                <tr>
                                    <td><strong><?php echo htmlspecialchars($cliente['razon_social_cliente']); ?></strong></td>
                                    <td><?php echo htmlspecialchars($cliente['rut_cliente']); ?></td>
                                    <td class="text-success"><?php echo number_format($cliente['vigente'], 0); ?></td>
                                    <td class="text-warning"><?php echo number_format($cliente['vencido_30'], 0); ?></td>
                                    <td class="text-warning"><?php echo number_format($cliente['vencido_60'], 0); ?></td>
                                    <td class="text-danger"><?php echo number_format($cliente['vencido_90'], 0); ?></td>
                                    <td class="text-danger"><strong><?php echo number_format($cliente['vencido_mas_90'], 0); ?></strong></td>
                                    <td><strong><?php echo number_format($cliente['total_saldo'], 0); ?></strong></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                        <tfoot>
                            <tr class="table-secondary">
                                <td colspan="2"><strong>TOTALES</strong></td>
                                <td class="text-success"><strong><?php echo number_format($totales['vigente'], 0); ?></strong></td>
                                <td class="text-warning"><strong><?php echo number_format($totales['vencido_30'], 0); ?></strong></td>
                                <td class="text-warning"><strong><?php echo number_format($totales['vencido_60'], 0); ?></strong></td>
                                <td class="text-danger"><strong><?php echo number_format($totales['vencido_90'], 0); ?></strong></td>
                                <td class="text-danger"><strong><?php echo number_format($totales['vencido_mas_90'], 0); ?></strong></td>
                                <td><strong><?php echo number_format($totales['total_saldo'], 0); ?></strong></td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- Modal Registrar Pago -->
    <div class="modal fade" id="modalRegistrarPago" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Registrar Pago</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="documento_id" id="pagoDocumentoId">
                        
                        <div class="alert alert-info">
                            <strong>Documento:</strong> <span id="pagoNumeroDocumento"></span><br>
                            <strong>Cliente:</strong> <span id="pagoCliente"></span><br>
                            <strong>Saldo Pendiente:</strong> <span id="pagoSaldoPendiente"></span>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Monto del Pago *</label>
                            <input type="number" name="monto_pago" id="pagoMonto" class="form-control" step="0.01" min="0.01" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Fecha del Pago *</label>
                            <input type="date" name="fecha_pago" class="form-control" value="<?php echo date('Y-m-d'); ?>" required>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Método de Pago *</label>
                            <select name="metodo_pago" class="form-control" required>
                                <option value="">Seleccionar método...</option>
                                <option value="efectivo">Efectivo</option>
                                <option value="transferencia">Transferencia Bancaria</option>
                                <option value="cheque">Cheque</option>
                                <option value="tarjeta_credito">Tarjeta de Crédito</option>
                                <option value="tarjeta_debito">Tarjeta de Débito</option>
                            </select>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Número de Referencia</label>
                            <input type="text" name="numero_referencia" class="form-control" placeholder="Ej: número de transferencia, cheque, etc.">
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Observaciones</label>
                            <textarea name="observaciones" class="form-control" rows="3" placeholder="Observaciones adicionales"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" name="registrar_pago" class="btn btn-success">
                            <i class="fas fa-save"></i> Registrar Pago
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Función para registrar pago
        function registrarPago(documentoId, numeroDocumento, cliente, saldoPendiente) {
            document.getElementById('pagoDocumentoId').value = documentoId;
            document.getElementById('pagoNumeroDocumento').textContent = numeroDocumento;
            document.getElementById('pagoCliente').textContent = cliente;
            document.getElementById('pagoSaldoPendiente').textContent = '<?php echo $empresa['moneda_principal'] ?? 'CLP'; ?> ' + saldoPendiente.toLocaleString();
            document.getElementById('pagoMonto').max = saldoPendiente;
            document.getElementById('pagoMonto').value = saldoPendiente;
            
            const modal = new bootstrap.Modal(document.getElementById('modalRegistrarPago'));
            modal.show();
        }
        
        // Animaciones de entrada
        document.addEventListener('DOMContentLoaded', function() {
            const cards = document.querySelectorAll('.stat-card, .pago-item');
            cards.forEach((card, index) => {
                card.style.opacity = '0';
                card.style.transform = 'translateY(20px)';
                setTimeout(() => {
                    card.style.transition = 'all 0.6s ease';
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                }, index * 100);
            });
        });
    </script>
</body>
</html>