<?php
// PUNTO DE VENTA MEJORADO - SISCONECTA ERP
// pos_mejorado.php

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Verificar sesión activa
if (!isset($_SESSION['user_id']) || !isset($_SESSION['empresa_id'])) {
    header('Location: login.php');
    exit;
}

// Configuración de base de datos
$host = 'localhost';
$dbname = 'conectae_sisconecta';
$username = 'conectae_sisconectauser';
$password = 'pt125824caraud';

$mensaje = '';
$tipo_mensaje = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $empresa_id = $_SESSION['empresa_id'];
    
    // Verificar estado de empresa y suscripción
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ? AND estado = 'activo'");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$empresa) {
        session_destroy();
        header('Location: login.php?error=empresa_inactiva');
        exit;
    }
    
    // Verificar vencimiento (excepto AUDITOREX CHILE)
    if ($empresa['codigo_empresa'] !== 'AUDITOREX_CHILE') {
        $fecha_actual = new DateTime();
        $fecha_vencimiento = new DateTime($empresa['fecha_vencimiento']);
        
        if ($fecha_actual > $fecha_vencimiento) {
            session_destroy();
            header('Location: login.php?error=suscripcion_vencida');
            exit;
        }
    }
    
    // Parámetros de filtrado y paginación
    $busqueda = $_GET['buscar'] ?? '';
    $categoria_filtro = $_GET['categoria'] ?? '';
    $stock_filtro = $_GET['stock'] ?? '';
    $precio_min = $_GET['precio_min'] ?? '';
    $precio_max = $_GET['precio_max'] ?? '';
    $ordenar = $_GET['ordenar'] ?? 'nombre';
    $direccion = $_GET['direccion'] ?? 'ASC';
    $vista = $_GET['vista'] ?? 'cards'; // cards o list
    $productos_por_pagina = (int)($_GET['per_page'] ?? 12);
    $pagina_actual = (int)($_GET['pagina'] ?? 1);
    
    // Construir consulta con filtros
    $where_conditions = ["empresa_id = ? AND activo = 1"];
    $params = [$empresa_id];
    
    // Filtro de búsqueda
    if (!empty($busqueda)) {
        $where_conditions[] = "(nombre LIKE ? OR codigo LIKE ? OR descripcion LIKE ?)";
        $busqueda_param = '%' . $busqueda . '%';
        $params[] = $busqueda_param;
        $params[] = $busqueda_param;
        $params[] = $busqueda_param;
    }
    
    // Filtro de categoría
    if (!empty($categoria_filtro)) {
        $where_conditions[] = "categoria = ?";
        $params[] = $categoria_filtro;
    }
    
    // Filtro de stock
    if (!empty($stock_filtro)) {
        switch ($stock_filtro) {
            case 'disponible':
                $where_conditions[] = "stock_actual > 0";
                break;
            case 'sin_stock':
                $where_conditions[] = "stock_actual <= 0";
                break;
            case 'stock_bajo':
                $where_conditions[] = "stock_actual <= stock_minimo AND stock_actual > 0";
                break;
        }
    }
    
    // Filtro de precio
    if (!empty($precio_min)) {
        $where_conditions[] = "precio_venta >= ?";
        $params[] = $precio_min;
    }
    if (!empty($precio_max)) {
        $where_conditions[] = "precio_venta <= ?";
        $params[] = $precio_max;
    }
    
    // Validar ordenamiento
    $campos_ordenamiento = ['nombre', 'codigo', 'precio_venta', 'stock_actual', 'categoria'];
    if (!in_array($ordenar, $campos_ordenamiento)) {
        $ordenar = 'nombre';
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Contar total de productos para paginación
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM productos WHERE $where_clause");
    $stmt->execute($params);
    $total_productos = $stmt->fetch()['total'];
    
    // Calcular paginación
    $total_paginas = ceil($total_productos / $productos_por_pagina);
    $offset = ($pagina_actual - 1) * $productos_por_pagina;
    
    // Obtener productos con paginación
    $stmt = $pdo->prepare("
        SELECT * FROM productos 
        WHERE $where_clause 
        ORDER BY $ordenar $direccion 
        LIMIT $productos_por_pagina OFFSET $offset
    ");
    $stmt->execute($params);
    $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Obtener categorías para filtro
    $stmt = $pdo->prepare("SELECT DISTINCT categoria FROM productos WHERE empresa_id = ? AND activo = 1 AND categoria IS NOT NULL ORDER BY categoria");
    $stmt->execute([$empresa_id]);
    $categorias = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    // Obtener monedas activas
    $stmt = $pdo->prepare("SELECT * FROM monedas WHERE empresa_id = ? AND activo = 1 ORDER BY codigo ASC");
    $stmt->execute([$empresa_id]);
    $monedas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Obtener cajas abiertas
    $stmt = $pdo->prepare("SELECT * FROM cajas WHERE empresa_id = ? AND estado = 'abierta' ORDER BY nombre_caja ASC");
    $stmt->execute([$empresa_id]);
    $cajas_abiertas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Procesar venta (código existente del procesamiento de venta)
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['procesar_venta'])) {
        $cliente_rut = trim($_POST['cliente_rut']);
        $cliente_nombre = trim($_POST['cliente_nombre']);
        $cliente_direccion = trim($_POST['cliente_direccion']);
        $tipo_documento = $_POST['tipo_documento'];
        $moneda_venta = $_POST['moneda_venta'];
        $caja_id = $_POST['caja_id'];
        $observaciones = trim($_POST['observaciones']);
        $productos_venta = json_decode($_POST['productos_venta'], true);
        
        if (!empty($cliente_rut) && !empty($cliente_nombre) && !empty($productos_venta) && !empty($caja_id)) {
            try {
                $pdo->beginTransaction();
                
                // Mapeo COMPLETO de tipos de documento a códigos SII
                $tipos_sii = [
                    'boleta' => 39, 'boleta_exenta' => 41, 'factura' => 33, 'factura_exenta' => 34, 'factura_compra' => 46,
                    'nota_credito' => 61, 'nota_debito' => 56, 'guia_despacho' => 52, 'conocimiento_embarque' => 50, 'carta_porte' => 51,
                    'liquidacion_factura' => 43, 'liquidacion_factura_electronica' => 43, 'factura_exportacion' => 110, 
                    'nota_debito_exportacion' => 111, 'nota_credito_exportacion' => 112, 'factura_territorio_preferencial' => 914,
                    'boleta_honorarios' => 29, 'factura_electronica_mixta' => 35, 'liquidacion_comisiones' => 40, 
                    'comprobante_liquidacion' => 914, 'documento_exportacion' => 108, 'set_pruebas' => 29, 
                    'resumen_ventas' => 35, 'documento_exento' => 55, 'nota_credito_exenta' => 60, 
                    'nota_debito_exenta' => 55, 'factura_electronica_afecta' => 33, 'comprobante_pago' => 914
                ];
                
                $codigo_sii = $tipos_sii[$tipo_documento] ?? 33;
                
                // Verificar caja
                $stmt = $pdo->prepare("SELECT * FROM cajas WHERE id = ? AND empresa_id = ? AND estado = 'abierta'");
                $stmt->execute([$caja_id, $empresa_id]);
                $caja = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$caja) {
                    throw new Exception('Caja no disponible o cerrada');
                }
                
                // Calcular totales correctamente
                $subtotal = 0;
                $total_exento = 0;
                $total_iva = 0;
                $total_final = 0;
                
                foreach ($productos_venta as $item) {
                    $cantidad = (float)$item['cantidad'];
                    $precio = (float)$item['precio'];
                    $iva_porcentaje = (float)$item['iva'];
                    
                    $subtotal_item = $cantidad * $precio;
                    
                    if ($iva_porcentaje > 0) {
                        $subtotal += $subtotal_item;
                        $total_iva += $subtotal_item * ($iva_porcentaje / 100);
                    } else {
                        $total_exento += $subtotal_item;
                    }
                }
                
                $total_final = $subtotal + $total_exento + $total_iva;
                
                // Generar folio
                $stmt = $pdo->prepare("SELECT COALESCE(MAX(folio), 0) + 1 as siguiente_folio FROM documentos_tributarios WHERE empresa_id = ? AND tipo_documento = ?");
                $stmt->execute([$empresa_id, $tipo_documento]);
                $folio = $stmt->fetch()['siguiente_folio'];
                
                // Generar número de documento
                $numero_documento = strtoupper(substr($tipo_documento, 0, 1)) . str_pad($folio, 8, '0', STR_PAD_LEFT);
                
                // Insertar documento tributario CON codigo_sii y total_exento
                $stmt = $pdo->prepare("
                    INSERT INTO documentos_tributarios 
                    (empresa_id, tipo_documento, codigo_sii, numero_documento, folio, fecha_emision, rut_cliente, razon_social_cliente, direccion_cliente, subtotal, total_exento, iva, total, moneda, estado, observaciones) 
                    VALUES (?, ?, ?, ?, ?, CURDATE(), ?, ?, ?, ?, ?, ?, ?, ?, 'emitido', ?)
                ");
                $stmt->execute([
                    $empresa_id, $tipo_documento, $codigo_sii, $numero_documento, $folio, $cliente_rut, $cliente_nombre, $cliente_direccion, $subtotal, $total_exento, $total_iva, $total_final, $moneda_venta, $observaciones
                ]);
                $documento_id = $pdo->lastInsertId();
                
                // Insertar detalles del documento y actualizar stock
                foreach ($productos_venta as $item) {
                    $producto_id = $item['id'];
                    $cantidad = (float)$item['cantidad'];
                    $precio = (float)$item['precio'];
                    $iva_porcentaje = (float)$item['iva'];
                    $descripcion = $item['nombre'];
                    
                    // Verificar stock actual antes de procesar
                    $stmt = $pdo->prepare("SELECT stock_actual FROM productos WHERE id = ? AND empresa_id = ?");
                    $stmt->execute([$producto_id, $empresa_id]);
                    $stock_actual = $stmt->fetch()['stock_actual'] ?? 0;
                    
                    if ($stock_actual < $cantidad) {
                        throw new Exception("Stock insuficiente para el producto: {$descripcion}. Stock actual: {$stock_actual}, cantidad solicitada: {$cantidad}");
                    }
                    
                    // Calcular todos los valores necesarios
                    $subtotal_item = $cantidad * $precio;
                    $iva_item = $subtotal_item * ($iva_porcentaje / 100);
                    $total_item = $subtotal_item + $iva_item;
                    
                    // Insertar detalle del documento
                    $stmt = $pdo->prepare("
                        INSERT INTO documento_detalles 
                        (documento_id, producto_id, descripcion, cantidad, precio_unitario, subtotal, porcentaje_iva, iva_item, total_item) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$documento_id, $producto_id, $descripcion, $cantidad, $precio, $subtotal_item, $iva_porcentaje, $iva_item, $total_item]);
                    
                    // ACTUALIZAR STOCK DEL PRODUCTO - REBAJO INMEDIATO
                    $stmt = $pdo->prepare("UPDATE productos SET stock_actual = stock_actual - ? WHERE id = ? AND empresa_id = ?");
                    $stmt->execute([$cantidad, $producto_id, $empresa_id]);
                    
                    // Verificar que la actualización fue exitosa
                    if ($stmt->rowCount() === 0) {
                        throw new Exception("Error al actualizar stock del producto: {$descripcion}");
                    }
                }
                
                // Actualizar saldo de caja
                $stmt = $pdo->prepare("UPDATE cajas SET saldo_actual = saldo_actual + ? WHERE id = ? AND empresa_id = ?");
                $stmt->execute([$total_final, $caja_id, $empresa_id]);
                
                // Crear asiento contable automático
                $stmt = $pdo->prepare("SELECT COALESCE(MAX(numero_asiento), 0) + 1 as siguiente_asiento FROM asientos_contables WHERE empresa_id = ?");
                $stmt->execute([$empresa_id]);
                $numero_asiento = $stmt->fetch()['siguiente_asiento'];
                
                $descripcion_asiento = "Venta $tipo_documento $numero_documento - $cliente_nombre";
                
                $stmt = $pdo->prepare("
                    INSERT INTO asientos_contables 
                    (empresa_id, numero_asiento, fecha, descripcion, documento_referencia, tipo_asiento, estado, usuario_registro) 
                    VALUES (?, ?, CURDATE(), ?, ?, 'automatico', 'contabilizado', ?)
                ");
                $stmt->execute([$empresa_id, $numero_asiento, $descripcion_asiento, $numero_documento, $_SESSION['user_id']]);
                $asiento_id = $pdo->lastInsertId();
                
                // Movimientos contables
                // DEBE: Caja
                $stmt = $pdo->prepare("
                    INSERT INTO movimientos_contables 
                    (asiento_id, cuenta_contable, nombre_cuenta, debe, haber) 
                    VALUES (?, '1101', 'CAJA', ?, 0.00)
                ");
                $stmt->execute([$asiento_id, $total_final]);
                
                // HABER: Ventas
                $stmt = $pdo->prepare("
                    INSERT INTO movimientos_contables 
                    (asiento_id, cuenta_contable, nombre_cuenta, debe, haber) 
                    VALUES (?, '4101', 'VENTAS', 0.00, ?)
                ");
                $stmt->execute([$asiento_id, $subtotal]);
                
                // HABER: IVA Débito Fiscal
                if ($total_iva > 0) {
                    $stmt = $pdo->prepare("
                        INSERT INTO movimientos_contables 
                        (asiento_id, cuenta_contable, nombre_cuenta, debe, haber) 
                        VALUES (?, '2102', 'IVA DÉBITO FISCAL', 0.00, ?)
                    ");
                    $stmt->execute([$asiento_id, $total_iva]);
                }
                
                $pdo->commit();
                
                $mensaje = "Venta procesada exitosamente. Documento: $numero_documento (Código SII: $codigo_sii)";
                $tipo_mensaje = 'success';
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $mensaje = 'Error al procesar venta: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Complete todos los campos obligatorios y agregue productos';
            $tipo_mensaje = 'warning';
        }
    }
    
} catch (PDOException $e) {
    die('Error de conexión: ' . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Punto de Venta Mejorado - SISCONECTA ERP</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
        }
        
        .pos-container {
            display: grid;
            grid-template-columns: 1fr 400px;
            gap: 20px;
            padding: 20px;
            min-height: 100vh;
        }
        
        .productos-section {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
            overflow-y: auto;
            max-height: calc(100vh - 40px);
        }
        
        .venta-section {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
            position: sticky;
            top: 20px;
            height: fit-content;
            max-height: calc(100vh - 40px);
            overflow-y: auto;
        }
        
        .header-section {
            background: linear-gradient(45deg, #2c3e50, #34495e);
            color: white;
            padding: 20px 25px;
            margin: -25px -25px 25px -25px;
            border-radius: 15px 15px 0 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header-title {
            font-size: 1.5rem;
            font-weight: 600;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .btn-dashboard {
            background: rgba(255,255,255,0.2);
            border: 1px solid rgba(255,255,255,0.3);
            color: white;
            border-radius: 8px;
            padding: 8px 15px;
            text-decoration: none;
            transition: all 0.3s ease;
        }
        
        .btn-dashboard:hover {
            background: rgba(255,255,255,0.3);
            color: white;
        }
        
        /* Sección de filtros y controles */
        .filtros-section {
            background: #f8f9fa;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #e9ecef;
        }
        
        .filtros-row {
            display: grid;
            grid-template-columns: 2fr 1fr 1fr 1fr auto;
            gap: 15px;
            align-items: end;
            margin-bottom: 15px;
        }
        
        .controles-vista {
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 15px;
        }
        
        .vista-toggle {
            display: flex;
            background: white;
            border-radius: 8px;
            overflow: hidden;
            border: 1px solid #ddd;
        }
        
        .vista-btn {
            padding: 8px 15px;
            border: none;
            background: white;
            color: #666;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .vista-btn.active {
            background: #3498db;
            color: white;
        }
        
        .ordenamiento-control {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        
        .productos-por-pagina {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        
        .search-box {
            position: relative;
        }
        
        .search-input {
            width: 100%;
            padding: 12px 15px 12px 45px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 1rem;
            transition: all 0.3s ease;
        }
        
        .search-input:focus {
            border-color: #3498db;
            box-shadow: 0 0 0 0.2rem rgba(52,152,219,0.25);
            outline: none;
        }
        
        .search-icon {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #666;
            font-size: 1.1rem;
        }
        
        .clear-filters {
            background: #dc3545;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .clear-filters:hover {
            background: #c82333;
        }
        
        /* Estilos de filtros */
        .form-select, .form-control {
            border: 2px solid #e9ecef;
            border-radius: 6px;
            padding: 8px 12px;
            transition: all 0.3s ease;
            font-size: 0.9rem;
        }
        
        .form-select:focus, .form-control:focus {
            border-color: #3498db;
            box-shadow: 0 0 0 0.2rem rgba(52,152,219,0.25);
            outline: none;
        }
        
        /* Vista de productos */
        .productos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 20px;
        }
        
        .productos-list {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        
        .producto-card {
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s ease;
            cursor: pointer;
            background: #fafafa;
        }
        
        .producto-card:hover {
            border-color: #3498db;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(52,152,219,0.2);
            background: white;
        }
        
        .producto-card.sin-stock {
            border-color: #e74c3c;
            background: #fdf2f2;
            cursor: not-allowed;
            opacity: 0.7;
        }
        
        /* Vista de lista */
        .producto-list-item {
            display: grid;
            grid-template-columns: auto 1fr auto auto auto auto;
            gap: 15px;
            align-items: center;
            padding: 15px;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            background: white;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .producto-list-item:hover {
            border-color: #3498db;
            box-shadow: 0 2px 8px rgba(52,152,219,0.15);
        }
        
        .producto-list-item.sin-stock {
            border-color: #e74c3c;
            background: #fdf2f2;
            cursor: not-allowed;
            opacity: 0.7;
        }
        
        .producto-codigo {
            font-size: 0.8rem;
            color: #666;
            background: #e9ecef;
            padding: 4px 8px;
            border-radius: 4px;
            display: inline-block;
            margin-bottom: 10px;
        }
        
        .producto-nombre {
            font-size: 1.1rem;
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 8px;
        }
        
        .producto-descripcion {
            font-size: 0.9rem;
            color: #666;
            margin-bottom: 12px;
            line-height: 1.4;
        }
        
        .producto-precio {
            font-size: 1.3rem;
            font-weight: 700;
            color: #27ae60;
            margin-bottom: 10px;
        }
        
        .producto-stock {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .stock-info {
            font-size: 0.9rem;
        }
        
        .stock-actual {
            font-weight: 600;
            color: #2c3e50;
        }
        
        .stock-minimo {
            color: #f39c12;
        }
        
        .stock-bajo {
            color: #e74c3c;
            font-weight: 600;
        }
        
        .btn-agregar {
            background: linear-gradient(45deg, #27ae60, #229954);
            border: none;
            color: white;
            padding: 8px 16px;
            border-radius: 6px;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        
        .btn-agregar:hover {
            transform: translateY(-1px);
            box-shadow: 0 3px 8px rgba(39,174,96,0.3);
        }
        
        /* Paginación */
        .paginacion-section {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 20px;
            padding: 15px 0;
            border-top: 1px solid #e9ecef;
        }
        
        .paginacion-info {
            color: #666;
            font-size: 0.9rem;
        }
        
        .paginacion-controles {
            display: flex;
            gap: 5px;
        }
        
        .pagina-btn {
            padding: 8px 12px;
            border: 1px solid #ddd;
            background: white;
            color: #333;
            text-decoration: none;
            border-radius: 4px;
            transition: all 0.3s ease;
        }
        
        .pagina-btn:hover {
            background: #f8f9fa;
            color: #333;
        }
        
        .pagina-btn.active {
            background: #3498db;
            color: white;
            border-color: #3498db;
        }
        
        .pagina-btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Carrito de venta */
        .carrito-section {
            border-bottom: 1px solid #eee;
            padding-bottom: 20px;
            margin-bottom: 20px;
        }
        
        .carrito-item {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 10px;
            border-left: 4px solid #3498db;
        }
        
        .item-nombre {
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 5px;
        }
        
        .item-details {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 10px;
        }
        
        .item-precio {
            color: #27ae60;
            font-weight: 600;
        }
        
        .cantidad-controls {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .btn-cantidad {
            background: #3498db;
            border: none;
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn-cantidad:hover {
            background: #2980b9;
        }
        
        .cantidad-input {
            width: 60px;
            text-align: center;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 5px;
        }
        
        .btn-eliminar {
            background: #e74c3c;
            border: none;
            color: white;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 0.8rem;
            cursor: pointer;
        }
        
        .totales-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        .total-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
        }
        
        .total-row.final {
            border-top: 2px solid #ddd;
            padding-top: 10px;
            margin-top: 15px;
            font-size: 1.2rem;
            font-weight: 700;
            color: #2c3e50;
        }
        
        .form-section {
            margin-bottom: 20px;
        }
        
        .form-label {
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 8px;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 12px 15px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #3498db;
            box-shadow: 0 0 0 0.2rem rgba(52,152,219,0.25);
        }
        
        .btn-procesar {
            background: linear-gradient(45deg, #3498db, #2980b9);
            border: none;
            color: white;
            padding: 15px 25px;
            border-radius: 10px;
            font-weight: 600;
            width: 100%;
            font-size: 1.1rem;
            transition: all 0.3s ease;
        }
        
        .btn-procesar:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(52,152,219,0.3);
        }
        
        .btn-limpiar {
            background: #6c757d;
            border: none;
            color: white;
            padding: 10px 20px;
            border-radius: 8px;
            font-weight: 500;
            width: 100%;
            margin-bottom: 10px;
        }
        
        .alert {
            border: none;
            border-radius: 10px;
            padding: 15px 20px;
            margin-bottom: 20px;
            font-weight: 500;
        }
        
        .empresa-info {
            background: linear-gradient(45deg, #3498db, #2980b9);
            color: white;
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .empresa-info h6 {
            margin: 0;
            font-weight: 600;
        }
        
        .empresa-info small {
            opacity: 0.9;
        }
        
        .no-productos {
            text-align: center;
            padding: 40px;
            color: #666;
        }
        
        .no-productos i {
            font-size: 3rem;
            margin-bottom: 20px;
            color: #ddd;
        }
        
        .categoria-badge {
            background: #e9ecef;
            color: #666;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.75rem;
            margin-bottom: 8px;
            display: inline-block;
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .pos-container {
                grid-template-columns: 1fr;
                gap: 15px;
                padding: 15px;
            }
            
            .filtros-row {
                grid-template-columns: 1fr;
                gap: 10px;
            }
            
            .controles-vista {
                flex-direction: column;
                align-items: stretch;
            }
            
            .productos-grid {
                grid-template-columns: 1fr;
            }
            
            .producto-list-item {
                grid-template-columns: 1fr;
                text-align: center;
                gap: 10px;
            }
            
            .venta-section {
                position: static;
                max-height: none;
            }
        }
        
        /* Animaciones de carga */
        .loading {
            opacity: 0.6;
            pointer-events: none;
        }
        
        .fade-in {
            animation: fadeIn 0.3s ease-in;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
    </style>
</head>
<body>
    <div class="pos-container">
        <!-- Sección de Productos -->
        <div class="productos-section">
            <div class="header-section">
                <h2 class="header-title">
                    <i class="fas fa-shopping-cart"></i>
                    Punto de Venta Mejorado
                </h2>
                <a href="dashboard.php" class="btn-dashboard">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
            </div>
            
            <!-- Sección de Filtros y Controles -->
            <div class="filtros-section">
                <form method="GET" id="filtrosForm">
                    <div class="filtros-row">
                        <!-- Búsqueda principal -->
                        <div class="search-box">
                            <i class="fas fa-search search-icon"></i>
                            <input type="text" name="buscar" class="search-input" placeholder="Buscar por nombre, código o descripción..." value="<?php echo htmlspecialchars($busqueda); ?>">
                        </div>
                        
                        <!-- Filtro por categoría -->
                        <select name="categoria" class="form-select">
                            <option value="">Todas las categorías</option>
                            <?php foreach ($categorias as $categoria): ?>
                                <option value="<?php echo htmlspecialchars($categoria); ?>" <?php echo $categoria === $categoria_filtro ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($categoria); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        
                        <!-- Filtro por stock -->
                        <select name="stock" class="form-select">
                            <option value="">Todo el stock</option>
                            <option value="disponible" <?php echo $stock_filtro === 'disponible' ? 'selected' : ''; ?>>Con stock</option>
                            <option value="sin_stock" <?php echo $stock_filtro === 'sin_stock' ? 'selected' : ''; ?>>Sin stock</option>
                            <option value="stock_bajo" <?php echo $stock_filtro === 'stock_bajo' ? 'selected' : ''; ?>>Stock bajo</option>
                        </select>
                        
                        <!-- Filtro por precio -->
                        <div class="d-flex gap-2">
                            <input type="number" name="precio_min" class="form-control" placeholder="Precio mín." value="<?php echo htmlspecialchars($precio_min); ?>" style="width: 100px;">
                            <input type="number" name="precio_max" class="form-control" placeholder="Precio máx." value="<?php echo htmlspecialchars($precio_max); ?>" style="width: 100px;">
                        </div>
                        
                        <!-- Botón limpiar filtros -->
                        <button type="button" class="clear-filters" onclick="limpiarFiltros()">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    
                    <div class="controles-vista">
                        <!-- Toggle de vista -->
                        <div class="vista-toggle">
                            <button type="button" class="vista-btn <?php echo $vista === 'cards' ? 'active' : ''; ?>" onclick="cambiarVista('cards')">
                                <i class="fas fa-th"></i> Tarjetas
                            </button>
                            <button type="button" class="vista-btn <?php echo $vista === 'list' ? 'active' : ''; ?>" onclick="cambiarVista('list')">
                                <i class="fas fa-list"></i> Lista
                            </button>
                        </div>
                        
                        <!-- Ordenamiento -->
                        <div class="ordenamiento-control">
                            <select name="ordenar" class="form-select" onchange="aplicarFiltros()">
                                <option value="nombre" <?php echo $ordenar === 'nombre' ? 'selected' : ''; ?>>Nombre</option>
                                <option value="codigo" <?php echo $ordenar === 'codigo' ? 'selected' : ''; ?>>Código</option>
                                <option value="precio_venta" <?php echo $ordenar === 'precio_venta' ? 'selected' : ''; ?>>Precio</option>
                                <option value="stock_actual" <?php echo $ordenar === 'stock_actual' ? 'selected' : ''; ?>>Stock</option>
                                <option value="categoria" <?php echo $ordenar === 'categoria' ? 'selected' : ''; ?>>Categoría</option>
                            </select>
                            <button type="button" class="btn btn-outline-secondary" onclick="toggleDireccion()">
                                <i class="fas fa-sort-amount-<?php echo $direccion === 'ASC' ? 'up' : 'down'; ?>"></i>
                            </button>
                        </div>
                        
                        <!-- Productos por página -->
                        <div class="productos-por-pagina">
                            <span>Mostrar:</span>
                            <select name="per_page" class="form-select" onchange="aplicarFiltros()" style="width: auto;">
                                <option value="12" <?php echo $productos_por_pagina === 12 ? 'selected' : ''; ?>>12</option>
                                <option value="24" <?php echo $productos_por_pagina === 24 ? 'selected' : ''; ?>>24</option>
                                <option value="48" <?php echo $productos_por_pagina === 48 ? 'selected' : ''; ?>>48</option>
                                <option value="96" <?php echo $productos_por_pagina === 96 ? 'selected' : ''; ?>>96</option>
                            </select>
                        </div>
                    </div>
                    
                    <!-- Campos ocultos -->
                    <input type="hidden" name="vista" value="<?php echo $vista; ?>">
                    <input type="hidden" name="direccion" value="<?php echo $direccion; ?>">
                    <input type="hidden" name="pagina" value="1">
                </form>
            </div>
            
            <!-- Información de resultados -->
            <div class="mb-3">
                <small class="text-muted">
                    <i class="fas fa-info-circle"></i>
                    Mostrando <?php echo count($productos); ?> de <?php echo $total_productos; ?> productos
                    <?php if (!empty($busqueda)): ?>
                        para "<?php echo htmlspecialchars($busqueda); ?>"
                    <?php endif; ?>
                </small>
            </div>
            
            <!-- Lista de Productos -->
            <div id="productosContainer" class="fade-in">
                <?php if (empty($productos)): ?>
                    <div class="no-productos">
                        <i class="fas fa-box-open"></i>
                        <h4>No hay productos disponibles</h4>
                        <p>No se encontraron productos que coincidan con los filtros aplicados</p>
                        <button type="button" class="btn btn-primary" onclick="limpiarFiltros()">
                            <i class="fas fa-refresh"></i> Limpiar filtros
                        </button>
                    </div>
                <?php else: ?>
                    <div class="<?php echo $vista === 'cards' ? 'productos-grid' : 'productos-list'; ?>">
                        <?php foreach ($productos as $producto): ?>
                            <?php if ($vista === 'cards'): ?>
                                <div class="producto-card <?php echo $producto['stock_actual'] <= 0 ? 'sin-stock' : ''; ?>" 
                                     data-producto='<?php echo json_encode($producto); ?>'
                                     onclick="<?php echo $producto['stock_actual'] > 0 ? 'agregarAlCarrito(this)' : ''; ?>">
                                    
                                    <div class="producto-codigo"><?php echo htmlspecialchars($producto['codigo']); ?></div>
                                    
                                    <?php if (!empty($producto['categoria'])): ?>
                                        <div class="categoria-badge"><?php echo htmlspecialchars($producto['categoria']); ?></div>
                                    <?php endif; ?>
                                    
                                    <div class="producto-nombre"><?php echo htmlspecialchars($producto['nombre']); ?></div>
                                    
                                    <?php if (!empty($producto['descripcion'])): ?>
                                        <div class="producto-descripcion"><?php echo htmlspecialchars($producto['descripcion']); ?></div>
                                    <?php endif; ?>
                                    
                                    <div class="producto-precio">
                                        <?php echo $empresa['moneda_principal']; ?> <?php echo number_format($producto['precio_venta'], 0, ',', '.'); ?>
                                    </div>
                                    
                                    <div class="producto-stock">
                                        <div class="stock-info">
                                            <span class="stock-actual <?php echo $producto['stock_actual'] <= $producto['stock_minimo'] ? 'stock-bajo' : ''; ?>">
                                                Stock: <?php echo $producto['stock_actual']; ?>
                                            </span>
                                            <?php if ($producto['stock_actual'] <= $producto['stock_minimo'] && $producto['stock_actual'] > 0): ?>
                                                <span class="stock-minimo">(Mín: <?php echo $producto['stock_minimo']; ?>)</span>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <?php if ($producto['stock_actual'] > 0): ?>
                                            <button class="btn-agregar" onclick="event.stopPropagation(); agregarAlCarrito(this.parentNode.parentNode)">
                                                <i class="fas fa-plus"></i> Agregar
                                            </button>
                                        <?php else: ?>
                                            <span class="badge bg-danger">Sin Stock</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php else: ?>
                                <div class="producto-list-item <?php echo $producto['stock_actual'] <= 0 ? 'sin-stock' : ''; ?>" 
                                     data-producto='<?php echo json_encode($producto); ?>'
                                     onclick="<?php echo $producto['stock_actual'] > 0 ? 'agregarAlCarrito(this)' : ''; ?>">
                                    
                                    <div class="producto-codigo"><?php echo htmlspecialchars($producto['codigo']); ?></div>
                                    
                                    <div>
                                        <div class="producto-nombre" style="margin-bottom: 5px;"><?php echo htmlspecialchars($producto['nombre']); ?></div>
                                        <?php if (!empty($producto['categoria'])): ?>
                                            <span class="categoria-badge"><?php echo htmlspecialchars($producto['categoria']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div class="producto-precio" style="margin: 0;">
                                        <?php echo $empresa['moneda_principal']; ?> <?php echo number_format($producto['precio_venta'], 0, ',', '.'); ?>
                                    </div>
                                    
                                    <div class="stock-info">
                                        <span class="stock-actual <?php echo $producto['stock_actual'] <= $producto['stock_minimo'] ? 'stock-bajo' : ''; ?>">
                                            <?php echo $producto['stock_actual']; ?>
                                        </span>
                                    </div>
                                    
                                    <div>
                                        <?php if ($producto['stock_actual'] <= $producto['stock_minimo'] && $producto['stock_actual'] > 0): ?>
                                            <span class="badge bg-warning">Stock Bajo</span>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div>
                                        <?php if ($producto['stock_actual'] > 0): ?>
                                            <button class="btn-agregar" onclick="event.stopPropagation(); agregarAlCarrito(this.parentNode.parentNode)">
                                                <i class="fas fa-plus"></i> Agregar
                                            </button>
                                        <?php else: ?>
                                            <span class="badge bg-danger">Sin Stock</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Paginación -->
            <?php if ($total_paginas > 1): ?>
                <div class="paginacion-section">
                    <div class="paginacion-info">
                        Página <?php echo $pagina_actual; ?> de <?php echo $total_paginas; ?>
                        (<?php echo number_format($total_productos); ?> productos en total)
                    </div>
                    
                    <div class="paginacion-controles">
                        <?php
                        $url_base = $_SERVER['PHP_SELF'] . '?' . http_build_query(array_merge($_GET, ['pagina' => '']));
                        $url_base = rtrim($url_base, '=');
                        ?>
                        
                        <!-- Primera página -->
                        <?php if ($pagina_actual > 1): ?>
                            <a href="<?php echo $url_base; ?>=1" class="pagina-btn">
                                <i class="fas fa-angle-double-left"></i>
                            </a>
                            <a href="<?php echo $url_base; ?>=<?php echo $pagina_actual - 1; ?>" class="pagina-btn">
                                <i class="fas fa-angle-left"></i>
                            </a>
                        <?php endif; ?>
                        
                        <!-- Páginas numéricas -->
                        <?php
                        $inicio = max(1, $pagina_actual - 2);
                        $fin = min($total_paginas, $pagina_actual + 2);
                        
                        for ($i = $inicio; $i <= $fin; $i++):
                        ?>
                            <a href="<?php echo $url_base; ?>=<?php echo $i; ?>" 
                               class="pagina-btn <?php echo $i === $pagina_actual ? 'active' : ''; ?>">
                                <?php echo $i; ?>
                            </a>
                        <?php endfor; ?>
                        
                        <!-- Última página -->
                        <?php if ($pagina_actual < $total_paginas): ?>
                            <a href="<?php echo $url_base; ?>=<?php echo $pagina_actual + 1; ?>" class="pagina-btn">
                                <i class="fas fa-angle-right"></i>
                            </a>
                            <a href="<?php echo $url_base; ?>=<?php echo $total_paginas; ?>" class="pagina-btn">
                                <i class="fas fa-angle-double-right"></i>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Sección de Venta (sin cambios) -->
        <div class="venta-section">
            <div class="header-section">
                <h3 class="header-title">
                    <i class="fas fa-cash-register"></i>
                    Carrito de Venta
                </h3>
            </div>
            
            <!-- Información de la Empresa -->
            <div class="empresa-info">
                <h6><?php echo htmlspecialchars($empresa['razon_social']); ?></h6>
                <small>RUT: <?php echo htmlspecialchars($empresa['rut']); ?></small>
            </div>
            
            <!-- Mensajes -->
            <?php if (!empty($mensaje)): ?>
                <div class="alert alert-<?php echo $tipo_mensaje; ?>">
                    <i class="fas fa-<?php echo $tipo_mensaje === 'success' ? 'check-circle' : ($tipo_mensaje === 'danger' ? 'exclamation-triangle' : 'info-circle'); ?>"></i>
                    <?php echo $mensaje; ?>
                </div>
            <?php endif; ?>
            
            <!-- Carrito de Compras -->
            <div class="carrito-section">
                <h5><i class="fas fa-shopping-basket"></i> Productos en Carrito</h5>
                <div id="carritoItems">
                    <div class="text-muted text-center">
                        <i class="fas fa-cart-plus"></i><br>
                        Agregue productos al carrito
                    </div>
                </div>
            </div>
            
            <!-- Totales -->
            <div class="totales-section" id="totalesSection" style="display: none;">
                <h6><i class="fas fa-calculator"></i> Totales</h6>
                <div class="total-row">
                    <span>Subtotal:</span>
                    <span id="subtotalMonto">$0</span>
                </div>
                <div class="total-row">
                    <span>IVA:</span>
                    <span id="ivaMonto">$0</span>
                </div>
                <div class="total-row final">
                    <span>TOTAL:</span>
                    <span id="totalMonto">$0</span>
                </div>
            </div>
            
            <!-- Formulario de Venta -->
            <form method="POST" id="ventaForm">
                <input type="hidden" name="productos_venta" id="productosVenta">
                
                <div class="form-section">
                    <label class="form-label">Cliente RUT *</label>
                    <input type="text" name="cliente_rut" class="form-control" placeholder="12.345.678-9" required>
                </div>
                
                <div class="form-section">
                    <label class="form-label">Cliente Nombre *</label>
                    <input type="text" name="cliente_nombre" class="form-control" placeholder="Nombre completo del cliente" required>
                </div>
                
                <div class="form-section">
                    <label class="form-label">Dirección</label>
                    <input type="text" name="cliente_direccion" class="form-control" placeholder="Dirección del cliente">
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-section">
                            <label class="form-label">Tipo Documento *</label>
                            <select name="tipo_documento" class="form-control" required>
                                <option value="">Seleccionar tipo...</option>
                                
                                <!-- DOCUMENTOS DE VENTA -->
                                <optgroup label="📄 DOCUMENTOS DE VENTA">
                                    <option value="boleta">Boleta de Venta [SII: 39]</option>
                                    <option value="boleta_exenta">Boleta Exenta [SII: 41]</option>
                                    <option value="factura">Factura de Venta [SII: 33]</option>
                                    <option value="factura_exenta">Factura Exenta [SII: 34]</option>
                                    <option value="factura_compra">Factura de Compra [SII: 46]</option>
                                </optgroup>
                                
                                <!-- NOTAS DE AJUSTE -->
                                <optgroup label="📝 NOTAS DE AJUSTE">
                                    <option value="nota_credito">Nota de Crédito [SII: 61]</option>
                                    <option value="nota_debito">Nota de Débito [SII: 56]</option>
                                </optgroup>
                                
                                <!-- DOCUMENTOS DE DESPACHO -->
                                <optgroup label="🚚 DOCUMENTOS DE DESPACHO">
                                    <option value="guia_despacho">Guía de Despacho [SII: 52]</option>
                                    <option value="conocimiento_embarque">Conocimiento de Embarque [SII: 50]</option>
                                    <option value="carta_porte">Carta de Porte [SII: 51]</option>
                                </optgroup>
                                
                                <!-- LIQUIDACIONES -->
                                <optgroup label="💰 LIQUIDACIONES">
                                    <option value="liquidacion_factura">Liquidación de Factura [SII: 43]</option>
                                    <option value="liquidacion_factura_electronica">Liquidación Factura Electrónica [SII: 43]</option>
                                </optgroup>
                                
                                <!-- EXPORTACIÓN -->
                                <optgroup label="🌍 EXPORTACIÓN">
                                    <option value="factura_exportacion">Factura de Exportación [SII: 110]</option>
                                    <option value="nota_debito_exportacion">Nota Débito Exportación [SII: 111]</option>
                                    <option value="nota_credito_exportacion">Nota Crédito Exportación [SII: 112]</option>
                                    <option value="factura_territorio_preferencial">Factura Territorio Preferencial [SII: 914]</option>
                                </optgroup>
                                
                                <!-- DOCUMENTOS ESPECIALES -->
                                <optgroup label="⭐ DOCUMENTOS ESPECIALES">
                                    <option value="boleta_honorarios">Boleta de Honorarios [SII: 29]</option>
                                    <option value="factura_electronica_mixta">Factura Electrónica Mixta [SII: 35]</option>
                                    <option value="liquidacion_comisiones">Liquidación de Comisiones [SII: 40]</option>
                                    <option value="comprobante_liquidacion">Comprobante de Liquidación [SII: 914]</option>
                                </optgroup>
                                
                                <!-- OTROS DOCUMENTOS -->
                                <optgroup label="📋 OTROS DOCUMENTOS">
                                    <option value="documento_exportacion">Documento de Exportación [SII: 108]</option>
                                    <option value="set_pruebas">SET de Pruebas [SII: 29]</option>
                                    <option value="resumen_ventas">Resumen de Ventas Diarias [SII: 35]</option>
                                    <option value="documento_exento">Documento Exento [SII: 55]</option>
                                    <option value="nota_credito_exenta">Nota Crédito Exenta [SII: 60]</option>
                                    <option value="nota_debito_exenta">Nota Débito Exenta [SII: 55]</option>
                                    <option value="factura_electronica_afecta">Factura Electrónica Afecta [SII: 33]</option>
                                    <option value="comprobante_pago">Comprobante de Pago [SII: 914]</option>
                                </optgroup>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-section">
                            <label class="form-label">Moneda *</label>
                            <select name="moneda_venta" class="form-control" required>
                                <?php foreach ($monedas as $moneda): ?>
                                    <option value="<?php echo $moneda['codigo']; ?>" <?php echo $moneda['codigo'] === $empresa['moneda_principal'] ? 'selected' : ''; ?>>
                                        <?php echo $moneda['codigo']; ?> - <?php echo $moneda['nombre']; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                </div>
                
                <div class="form-section">
                    <label class="form-label">Caja *</label>
                    <select name="caja_id" class="form-control" required>
                        <option value="">Seleccionar caja...</option>
                        <?php foreach ($cajas_abiertas as $caja): ?>
                            <option value="<?php echo $caja['id']; ?>">
                                <?php echo htmlspecialchars($caja['nombre_caja']); ?> 
                                (<?php echo $caja['moneda']; ?> <?php echo number_format($caja['saldo_actual'], 2, ',', '.'); ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-section">
                    <label class="form-label">Observaciones</label>
                    <textarea name="observaciones" class="form-control" rows="2" placeholder="Observaciones adicionales"></textarea>
                </div>
                
                <button type="button" class="btn-limpiar" onclick="limpiarCarrito()">
                    <i class="fas fa-trash"></i> Limpiar Carrito
                </button>
                
                <button type="submit" name="procesar_venta" class="btn-procesar" id="btnProcesar" disabled>
                    <i class="fas fa-credit-card"></i> Procesar Venta
                </button>
            </form>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let carrito = [];
        let monedaPrincipal = '<?php echo $empresa['moneda_principal']; ?>';
        let filtrosTimeout;
        
        // Funciones de filtrado y navegación
        function aplicarFiltros() {
            clearTimeout(filtrosTimeout);
            filtrosTimeout = setTimeout(() => {
                document.getElementById('filtrosForm').submit();
            }, 300);
        }
        
        function limpiarFiltros() {
            const form = document.getElementById('filtrosForm');
            const inputs = form.querySelectorAll('input, select');
            inputs.forEach(input => {
                if (input.name !== 'vista' && input.name !== 'per_page') {
                    if (input.type === 'text' || input.type === 'number') {
                        input.value = '';
                    } else if (input.tagName === 'SELECT') {
                        input.value = '';
                    }
                }
            });
            aplicarFiltros();
        }
        
        function cambiarVista(vista) {
            document.querySelector('input[name="vista"]').value = vista;
            aplicarFiltros();
        }
        
        function toggleDireccion() {
            const direccionInput = document.querySelector('input[name="direccion"]');
            direccionInput.value = direccionInput.value === 'ASC' ? 'DESC' : 'ASC';
            aplicarFiltros();
        }
        
        // Event listeners para filtros automáticos
        document.addEventListener('DOMContentLoaded', function() {
            // Búsqueda automática con debounce
            const buscarInput = document.querySelector('input[name="buscar"]');
            buscarInput.addEventListener('input', function() {
                clearTimeout(filtrosTimeout);
                filtrosTimeout = setTimeout(() => {
                    aplicarFiltros();
                }, 500);
            });
            
            // Aplicar filtros al cambiar selects
            const filtroSelects = document.querySelectorAll('.filtros-section select:not([name="ordenar"]):not([name="per_page"])');
            filtroSelects.forEach(select => {
                select.addEventListener('change', aplicarFiltros);
            });
            
            // Aplicar filtros al cambiar precio
            const precioInputs = document.querySelectorAll('input[name="precio_min"], input[name="precio_max"]');
            precioInputs.forEach(input => {
                input.addEventListener('input', function() {
                    clearTimeout(filtrosTimeout);
                    filtrosTimeout = setTimeout(() => {
                        aplicarFiltros();
                    }, 1000);
                });
            });
        });
        
        // Funciones del carrito (sin cambios)
        function agregarAlCarrito(elemento) {
            const productoData = JSON.parse(elemento.getAttribute('data-producto'));
            
            // Verificar si el producto ya está en el carrito
            const existe = carrito.find(item => item.id === productoData.id);
            
            if (existe) {
                // Verificar stock disponible
                if (existe.cantidad < productoData.stock_actual) {
                    existe.cantidad++;
                } else {
                    alert('No hay más stock disponible');
                    return;
                }
            } else {
                // Agregar nuevo producto
                carrito.push({
                    id: productoData.id,
                    codigo: productoData.codigo,
                    nombre: productoData.nombre,
                    precio: parseFloat(productoData.precio_venta),
                    iva: parseFloat(productoData.iva_porcentaje),
                    cantidad: 1,
                    stock_disponible: productoData.stock_actual
                });
            }
            
            actualizarCarrito();
        }
        
        function actualizarCarrito() {
            const carritoItems = document.getElementById('carritoItems');
            const totalesSection = document.getElementById('totalesSection');
            const btnProcesar = document.getElementById('btnProcesar');
            
            if (carrito.length === 0) {
                carritoItems.innerHTML = `
                    <div class="text-muted text-center">
                        <i class="fas fa-cart-plus"></i><br>
                        Agregue productos al carrito
                    </div>
                `;
                totalesSection.style.display = 'none';
                btnProcesar.disabled = true;
                return;
            }
            
            let html = '';
            let subtotal = 0;
            let totalIva = 0;
            
            carrito.forEach((item, index) => {
                const subtotalItem = item.cantidad * item.precio;
                const ivaItem = subtotalItem * (item.iva / 100);
                
                subtotal += subtotalItem;
                totalIva += ivaItem;
                
                html += `
                    <div class="carrito-item">
                        <div class="item-nombre">${item.nombre}</div>
                        <div class="item-details">
                            <span class="item-precio">${monedaPrincipal} ${item.precio.toLocaleString()}</span>
                            <button type="button" class="btn-eliminar" onclick="eliminarDelCarrito(${index})">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                        <div class="cantidad-controls">
                            <button type="button" class="btn-cantidad" onclick="cambiarCantidad(${index}, -1)">
                                <i class="fas fa-minus"></i>
                            </button>
                            <input type="number" class="cantidad-input" value="${item.cantidad}" min="1" max="${item.stock_disponible}" 
                                   onchange="cambiarCantidadDirecta(${index}, this.value)">
                            <button type="button" class="btn-cantidad" onclick="cambiarCantidad(${index}, 1)">
                                <i class="fas fa-plus"></i>
                            </button>
                        </div>
                        <small class="text-muted">Subtotal: ${monedaPrincipal} ${subtotalItem.toLocaleString()}</small>
                    </div>
                `;
            });
            
            carritoItems.innerHTML = html;
            
            // Actualizar totales
            const total = subtotal + totalIva;
            document.getElementById('subtotalMonto').textContent = monedaPrincipal + ' ' + subtotal.toLocaleString();
            document.getElementById('ivaMonto').textContent = monedaPrincipal + ' ' + totalIva.toLocaleString();
            document.getElementById('totalMonto').textContent = monedaPrincipal + ' ' + total.toLocaleString();
            
            totalesSection.style.display = 'block';
            btnProcesar.disabled = false;
            
            // Actualizar campo hidden
            document.getElementById('productosVenta').value = JSON.stringify(carrito);
        }
        
        function cambiarCantidad(index, cambio) {
            const item = carrito[index];
            const nuevaCantidad = item.cantidad + cambio;
            
            if (nuevaCantidad <= 0) {
                eliminarDelCarrito(index);
                return;
            }
            
            if (nuevaCantidad > item.stock_disponible) {
                alert('No hay suficiente stock disponible');
                return;
            }
            
            item.cantidad = nuevaCantidad;
            actualizarCarrito();
        }
        
        function cambiarCantidadDirecta(index, nuevaCantidad) {
            nuevaCantidad = parseInt(nuevaCantidad);
            
            if (isNaN(nuevaCantidad) || nuevaCantidad <= 0) {
                eliminarDelCarrito(index);
                return;
            }
            
            const item = carrito[index];
            
            if (nuevaCantidad > item.stock_disponible) {
                alert('No hay suficiente stock disponible');
                actualizarCarrito(); // Restaurar valor anterior
                return;
            }
            
            item.cantidad = nuevaCantidad;
            actualizarCarrito();
        }
        
        function eliminarDelCarrito(index) {
            carrito.splice(index, 1);
            actualizarCarrito();
        }
        
        function limpiarCarrito() {
            if (carrito.length > 0) {
                if (confirm('¿Está seguro de limpiar el carrito?')) {
                    carrito = [];
                    actualizarCarrito();
                }
            }
        }
        
        // Validación del formulario
        document.getElementById('ventaForm').addEventListener('submit', function(e) {
            if (carrito.length === 0) {
                e.preventDefault();
                alert('Debe agregar al menos un producto al carrito');
                return false;
            }
        });
        
        // Validación RUT chileno formato XX.XXX.XXX-X
        document.querySelector('input[name="cliente_rut"]').addEventListener('input', function(e) {
            let value = e.target.value.replace(/[^0-9kK]/g, '');
            
            if (value.length >= 2) {
                let dv = value.slice(-1).toUpperCase();
                let numero = value.slice(0, -1);
                
                // Formatear número XX.XXX.XXX-X
                if (numero.length <= 2) {
                    value = numero + '-' + dv;
                } else if (numero.length <= 5) {
                    value = numero.slice(0, 2) + '.' + numero.slice(2) + '-' + dv;
                } else if (numero.length <= 8) {
                    value = numero.slice(0, 2) + '.' + numero.slice(2, 5) + '.' + numero.slice(5) + '-' + dv;
                } else {
                    // Máximo 8 dígitos + DV
                    numero = numero.slice(0, 8);
                    value = numero.slice(0, 2) + '.' + numero.slice(2, 5) + '.' + numero.slice(5) + '-' + dv;
                }
                
                e.target.value = value;
            } else {
                e.target.value = value;
            }
        });
        
        // Auto-focus en búsqueda
        document.querySelector('input[name="buscar"]').focus();
        
        // Atajos de teclado
        document.addEventListener('keydown', function(e) {
            // F1 - Focus en búsqueda
            if (e.key === 'F1') {
                e.preventDefault();
                document.querySelector('input[name="buscar"]').focus();
            }
            
            // F2 - Limpiar filtros
            if (e.key === 'F2') {
                e.preventDefault();
                limpiarFiltros();
            }
            
            // F3 - Cambiar vista
            if (e.key === 'F3') {
                e.preventDefault();
                const vistaActual = document.querySelector('input[name="vista"]').value;
                cambiarVista(vistaActual === 'cards' ? 'list' : 'cards');
            }
        });
        
        // Mostrar indicador de carga al aplicar filtros
        function mostrarCarga() {
            document.getElementById('productosContainer').classList.add('loading');
        }
        
        // Aplicar animaciones al cargar la página
        document.addEventListener('DOMContentLoaded', function() {
            const elementos = document.querySelectorAll('.producto-card, .producto-list-item');
            elementos.forEach((elemento, index) => {
                setTimeout(() => {
                    elemento.classList.add('fade-in');
                }, index * 50);
            });
        });
    </script>
</body>
</html>