<?php
// GESTIÓN DE PRODUCTOS E INVENTARIO - SISCONECTA ERP
// productos.php - Versión limpia y funcional

session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Verificar sesión activa
if (!isset($_SESSION['user_id']) || !isset($_SESSION['empresa_id'])) {
    header('Location: login.php');
    exit;
}

// Configuración de base de datos
$host = 'localhost';
$dbname = 'conectae_sisconecta';
$username = 'conectae_sisconectauser';
$password = 'pt125824caraud';

$mensaje = '';
$tipo_mensaje = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $empresa_id = $_SESSION['empresa_id'];
    
    // Verificar estado de empresa
    $stmt = $pdo->prepare("SELECT * FROM empresas WHERE id = ? AND estado = 'activo'");
    $stmt->execute([$empresa_id]);
    $empresa = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$empresa) {
        session_destroy();
        header('Location: login.php?error=empresa_inactiva');
        exit;
    }
    
    // Verificar vencimiento (excepto AUDITOREX CHILE)
    if ($empresa['codigo_empresa'] !== 'AUDITOREX_CHILE') {
        $fecha_actual = new DateTime();
        $fecha_vencimiento = new DateTime($empresa['fecha_vencimiento']);
        
        if ($fecha_actual > $fecha_vencimiento) {
            session_destroy();
            header('Location: login.php?error=suscripcion_vencida');
            exit;
        }
    }
    
    // Variables de estado
    $accion = $_GET['accion'] ?? 'listar';
    $producto_id = $_GET['id'] ?? null;
    
    // EXPORTAR CSV
    if (isset($_GET['exportar']) && $_GET['exportar'] === 'csv') {
        $stmt = $pdo->prepare("SELECT * FROM productos WHERE empresa_id = ? ORDER BY nombre ASC");
        $stmt->execute([$empresa_id]);
        $productos_export = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=productos_' . date('Y-m-d') . '.csv');
        
        $output = fopen('php://output', 'w');
        fputcsv($output, ['Código', 'Nombre', 'Descripción', 'Categoría', 'Precio Compra', 'Precio Venta', 'Stock Actual', 'Stock Mínimo', 'Unidad', 'IVA %', 'Estado']);
        
        foreach ($productos_export as $prod) {
            fputcsv($output, [
                $prod['codigo'],
                $prod['nombre'],
                $prod['descripcion'],
                $prod['categoria'],
                $prod['precio_compra'],
                $prod['precio_venta'],
                $prod['stock_actual'],
                $prod['stock_minimo'],
                $prod['unidad_medida'],
                $prod['iva_porcentaje'],
                $prod['activo'] ? 'Activo' : 'Inactivo'
            ]);
        }
        fclose($output);
        exit;
    }
    
    // ACCIONES MASIVAS
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['accion_masiva'])) {
        $productos_seleccionados = explode(',', $_POST['productos_seleccionados']);
        $accion_masiva = $_POST['accion_masiva'];
        
        if (!empty($productos_seleccionados[0])) {
            $placeholders = str_repeat('?,', count($productos_seleccionados) - 1) . '?';
            $params = array_merge($productos_seleccionados, [$empresa_id]);
            
            switch ($accion_masiva) {
                case 'activar':
                    $stmt = $pdo->prepare("UPDATE productos SET activo = 1 WHERE id IN ($placeholders) AND empresa_id = ?");
                    $stmt->execute($params);
                    $mensaje = count($productos_seleccionados) . ' productos activados';
                    $tipo_mensaje = 'success';
                    break;
                    
                case 'desactivar':
                    $stmt = $pdo->prepare("UPDATE productos SET activo = 0 WHERE id IN ($placeholders) AND empresa_id = ?");
                    $stmt->execute($params);
                    $mensaje = count($productos_seleccionados) . ' productos desactivados';
                    $tipo_mensaje = 'success';
                    break;
                    
                case 'aumentar_precio':
                    $porcentaje = (float)$_POST['porcentaje_precio'];
                    if ($porcentaje > 0) {
                        $stmt = $pdo->prepare("UPDATE productos SET precio_venta = precio_venta * (1 + ? / 100) WHERE id IN ($placeholders) AND empresa_id = ?");
                        $params_precio = array_merge([$porcentaje], $productos_seleccionados, [$empresa_id]);
                        $stmt->execute($params_precio);
                        $mensaje = "Precios aumentados en {$porcentaje}% para " . count($productos_seleccionados) . ' productos';
                        $tipo_mensaje = 'success';
                    }
                    break;
            }
        }
    }
    
    // CREAR/EDITAR PRODUCTO
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['guardar_producto'])) {
        $codigo = strtoupper(trim($_POST['codigo']));
        $nombre = trim($_POST['nombre']);
        $descripcion = trim($_POST['descripcion']);
        $categoria = trim($_POST['categoria']);
        $precio_compra = (float)$_POST['precio_compra'];
        $precio_venta = (float)$_POST['precio_venta'];
        $stock_actual = (int)$_POST['stock_actual'];
        $stock_minimo = (int)$_POST['stock_minimo'];
        $unidad_medida = trim($_POST['unidad_medida']);
        $iva_porcentaje = (float)$_POST['iva_porcentaje'];
        $codigo_sii = trim($_POST['codigo_sii']);
        $activo = isset($_POST['activo']) ? 1 : 0;
        
        if (!empty($codigo) && !empty($nombre) && $precio_venta > 0) {
            try {
                if ($accion === 'editar' && $producto_id) {
                    $stmt = $pdo->prepare("
                        UPDATE productos SET 
                        codigo = ?, nombre = ?, descripcion = ?, categoria = ?, 
                        precio_compra = ?, precio_venta = ?, stock_actual = ?, stock_minimo = ?, 
                        unidad_medida = ?, iva_porcentaje = ?, codigo_sii = ?, activo = ?
                        WHERE id = ? AND empresa_id = ?
                    ");
                    $stmt->execute([
                        $codigo, $nombre, $descripcion, $categoria, $precio_compra, $precio_venta,
                        $stock_actual, $stock_minimo, $unidad_medida, $iva_porcentaje, $codigo_sii, $activo,
                        $producto_id, $empresa_id
                    ]);
                    $mensaje = 'Producto actualizado exitosamente';
                } else {
                    // Verificar código único
                    $stmt = $pdo->prepare("SELECT id FROM productos WHERE codigo = ? AND empresa_id = ?");
                    $stmt->execute([$codigo, $empresa_id]);
                    if ($stmt->fetch()) {
                        throw new Exception('El código de producto ya existe');
                    }
                    
                    $stmt = $pdo->prepare("
                        INSERT INTO productos 
                        (empresa_id, codigo, nombre, descripcion, categoria, precio_compra, precio_venta, 
                         stock_actual, stock_minimo, unidad_medida, iva_porcentaje, codigo_sii, activo) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([
                        $empresa_id, $codigo, $nombre, $descripcion, $categoria, $precio_compra, $precio_venta,
                        $stock_actual, $stock_minimo, $unidad_medida, $iva_porcentaje, $codigo_sii, $activo
                    ]);
                    $mensaje = 'Producto creado exitosamente';
                }
                $tipo_mensaje = 'success';
                $accion = 'listar';
            } catch (Exception $e) {
                $mensaje = 'Error: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        } else {
            $mensaje = 'Complete todos los campos obligatorios';
            $tipo_mensaje = 'warning';
        }
    }
    
    // ELIMINAR PRODUCTO
    if (isset($_GET['eliminar'])) {
        $stmt = $pdo->prepare("UPDATE productos SET activo = 0 WHERE id = ? AND empresa_id = ?");
        $stmt->execute([$_GET['eliminar'], $empresa_id]);
        $mensaje = 'Producto desactivado exitosamente';
        $tipo_mensaje = 'success';
    }
    
    // AJUSTE DE INVENTARIO
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajustar_inventario'])) {
        $producto_ajuste_id = $_POST['producto_ajuste_id'];
        $tipo_ajuste = $_POST['tipo_ajuste'];
        $cantidad_ajuste = (int)$_POST['cantidad_ajuste'];
        $motivo_ajuste = trim($_POST['motivo_ajuste']);
        
        if ($producto_ajuste_id && $cantidad_ajuste > 0 && $motivo_ajuste) {
            try {
                $pdo->beginTransaction();
                
                // Obtener producto
                $stmt = $pdo->prepare("SELECT * FROM productos WHERE id = ? AND empresa_id = ?");
                $stmt->execute([$producto_ajuste_id, $empresa_id]);
                $producto = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$producto) {
                    throw new Exception('Producto no encontrado');
                }
                
                // Calcular nuevo stock
                $stock_anterior = $producto['stock_actual'];
                if ($tipo_ajuste === 'entrada') {
                    $nuevo_stock = $stock_anterior + $cantidad_ajuste;
                } else {
                    $nuevo_stock = $stock_anterior - $cantidad_ajuste;
                    if ($nuevo_stock < 0) {
                        throw new Exception('Stock insuficiente');
                    }
                }
                
                // Actualizar stock
                $stmt = $pdo->prepare("UPDATE productos SET stock_actual = ? WHERE id = ? AND empresa_id = ?");
                $stmt->execute([$nuevo_stock, $producto_ajuste_id, $empresa_id]);
                
                // Crear tabla de movimientos si no existe
                $pdo->exec("
                    CREATE TABLE IF NOT EXISTS movimientos_inventario (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        empresa_id INT NOT NULL,
                        producto_id INT NOT NULL,
                        tipo_movimiento VARCHAR(50) NOT NULL,
                        cantidad INT NOT NULL,
                        stock_anterior INT NOT NULL,
                        stock_nuevo INT NOT NULL,
                        motivo TEXT,
                        usuario_id INT,
                        fecha_registro TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                    ) ENGINE=InnoDB
                ");
                
                // Registrar movimiento
                $stmt = $pdo->prepare("
                    INSERT INTO movimientos_inventario 
                    (empresa_id, producto_id, tipo_movimiento, cantidad, stock_anterior, stock_nuevo, motivo, usuario_id) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $empresa_id, $producto_ajuste_id, 'ajuste_' . $tipo_ajuste, $cantidad_ajuste,
                    $stock_anterior, $nuevo_stock, $motivo_ajuste, $_SESSION['user_id']
                ]);
                
                $pdo->commit();
                $mensaje = "Stock actualizado: $stock_anterior → $nuevo_stock";
                $tipo_mensaje = 'success';
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $mensaje = 'Error: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        }
    }
    
    // OBTENER DATOS PARA EDICIÓN
    $producto_editar = null;
    if ($accion === 'editar' && $producto_id) {
        $stmt = $pdo->prepare("SELECT * FROM productos WHERE id = ? AND empresa_id = ?");
        $stmt->execute([$producto_id, $empresa_id]);
        $producto_editar = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$producto_editar) {
            $accion = 'listar';
            $mensaje = 'Producto no encontrado';
            $tipo_mensaje = 'danger';
        }
    }
    
    // HISTORIAL DE MOVIMIENTOS
    $movimientos = [];
    $producto_historial = null;
    if ($accion === 'historial' && $producto_id) {
        $stmt = $pdo->prepare("SELECT * FROM productos WHERE id = ? AND empresa_id = ?");
        $stmt->execute([$producto_id, $empresa_id]);
        $producto_historial = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($producto_historial) {
            try {
                $stmt = $pdo->prepare("
                    SELECT mi.*, u.nombre as usuario_nombre 
                    FROM movimientos_inventario mi 
                    LEFT JOIN usuarios u ON mi.usuario_id = u.id 
                    WHERE mi.producto_id = ? AND mi.empresa_id = ? 
                    ORDER BY mi.fecha_registro DESC 
                    LIMIT 50
                ");
                $stmt->execute([$producto_id, $empresa_id]);
                $movimientos = $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch (Exception $e) {
                $movimientos = [];
            }
        }
    }
    
    // OBTENER LISTA DE PRODUCTOS CON FILTROS
    $filtro_busqueda = $_GET['buscar'] ?? '';
    $filtro_categoria = $_GET['categoria'] ?? '';
    $filtro_estado = $_GET['estado'] ?? 'todos';
    $vista = $_GET['vista'] ?? 'tabla';
    
    $where_conditions = ["p.empresa_id = ?"];
    $params = [$empresa_id];
    
    if ($filtro_busqueda) {
        $where_conditions[] = "(p.nombre LIKE ? OR p.codigo LIKE ?)";
        $busqueda_param = '%' . $filtro_busqueda . '%';
        $params[] = $busqueda_param;
        $params[] = $busqueda_param;
    }
    
    if ($filtro_categoria) {
        $where_conditions[] = "p.categoria = ?";
        $params[] = $filtro_categoria;
    }
    
    if ($filtro_estado === 'activos') {
        $where_conditions[] = "p.activo = 1";
    } elseif ($filtro_estado === 'inactivos') {
        $where_conditions[] = "p.activo = 0";
    } elseif ($filtro_estado === 'stock_bajo') {
        $where_conditions[] = "p.stock_actual <= p.stock_minimo AND p.activo = 1";
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Verificar si existen tablas para ventas
    $consulta_ventas = "0";
    try {
        $stmt = $pdo->query("SHOW TABLES LIKE 'documento_detalles'");
        if ($stmt->fetch()) {
            $consulta_ventas = "(SELECT COUNT(*) FROM documento_detalles dd 
                               JOIN documentos_tributarios dt ON dd.documento_id = dt.id 
                               WHERE dd.producto_id = p.id AND dt.fecha_emision >= DATE_SUB(CURDATE(), INTERVAL 30 DAY))";
        }
    } catch (Exception $e) {
        $consulta_ventas = "0";
    }
    
    $stmt = $pdo->prepare("
        SELECT p.*, 
               $consulta_ventas as ventas_ultimo_mes,
               (p.precio_venta - p.precio_compra) as margen_absoluto,
               CASE WHEN p.precio_compra > 0 THEN ((p.precio_venta - p.precio_compra) / p.precio_compra * 100) ELSE 0 END as margen_porcentual
        FROM productos p 
        WHERE $where_clause 
        ORDER BY p.nombre ASC
    ");
    $stmt->execute($params);
    $productos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // OBTENER CATEGORÍAS
    $stmt = $pdo->prepare("SELECT DISTINCT categoria FROM productos WHERE empresa_id = ? AND categoria IS NOT NULL AND categoria != '' ORDER BY categoria ASC");
    $stmt->execute([$empresa_id]);
    $categorias = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    // ESTADÍSTICAS
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM productos WHERE empresa_id = ? AND activo = 1");
    $stmt->execute([$empresa_id]);
    $total_productos = $stmt->fetch()['total'];
    
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM productos WHERE empresa_id = ? AND stock_actual <= stock_minimo AND activo = 1");
    $stmt->execute([$empresa_id]);
    $productos_stock_bajo = $stmt->fetch()['total'];
    
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM productos WHERE empresa_id = ? AND stock_actual = 0 AND activo = 1");
    $stmt->execute([$empresa_id]);
    $productos_sin_stock = $stmt->fetch()['total'];
    
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(stock_actual * precio_compra), 0) as valor FROM productos WHERE empresa_id = ? AND activo = 1");
    $stmt->execute([$empresa_id]);
    $valor_inventario = $stmt->fetch()['valor'];
    
} catch (PDOException $e) {
    $mensaje = 'Error de conexión a la base de datos';
    $tipo_mensaje = 'danger';
    
    // Valores por defecto
    $empresa = ['moneda_principal' => '$'];
    $productos = [];
    $categorias = [];
    $total_productos = 0;
    $productos_stock_bajo = 0;
    $productos_sin_stock = 0;
    $valor_inventario = 0;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Productos - SISCONECTA ERP</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body { 
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .main-container { 
            padding: 20px; 
            max-width: 1400px; 
            margin: 0 auto; 
        }
        
        .header-section {
            background: linear-gradient(45deg, #2c3e50, #34495e);
            color: white;
            padding: 25px 30px;
            border-radius: 15px;
            margin-bottom: 25px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.15);
        }
        
        .header-title {
            font-size: 1.8rem;
            font-weight: 600;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .header-actions {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
            margin-top: 15px;
        }
        
        .btn-header {
            background: rgba(255,255,255,0.2);
            border: 1px solid rgba(255,255,255,0.3);
            color: white;
            border-radius: 8px;
            padding: 8px 16px;
            text-decoration: none;
            transition: all 0.3s ease;
            font-size: 0.9rem;
        }
        
        .btn-header:hover {
            background: rgba(255,255,255,0.3);
            color: white;
            transform: translateY(-1px);
        }
        
        .stats-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 25px;
        }
        
        .stat-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            border-left: 4px solid;
            transition: transform 0.3s ease;
        }
        
        .stat-card:hover { transform: translateY(-3px); }
        .stat-card.primary { border-left-color: #3498db; }
        .stat-card.warning { border-left-color: #f39c12; }
        .stat-card.danger { border-left-color: #e74c3c; }
        .stat-card.success { border-left-color: #27ae60; }
        
        .stat-value {
            font-size: 1.8rem;
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 5px;
        }
        
        .stat-label {
            color: #666;
            font-weight: 500;
            font-size: 0.9rem;
        }
        
        .content-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 6px 20px rgba(0,0,0,0.1);
            margin-bottom: 25px;
        }
        
        .filters-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        .search-box {
            position: relative;
            margin-bottom: 15px;
        }
        
        .search-input {
            width: 100%;
            padding: 12px 20px 12px 45px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 1rem;
            transition: border-color 0.3s ease;
        }
        
        .search-input:focus {
            border-color: #3498db;
            outline: none;
            box-shadow: 0 0 0 0.2rem rgba(52,152,219,0.25);
        }
        
        .search-icon {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #666;
        }
        
        .actions-toolbar {
            background: #e9ecef;
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 15px;
        }
        
        .view-toggles {
            display: flex;
            gap: 5px;
        }
        
        .view-toggle {
            padding: 8px 15px;
            border: 1px solid #dee2e6;
            background: white;
            color: #666;
            border-radius: 6px;
            text-decoration: none;
            transition: all 0.3s ease;
        }
        
        .view-toggle.active {
            background: #3498db;
            color: white;
            border-color: #3498db;
        }
        
        .table-container {
            overflow-x: auto;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .table {
            margin: 0;
            background: white;
        }
        
        .table th {
            background: #f8f9fa;
            border-top: none;
            border-bottom: 2px solid #dee2e6;
            font-weight: 600;
            color: #2c3e50;
            padding: 15px 12px;
            white-space: nowrap;
        }
        
        .table td {
            padding: 12px;
            vertical-align: middle;
            border-top: 1px solid #f1f3f4;
        }
        
        .table tbody tr:hover {
            background: #f8f9fa;
        }
        
        .producto-codigo {
            font-family: 'Courier New', monospace;
            background: #e9ecef;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 0.85rem;
            font-weight: 600;
        }
        
        .btn-action {
            padding: 6px 10px;
            border-radius: 6px;
            border: none;
            font-size: 0.8rem;
            margin: 1px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn-edit { background: #3498db; color: white; }
        .btn-edit:hover { background: #2980b9; }
        .btn-delete { background: #e74c3c; color: white; }
        .btn-delete:hover { background: #c0392b; }
        .btn-adjust { background: #f39c12; color: white; }
        .btn-adjust:hover { background: #e67e22; }
        .btn-history { background: #9b59b6; color: white; }
        .btn-history:hover { background: #8e44ad; }
        
        .productos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .producto-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
            border: 1px solid #f1f3f4;
        }
        
        .producto-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.15);
        }
        
        .form-floating {
            margin-bottom: 20px;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 12px 15px;
            transition: border-color 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #3498db;
            box-shadow: 0 0 0 0.2rem rgba(52,152,219,0.25);
        }
        
        .btn-primary {
            background: linear-gradient(45deg, #3498db, #2980b9);
            border: none;
            border-radius: 8px;
            padding: 12px 25px;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(52,152,219,0.3);
        }
        
        .modal-content {
            border: none;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
        
        .modal-header {
            background: linear-gradient(45deg, #3498db, #2980b9);
            color: white;
            border-radius: 15px 15px 0 0;
            border-bottom: none;
        }
        
        .no-productos {
            text-align: center;
            padding: 60px 20px;
            color: #666;
        }
        
        .no-productos i {
            font-size: 4rem;
            margin-bottom: 20px;
            color: #ddd;
        }
        
        @media (max-width: 768px) {
            .header-section { text-align: center; }
            .stats-cards { grid-template-columns: 1fr; }
            .actions-toolbar { flex-direction: column; align-items: stretch; }
            .productos-grid { grid-template-columns: 1fr; }
        }
    </style>
</head>
<body>
    <div class="main-container">
        <!-- Header -->
        <div class="header-section">
            <h1 class="header-title">
                <i class="fas fa-boxes"></i>
                Gestión de Productos e Inventario
            </h1>
            <div class="header-actions">
                <a href="dashboard.php" class="btn-header">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <a href="pos.php" class="btn-header">
                    <i class="fas fa-shopping-cart"></i> Punto de Venta
                </a>
                <a href="?exportar=csv" class="btn-header">
                    <i class="fas fa-download"></i> Exportar CSV
                </a>
                <?php if ($accion === 'listar'): ?>
                    <a href="?accion=crear" class="btn-header">
                        <i class="fas fa-plus"></i> Nuevo Producto
                    </a>
                <?php endif; ?>
            </div>
        </div>

        <!-- Mensajes -->
        <?php if ($mensaje): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show">
                <i class="fas fa-<?php echo $tipo_mensaje === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                <?php echo htmlspecialchars($mensaje); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if ($accion === 'listar'): ?>
            <!-- Estadísticas -->
            <div class="stats-cards">
                <div class="stat-card primary">
                    <div class="stat-value"><?php echo number_format($total_productos); ?></div>
                    <div class="stat-label">Productos Activos</div>
                </div>
                <div class="stat-card warning">
                    <div class="stat-value"><?php echo number_format($productos_stock_bajo); ?></div>
                    <div class="stat-label">Stock Bajo</div>
                </div>
                <div class="stat-card danger">
                    <div class="stat-value"><?php echo number_format($productos_sin_stock); ?></div>
                    <div class="stat-label">Sin Stock</div>
                </div>
                <div class="stat-card success">
                    <div class="stat-value"><?php echo $empresa['moneda_principal']; ?> <?php echo number_format($valor_inventario, 0, ',', '.'); ?></div>
                    <div class="stat-label">Valor Inventario</div>
                </div>
            </div>

            <!-- Contenido principal -->
            <div class="content-card">
                <!-- Filtros -->
                <div class="filters-section">
                    <form method="GET" class="row g-3">
                        <div class="col-md-4">
                            <div class="search-box">
                                <i class="fas fa-search search-icon"></i>
                                <input type="text" name="buscar" class="search-input" placeholder="Buscar productos..." value="<?php echo htmlspecialchars($filtro_busqueda); ?>">
                            </div>
                        </div>
                        <div class="col-md-3">
                            <select name="categoria" class="form-control">
                                <option value="">Todas las categorías</option>
                                <?php foreach ($categorias as $categoria): ?>
                                    <option value="<?php echo htmlspecialchars($categoria); ?>" <?php echo $filtro_categoria === $categoria ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($categoria); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <select name="estado" class="form-control">
                                <option value="todos">Todos</option>
                                <option value="activos" <?php echo $filtro_estado === 'activos' ? 'selected' : ''; ?>>Activos</option>
                                <option value="inactivos" <?php echo $filtro_estado === 'inactivos' ? 'selected' : ''; ?>>Inactivos</option>
                                <option value="stock_bajo" <?php echo $filtro_estado === 'stock_bajo' ? 'selected' : ''; ?>>Stock Bajo</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <button type="submit" class="btn btn-primary w-100">
                                <i class="fas fa-filter"></i> Filtrar
                            </button>
                        </div>
                        <input type="hidden" name="vista" value="<?php echo $vista; ?>">
                    </form>
                </div>

                <!-- Toolbar de acciones -->
                <div class="actions-toolbar">
                    <div class="d-flex align-items-center gap-3">
                        <div class="view-toggles">
                            <a href="?vista=tabla&<?php echo http_build_query(array_filter($_GET, function($k) { return $k !== 'vista'; }, ARRAY_FILTER_USE_KEY)); ?>" 
                               class="view-toggle <?php echo $vista === 'tabla' ? 'active' : ''; ?>">
                                <i class="fas fa-table"></i> Tabla
                            </a>
                            <a href="?vista=tarjetas&<?php echo http_build_query(array_filter($_GET, function($k) { return $k !== 'vista'; }, ARRAY_FILTER_USE_KEY)); ?>" 
                               class="view-toggle <?php echo $vista === 'tarjetas' ? 'active' : ''; ?>">
                                <i class="fas fa-th-large"></i> Tarjetas
                            </a>
                        </div>
                        <div class="text-muted">
                            <?php echo count($productos); ?> productos encontrados
                        </div>
                    </div>
                    
                    <?php if (!empty($productos)): ?>
                        <div class="d-flex align-items-center gap-2">
                            <button type="button" class="btn btn-sm btn-outline-primary" onclick="seleccionarTodos()">
                                <i class="fas fa-check-square"></i> Todos
                            </button>
                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="limpiarSeleccion()">
                                <i class="fas fa-square"></i> Limpiar
                            </button>
                            <button type="button" class="btn btn-sm btn-success" onclick="mostrarAccionesMasivas()" id="btnAccionesMasivas" style="display: none;">
                                <i class="fas fa-cogs"></i> Acciones Masivas
                            </button>
                        </div>
                    <?php endif; ?>
                </div>

                <?php if (empty($productos)): ?>
                    <div class="no-productos">
                        <i class="fas fa-box-open"></i>
                        <h4>No hay productos</h4>
                        <p>Comience agregando productos a su inventario</p>
                        <a href="?accion=crear" class="btn btn-primary">
                            <i class="fas fa-plus"></i> Agregar Producto
                        </a>
                    </div>
                <?php else: ?>
                    
                    <?php if ($vista === 'tabla'): ?>
                        <!-- Vista de tabla -->
                        <div class="table-container">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th width="40px">
                                            <input type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                                        </th>
                                        <th>Código</th>
                                        <th>Producto</th>
                                        <th>Categoría</th>
                                        <th>Precios</th>
                                        <th>Stock</th>
                                        <th>Estado</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($productos as $producto): ?>
                                        <tr>
                                            <td>
                                                <input type="checkbox" class="producto-checkbox" value="<?php echo $producto['id']; ?>" onchange="actualizarContador()">
                                            </td>
                                            <td>
                                                <span class="producto-codigo"><?php echo htmlspecialchars($producto['codigo']); ?></span>
                                            </td>
                                            <td>
                                                <strong><?php echo htmlspecialchars($producto['nombre']); ?></strong>
                                                <?php if ($producto['descripcion']): ?>
                                                    <br><small class="text-muted"><?php echo htmlspecialchars($producto['descripcion']); ?></small>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($producto['categoria']): ?>
                                                    <span class="badge bg-secondary"><?php echo htmlspecialchars($producto['categoria']); ?></span>
                                                <?php else: ?>
                                                    <span class="text-muted">-</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <div>
                                                    <strong><?php echo $empresa['moneda_principal']; ?> <?php echo number_format($producto['precio_venta'], 0, ',', '.'); ?></strong><br>
                                                    <small class="text-muted">Compra: <?php echo $empresa['moneda_principal']; ?> <?php echo number_format($producto['precio_compra'], 0, ',', '.'); ?></small>
                                                </div>
                                            </td>
                                            <td>
                                                <span class="fw-bold <?php 
                                                    if ($producto['stock_actual'] <= 0) echo 'text-danger';
                                                    elseif ($producto['stock_actual'] <= $producto['stock_minimo']) echo 'text-warning';
                                                    else echo 'text-success';
                                                ?>">
                                                    <?php echo $producto['stock_actual']; ?>
                                                </span>
                                                <small class="text-muted"> <?php echo $producto['unidad_medida']; ?></small>
                                                <?php if ($producto['stock_actual'] <= $producto['stock_minimo']): ?>
                                                    <br><i class="fas fa-exclamation-triangle text-warning"></i>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="badge <?php echo $producto['activo'] ? 'bg-success' : 'bg-danger'; ?>">
                                                    <?php echo $producto['activo'] ? 'Activo' : 'Inactivo'; ?>
                                                </span>
                                            </td>
                                            <td>
                                                <div class="btn-group" role="group">
                                                    <a href="?accion=editar&id=<?php echo $producto['id']; ?>" class="btn-action btn-edit" title="Editar">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <button class="btn-action btn-adjust" onclick="abrirModalAjuste(<?php echo $producto['id']; ?>, '<?php echo htmlspecialchars($producto['nombre'], ENT_QUOTES); ?>', <?php echo $producto['stock_actual']; ?>)" title="Ajustar Stock">
                                                        <i class="fas fa-balance-scale"></i>
                                                    </button>
                                                    <a href="?accion=historial&id=<?php echo $producto['id']; ?>" class="btn-action btn-history" title="Historial">
                                                        <i class="fas fa-history"></i>
                                                    </a>
                                                    <?php if ($producto['activo']): ?>
                                                        <a href="?eliminar=<?php echo $producto['id']; ?>" class="btn-action btn-delete" onclick="return confirm('¿Desactivar este producto?')" title="Desactivar">
                                                            <i class="fas fa-times"></i>
                                                        </a>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    
                    <?php else: ?>
                        <!-- Vista de tarjetas -->
                        <div class="productos-grid">
                            <?php foreach ($productos as $producto): ?>
                                <div class="producto-card">
                                    <div class="d-flex justify-content-between align-items-start mb-3">
                                        <div class="d-flex align-items-center gap-2">
                                            <input type="checkbox" class="producto-checkbox" value="<?php echo $producto['id']; ?>" onchange="actualizarContador()">
                                            <h6 class="mb-0"><?php echo htmlspecialchars($producto['nombre']); ?></h6>
                                        </div>
                                        <?php if ($producto['categoria']): ?>
                                            <span class="badge bg-secondary"><?php echo htmlspecialchars($producto['categoria']); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <div class="d-flex justify-content-between">
                                            <span>Código:</span>
                                            <span class="producto-codigo"><?php echo htmlspecialchars($producto['codigo']); ?></span>
                                        </div>
                                        <div class="d-flex justify-content-between">
                                            <span>Precio:</span>
                                            <strong><?php echo $empresa['moneda_principal']; ?> <?php echo number_format($producto['precio_venta'], 0, ',', '.'); ?></strong>
                                        </div>
                                        <div class="d-flex justify-content-between">
                                            <span>Stock:</span>
                                            <span class="fw-bold <?php 
                                                if ($producto['stock_actual'] <= 0) echo 'text-danger';
                                                elseif ($producto['stock_actual'] <= $producto['stock_minimo']) echo 'text-warning';
                                                else echo 'text-success';
                                            ?>">
                                                <?php echo $producto['stock_actual']; ?> <?php echo $producto['unidad_medida']; ?>
                                            </span>
                                        </div>
                                        <div class="d-flex justify-content-between">
                                            <span>Estado:</span>
                                            <span class="badge <?php echo $producto['activo'] ? 'bg-success' : 'bg-danger'; ?>">
                                                <?php echo $producto['activo'] ? 'Activo' : 'Inactivo'; ?>
                                            </span>
                                        </div>
                                    </div>
                                    
                                    <div class="d-flex gap-2 flex-wrap">
                                        <a href="?accion=editar&id=<?php echo $producto['id']; ?>" class="btn-action btn-edit">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <button class="btn-action btn-adjust" onclick="abrirModalAjuste(<?php echo $producto['id']; ?>, '<?php echo htmlspecialchars($producto['nombre'], ENT_QUOTES); ?>', <?php echo $producto['stock_actual']; ?>)">
                                            <i class="fas fa-balance-scale"></i>
                                        </button>
                                        <a href="?accion=historial&id=<?php echo $producto['id']; ?>" class="btn-action btn-history">
                                            <i class="fas fa-history"></i>
                                        </a>
                                        <?php if ($producto['activo']): ?>
                                            <a href="?eliminar=<?php echo $producto['id']; ?>" class="btn-action btn-delete" onclick="return confirm('¿Desactivar?')">
                                                <i class="fas fa-times"></i>
                                            </a>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                    
                <?php endif; ?>
            </div>

        <?php elseif ($accion === 'historial' && $producto_historial): ?>
            <!-- Historial de movimientos -->
            <div class="content-card">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3>
                        <i class="fas fa-history"></i>
                        Historial - <?php echo htmlspecialchars($producto_historial['nombre']); ?>
                    </h3>
                    <div>
                        <a href="?accion=listar" class="btn btn-secondary">
                            <i class="fas fa-arrow-left"></i> Volver
                        </a>
                        <a href="?accion=editar&id=<?php echo $producto_historial['id']; ?>" class="btn btn-primary">
                            <i class="fas fa-edit"></i> Editar
                        </a>
                    </div>
                </div>

                <div class="row mb-4">
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-body">
                                <h6>Información del Producto</h6>
                                <div class="row">
                                    <div class="col-sm-6">
                                        <strong>Código:</strong> <span class="producto-codigo"><?php echo htmlspecialchars($producto_historial['codigo']); ?></span><br>
                                        <strong>Stock Actual:</strong> <?php echo $producto_historial['stock_actual']; ?> <?php echo $producto_historial['unidad_medida']; ?>
                                    </div>
                                    <div class="col-sm-6">
                                        <strong>Stock Mínimo:</strong> <?php echo $producto_historial['stock_minimo']; ?><br>
                                        <strong>Precio:</strong> <?php echo $empresa['moneda_principal']; ?> <?php echo number_format($producto_historial['precio_venta'], 0, ',', '.'); ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="text-end">
                            <button class="btn btn-warning" onclick="abrirModalAjuste(<?php echo $producto_historial['id']; ?>, '<?php echo htmlspecialchars($producto_historial['nombre'], ENT_QUOTES); ?>', <?php echo $producto_historial['stock_actual']; ?>)">
                                <i class="fas fa-balance-scale"></i> Ajustar Stock
                            </button>
                        </div>
                    </div>
                </div>

                <?php if (!empty($movimientos)): ?>
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Fecha</th>
                                    <th>Tipo</th>
                                    <th>Cantidad</th>
                                    <th>Stock Anterior</th>
                                    <th>Stock Nuevo</th>
                                    <th>Motivo</th>
                                    <th>Usuario</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($movimientos as $movimiento): ?>
                                    <tr>
                                        <td><?php echo date('d/m/Y H:i', strtotime($movimiento['fecha_registro'])); ?></td>
                                        <td>
                                            <span class="badge <?php echo strpos($movimiento['tipo_movimiento'], 'entrada') !== false ? 'bg-success' : 'bg-warning'; ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $movimiento['tipo_movimiento'])); ?>
                                            </span>
                                        </td>
                                        <td class="fw-bold"><?php echo $movimiento['cantidad']; ?></td>
                                        <td><?php echo $movimiento['stock_anterior']; ?></td>
                                        <td><?php echo $movimiento['stock_nuevo']; ?></td>
                                        <td><?php echo htmlspecialchars($movimiento['motivo']); ?></td>
                                        <td><?php echo htmlspecialchars($movimiento['usuario_nombre'] ?? 'Sistema'); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-5">
                        <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                        <h5>No hay movimientos registrados</h5>
                    </div>
                <?php endif; ?>
            </div>

        <?php elseif ($accion === 'crear' || $accion === 'editar'): ?>
            <!-- Formulario de Producto -->
            <div class="content-card">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3>
                        <i class="fas fa-<?php echo $accion === 'crear' ? 'plus' : 'edit'; ?>"></i>
                        <?php echo $accion === 'crear' ? 'Nuevo Producto' : 'Editar Producto'; ?>
                    </h3>
                    <a href="?accion=listar" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> Volver
                    </a>
                </div>

                <form method="POST" class="row g-3">
                    <div class="col-md-6">
                        <div class="form-floating">
                            <input type="text" class="form-control" id="codigo" name="codigo" placeholder="Código" 
                                   value="<?php echo htmlspecialchars($producto_editar['codigo'] ?? ''); ?>" required>
                            <label for="codigo">Código *</label>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="form-floating">
                            <input type="text" class="form-control" id="categoria" name="categoria" placeholder="Categoría" 
                                   value="<?php echo htmlspecialchars($producto_editar['categoria'] ?? ''); ?>" list="categorias">
                            <label for="categoria">Categoría</label>
                            <datalist id="categorias">
                                <?php foreach ($categorias as $categoria): ?>
                                    <option value="<?php echo htmlspecialchars($categoria); ?>">
                                <?php endforeach; ?>
                            </datalist>
                        </div>
                    </div>
                    
                    <div class="col-12">
                        <div class="form-floating">
                            <input type="text" class="form-control" id="nombre" name="nombre" placeholder="Nombre" 
                                   value="<?php echo htmlspecialchars($producto_editar['nombre'] ?? ''); ?>" required>
                            <label for="nombre">Nombre del Producto *</label>
                        </div>
                    </div>
                    
                    <div class="col-12">
                        <div class="form-floating">
                            <textarea class="form-control" id="descripcion" name="descripcion" placeholder="Descripción" style="height: 100px"><?php echo htmlspecialchars($producto_editar['descripcion'] ?? ''); ?></textarea>
                            <label for="descripcion">Descripción</label>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="form-floating">
                            <input type="number" class="form-control" id="precio_compra" name="precio_compra" placeholder="Precio Compra" step="0.01" 
                                   value="<?php echo $producto_editar['precio_compra'] ?? ''; ?>">
                            <label for="precio_compra">Precio Compra</label>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="form-floating">
                            <input type="number" class="form-control" id="precio_venta" name="precio_venta" placeholder="Precio Venta" step="0.01" 
                                   value="<?php echo $producto_editar['precio_venta'] ?? ''; ?>" required>
                            <label for="precio_venta">Precio Venta *</label>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-floating">
                            <input type="number" class="form-control" id="stock_actual" name="stock_actual" placeholder="Stock Actual" 
                                   value="<?php echo $producto_editar['stock_actual'] ?? '0'; ?>">
                            <label for="stock_actual">Stock Actual</label>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-floating">
                            <input type="number" class="form-control" id="stock_minimo" name="stock_minimo" placeholder="Stock Mínimo" 
                                   value="<?php echo $producto_editar['stock_minimo'] ?? '0'; ?>">
                            <label for="stock_minimo">Stock Mínimo</label>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-floating">
                            <select class="form-control" id="unidad_medida" name="unidad_medida">
                                <option value="UN" <?php echo ($producto_editar['unidad_medida'] ?? 'UN') === 'UN' ? 'selected' : ''; ?>>Unidad</option>
                                <option value="KG" <?php echo ($producto_editar['unidad_medida'] ?? '') === 'KG' ? 'selected' : ''; ?>>Kilogramo</option>
                                <option value="LT" <?php echo ($producto_editar['unidad_medida'] ?? '') === 'LT' ? 'selected' : ''; ?>>Litro</option>
                                <option value="MT" <?php echo ($producto_editar['unidad_medida'] ?? '') === 'MT' ? 'selected' : ''; ?>>Metro</option>
                            </select>
                            <label for="unidad_medida">Unidad de Medida</label>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="form-floating">
                            <input type="number" class="form-control" id="iva_porcentaje" name="iva_porcentaje" placeholder="IVA %" step="0.01" 
                                   value="<?php echo $producto_editar['iva_porcentaje'] ?? '19.00'; ?>">
                            <label for="iva_porcentaje">IVA %</label>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="form-floating">
                            <input type="text" class="form-control" id="codigo_sii" name="codigo_sii" placeholder="Código SII" 
                                   value="<?php echo htmlspecialchars($producto_editar['codigo_sii'] ?? ''); ?>">
                            <label for="codigo_sii">Código SII</label>
                        </div>
                    </div>
                    
                    <div class="col-12">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="activo" name="activo" 
                                   <?php echo ($producto_editar['activo'] ?? 1) ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="activo">
                                Producto activo
                            </label>
                        </div>
                    </div>
                    
                    <div class="col-12">
                        <hr>
                        <div class="d-flex gap-3">
                            <button type="submit" name="guardar_producto" class="btn btn-primary">
                                <i class="fas fa-save"></i> <?php echo $accion === 'crear' ? 'Crear Producto' : 'Actualizar'; ?>
                            </button>
                            <a href="?accion=listar" class="btn btn-secondary">
                                <i class="fas fa-times"></i> Cancelar
                            </a>
                        </div>
                    </div>
                </form>
            </div>
        <?php endif; ?>
    </div>

    <!-- Modal Ajuste de Inventario -->
    <div class="modal fade" id="modalAjusteInventario" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-balance-scale"></i> Ajuste de Inventario
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" id="producto_ajuste_id" name="producto_ajuste_id">
                        
                        <div class="alert alert-info">
                            <strong>Producto:</strong> <span id="producto_ajuste_nombre"></span><br>
                            <strong>Stock Actual:</strong> <span id="producto_ajuste_stock"></span>
                        </div>
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Tipo de Ajuste</label>
                                <select name="tipo_ajuste" class="form-control" required>
                                    <option value="">Seleccionar...</option>
                                    <option value="entrada">Entrada (+)</option>
                                    <option value="salida">Salida (-)</option>
                                </select>
                            </div>
                            
                            <div class="col-md-6">
                                <label class="form-label">Cantidad</label>
                                <input type="number" name="cantidad_ajuste" class="form-control" min="1" required>
                            </div>
                        </div>
                        
                        <div class="mt-3">
                            <label class="form-label">Motivo del Ajuste</label>
                            <textarea name="motivo_ajuste" class="form-control" rows="3" placeholder="Describa el motivo..." required></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" name="ajustar_inventario" class="btn btn-primary">
                            <i class="fas fa-check"></i> Procesar Ajuste
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Acciones Masivas -->
    <div class="modal fade" id="modalAccionesMasivas" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-cogs"></i> Acciones Masivas
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="alert alert-warning">
                            <strong>Productos seleccionados:</strong> <span id="contadorSeleccionados">0</span>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Acción a realizar</label>
                            <select name="accion_masiva" class="form-control" onchange="mostrarCamposAdicionales()" required>
                                <option value="">Seleccionar acción...</option>
                                <option value="activar">Activar productos</option>
                                <option value="desactivar">Desactivar productos</option>
                                <option value="aumentar_precio">Aumentar precios</option>
                            </select>
                        </div>
                        
                        <div id="campos_precio" style="display: none;">
                            <div class="mb-3">
                                <label class="form-label">Porcentaje de aumento (%)</label>
                                <input type="number" name="porcentaje_precio" class="form-control" min="0" step="0.1" placeholder="10.5">
                            </div>
                        </div>
                        
                        <input type="hidden" name="productos_seleccionados" id="productos_seleccionados_input">
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-warning">
                            <i class="fas fa-bolt"></i> Ejecutar
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let productosSeleccionados = [];
        
        function toggleSelectAll() {
            const selectAll = document.getElementById('selectAll');
            const checkboxes = document.querySelectorAll('.producto-checkbox');
            
            checkboxes.forEach(checkbox => {
                checkbox.checked = selectAll.checked;
            });
            
            actualizarContador();
        }
        
        function seleccionarTodos() {
            const checkboxes = document.querySelectorAll('.producto-checkbox');
            checkboxes.forEach(checkbox => {
                checkbox.checked = true;
            });
            if (document.getElementById('selectAll')) {
                document.getElementById('selectAll').checked = true;
            }
            actualizarContador();
        }
        
        function limpiarSeleccion() {
            const checkboxes = document.querySelectorAll('.producto-checkbox');
            checkboxes.forEach(checkbox => {
                checkbox.checked = false;
            });
            if (document.getElementById('selectAll')) {
                document.getElementById('selectAll').checked = false;
            }
            actualizarContador();
        }
        
        function actualizarContador() {
            const checkboxes = document.querySelectorAll('.producto-checkbox:checked');
            productosSeleccionados = Array.from(checkboxes).map(cb => cb.value);
            
            const btnAcciones = document.getElementById('btnAccionesMasivas');
            const contador = productosSeleccionados.length;
            
            if (contador > 0) {
                btnAcciones.style.display = 'inline-block';
                btnAcciones.innerHTML = `<i class="fas fa-cogs"></i> Acciones (${contador})`;
            } else {
                btnAcciones.style.display = 'none';
            }
        }
        
        function mostrarAccionesMasivas() {
            if (productosSeleccionados.length === 0) {
                alert('Seleccione al menos un producto');
                return;
            }
            
            document.getElementById('contadorSeleccionados').textContent = productosSeleccionados.length;
            document.getElementById('productos_seleccionados_input').value = productosSeleccionados.join(',');
            
            const modal = new bootstrap.Modal(document.getElementById('modalAccionesMasivas'));
            modal.show();
        }
        
        function mostrarCamposAdicionales() {
            const accion = document.querySelector('select[name="accion_masiva"]').value;
            const camposPrecio = document.getElementById('campos_precio');
            
            if (accion === 'aumentar_precio') {
                camposPrecio.style.display = 'block';
            } else {
                camposPrecio.style.display = 'none';
            }
        }
        
        function abrirModalAjuste(id, nombre, stock) {
            document.getElementById('producto_ajuste_id').value = id;
            document.getElementById('producto_ajuste_nombre').textContent = nombre;
            document.getElementById('producto_ajuste_stock').textContent = stock;
            
            const modal = new bootstrap.Modal(document.getElementById('modalAjusteInventario'));
            modal.show();
        }
        
        // Auto-generar código
        document.getElementById('nombre')?.addEventListener('input', function() {
            const codigoField = document.getElementById('codigo');
            if (codigoField.value === '') {
                const nombre = this.value.toUpperCase();
                const codigo = nombre.substring(0, 8).replace(/[^A-Z0-9]/g, '');
                codigoField.value = codigo;
            }
        });
        
        // Búsqueda en tiempo real
        let timeoutId;
        document.querySelector('input[name="buscar"]')?.addEventListener('input', function() {
            clearTimeout(timeoutId);
            timeoutId = setTimeout(() => {
                this.form.submit();
            }, 800);
        });
    </script>
</body>
</html>